{ 
	open Parser 
	exception LexError of string
}

let DIGIT = ['0'-'9']+
let ALPHABET      = ['a'-'z' 'A'-'Z']
let ALPHA_NUMERIC   = DIGIT | ALPHABET
let IDENTIFIER = ALPHABET (ALPHA_NUMERIC | '_')*
let WHITESPACE = [' ' '\t' '\r' '\n']


rule token = parse
	WHITESPACE { token lexbuf }
	| "/*"   { block_comment lexbuf }
	| "//"	{ line_comment lexbuf }

	| '('      { LPAREN }
	| ')'      { RPAREN }
	| '['      { LBRACK }
	| ']'      { RBRACK }
	| '{'      { LBRACE }
	| '}'      { RBRACE }
	| ';'      { SEMI }
	| ','      { COMMA }
	| '+'      { PLUS }
	| '-'      { MINUS }
	| '*'      { TIMES }
	| '/'      { DIVIDE }
	| '='      { ASSIGN }
	| "=="     { EQ }
	| "!="     { NEQ }
	| '<'      { LT }
	| "<="     { LEQ }
	| ">"      { GT }
	| ">="     { GEQ }
	| "if"     { IF }
	| "else"   { ELSE }
	| "for"    { FOR }
	| "while"  { WHILE }
	| "while"  { WHILE }
	| "return" { RETURN }
	| "int"    { INT }
	| "Point"  { POINT }
	| "Curve"  { CURVE }
	| "Layer"  { LAYER }
	| "."      { DOT }
	| "++"     { PLUSONE }
	| "--"     { MINUSONE }
	| DIGIT as lxm { LITERAL(int_of_string lxm) }
	| IDENTIFIER as lxm { ID(lxm) }
	| eof { EOF }
	| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and block_comment = parse
  | "*/"    { token lexbuf }
  | eof     { raise (LexError("unterminated block_comment!")) }
  | _       { block_comment lexbuf }

and line_comment = parse
  | ['\n' '\r'] { token lexbuf }
  | _           { line_comment lexbuf }
