open Ast
open Str

(* Translate a program in AST form into C#.  Throw an exception if something is wrong, e.g. a reference to an unknown function *)
let translate (statements, functions) =
	(* A built-in function lookup hash table so we can check if a function is defined and has the correct number of arguments *)
	let builtin_functions = Hashtbl.create 20 in
		Hashtbl.replace builtin_functions "GetTickCount" 0;
		Hashtbl.replace builtin_functions "Sqr" 1;
		Hashtbl.replace builtin_functions "LCase" 1;
		Hashtbl.replace builtin_functions "UCase" 1;
		Hashtbl.replace builtin_functions "Len" 1;
		Hashtbl.replace builtin_functions "Left" 2;
		Hashtbl.replace builtin_functions "Mid" 3;
		Hashtbl.replace builtin_functions "Right" 2;
		Hashtbl.replace builtin_functions "ListLen" 1;
		Hashtbl.replace builtin_functions "ListGetAt" 2;
		Hashtbl.replace builtin_functions "ListSetAt" 3;
		Hashtbl.replace builtin_functions "ListAppend" 2;
		Hashtbl.replace builtin_functions "ListFind" 2;
		Hashtbl.replace builtin_functions "Rand" 0;
		Hashtbl.replace builtin_functions "Now" 0;
		Hashtbl.replace builtin_functions "HTMLEditFormat" 1;
		Hashtbl.replace builtin_functions "URLEncodedFormat" 1;
		Hashtbl.replace builtin_functions "GetQueryParameter" 1;
		
	(* A hash table of user defined functions and the number of arguments for each function *)
	let user_defined_functions = Hashtbl.create 100 in
		(* Check all the functions *)
		List.iter (function f ->
			(* Don't allow the user to define multiple functions with the same name *) 
			if Hashtbl.mem user_defined_functions f.fname then
				raise (Failure ("Duplicate definition of function " ^ f.fname))
			(* Make sure function names are 100 characters or less *)
			else if String.length f.fname > 100 then
				raise (Failure ("Function name too long: " ^ f.fname))
			(* If there are no problems with the function, add it to the hash table of hashTbl[functionname] = (number of args) *) 
			else
				Hashtbl.replace user_defined_functions f.fname (List.length f.formals);
			
			(* Make sure there are no more than 100 arguments in a function *)
			if (List.length f.formals) > 100 then
				raise (Failure ("Excessive number of arguments in function " ^ f.fname));
				
			(* Check all the arguments within each function *)
			List.iter (function argname ->
				(* Check if multiple arguments with the same name within a function  *)
				if (List.length (List.find_all (function x -> x = argname) f.formals)) > 1 then
					raise (Failure ("The argument " ^ argname ^ " cannot occur multiple times in " ^ f.fname))
				(* Make sure argument names are 100 characters or less *)
				else if String.length argname > 100 then
					raise (Failure ("Argument name too long: " ^ argname))
				) f.formals
  		) functions;
			
	(* wfreturn statements can only be placed within functions *)
	let rec check_statements = function
			Block s1 -> ignore(List.map check_statements s1)
		| Return e -> raise (Failure ("wfreturn can only be used within functions"))
		| _ -> ignore() in
	let _ = List.map check_statements statements in
			
	(* Keep track of global variables used - global variables are all variables except function arguments *)
	let global_vars = Hashtbl.create 100 in
	
	(*****************************************************************************)
  (* Converts a binary operator to the equivalent compiler method *)
  let operator_to_cs = function
  		| Add -> "Add"
  		| Sub -> "Sub"
  		| Mult -> "Mult"
  		| Div -> "Div"
  		| Modulus -> "Modulus"
  		| Concatenate -> "Concatenate"
  		| Equal -> "Equal"
  		| Neq -> "Neq"
  		| Less -> "Less"
  		| Leq -> "Leq"
  		| Greater -> "Greater"
  		| Geq -> "Geq"
  		| And -> "And"
  		| Or -> "Or"
  
	(*****************************************************************************)
  (* Translate an expression *)
  in let rec expr = function
  	(* For numbers - A number can include a decimal portion *)
		(* WarmFusion has dynamic/loose typing so internally numbers are treated as strings and conversions are done when necessary *)
  	Literal i ->
			(* Simple check for numbers - the string cannot be longer than 25 characters.  We can actually handle numbers that big. *)
			if String.length i > 25 then
				 raise (Failure ("Numeric constant too long: " ^ i))
			else
				"\"" ^ i ^ "\""
		(* For strings *)
  	| String s -> "\"" ^ s ^ "\""
		(* To get the negative value of a number *)
  	| Negative e -> "WF.Negative(" ^ (expr e) ^ ")"
		(* To negate a boolean expression *)
  	| Negate e -> "WF.Negate(" ^ (expr e) ^ ")"
  	(* For variables - All variables are internally prefixed with udv_ to prevent naming collisions *)
  	| Id s -> Hashtbl.replace global_vars s 0;
							"udv_" ^ s
  	(* Binary operators *)
    | Binop (e1, op, e2) -> "WF." ^ (operator_to_cs op) ^ "(" ^ (expr e1) ^ ", " ^ (expr e2) ^ ")"
  	(* Call a function *)
  	| Call (fname, actuals) -> 
				(* Check if the function is a built-in function *)
  			if Hashtbl.mem builtin_functions fname then
					(* Number of arguments must match *)
					if (List.length actuals) != (Hashtbl.find builtin_functions fname) then
						raise (Failure ("Call of function " ^ fname ^ " does not have the correct number of arguments."))
					else
  					"WF." ^ fname ^ "(" ^ String.concat ", " (List.map expr actuals) ^ ")"
				(* Check if the function is a user-defined function *)
				else if Hashtbl.mem user_defined_functions fname then
					(* Number of arguments must match *)
					if (List.length actuals) != (Hashtbl.find user_defined_functions fname) then
						raise (Failure ("Call of function " ^ fname ^ " does not have the correct number of arguments."))
					else
						(* All user-defined functions are internally prefixed with udf_ to prevent naming collisions *)
						"udf_" ^ fname ^ "(" ^ String.concat ", " (List.map expr actuals) ^ ")"
				(* If the function is not defined *)
  			else
  				raise (Failure ("Undefined function " ^ fname))
  	(* *No expression *)
  	| Noexpr -> ""

	(*****************************************************************************)
  (* Handle wfoutput with variable names inside *)
  in let rec wfoutput = function
  	| Output s -> "Console.Write(@\"" ^ (Str.global_replace (Str.regexp_string "\"") "\"\"" s) ^ "\");\n"
  	| OutputVariable v -> 
			Hashtbl.replace global_vars v 0;
			"Console.Write(udv_" ^ v ^ ");\n" 
  
	(*****************************************************************************)
  (* Translate a statement *)
  in let rec stmt = function
  	(* For a block of statements, do one statement after the other *)
  	Block sl     ->  "\n{\n" ^ String.concat "" (List.map stmt sl) ^ "}\n"
		(* wfset *)
  	| Assign (v, e) -> 
			Hashtbl.replace global_vars v 0;
			"udv_" ^ v ^ " = " ^ (expr e) ^ ";\n"
		(* End of line *) 
    | Expr e       -> expr e ^ ";\n"
		(* wfreturn *)
  	| Return e -> "return " ^ (expr e) ^ ";\n"
		(* wfif/wfelse *)
  	| If (p, t, f) -> "if (WF.ConvertStringToBool(" ^ (expr p) ^ "))" ^ (stmt t) ^ "else" ^ (stmt f)
		(* wfloop with index *)  
  	| For (v, e2, e3, b) -> 
			Hashtbl.replace global_vars v 0;
			"for (udv_" ^ v ^ " = (" ^ (expr e2) ^ "); " ^
			"WF.ConvertStringToBool(WF.Leq(udv_" ^ v ^ ", " ^ (expr e3) ^ ")); " ^
			"udv_" ^ v ^ " = WF.Add(udv_" ^ v ^ ", \"1\"))" ^
			(stmt b)
		(* wfloop with condition *)
  	| While (e, b) -> "while (WF.ConvertStringToBool(" ^ (expr e) ^ ")) " ^ (stmt b)
		(* wfoutput - We need to do separate scanning for stuff inside the wfoutput tag.  This parses #variable# output *)
  	| RawOutput (s) -> 	let lexbuf = Lexing.from_string s in
  		 									let wfoutput_ast = Parser.wfoutput_parser Scanner.wfoutput_scanner lexbuf in
  											String.concat "" (List.map wfoutput (List.rev wfoutput_ast))
  
	(*****************************************************************************)
	(* Translate a function *)
  in let func f = 
		(* Make sure the user isn't trying to re-define a system function *)
		if Hashtbl.mem builtin_functions f.fname then
			raise (Failure ("Redefinition of a built-in system function " ^ f.fname))
		else
    	let arguments = List.map (function s -> "string udv_" ^ s) (List.rev f.formals) in
      	"public static string udf_" ^ f.fname ^ "(" ^ (String.concat ", " arguments) ^ ") {\n" ^
    		(String.concat "" (List.map stmt f.body)) ^ "\nreturn \"\";}\n\n"

	(*****************************************************************************)
  (* Generate code for all of the functions *)
  in
		(* Generate code for all functions and statements that aren't in functions *) 
		let strFunctions = String.concat "" (List.map func (List.rev functions)) in
		let strStatements = String.concat "" (List.map stmt (List.rev statements)) in
		(* WarmFusion doesn't need variable declarations.  You can just start using the variables without declaring them.  This does some processing to prepare for that. *)
		let varDeclarations = Hashtbl.fold (
				fun key _ str ->
					(* Make sure variable names are 100 characters or less *)
					if String.length key > 100 then
						 raise (Failure ("Variable name too long: " ^ key))
					else
						str ^ "public static string udv_" ^ key ^ " = \"\";\n"
			) global_vars "" in
		"public class Program
		{
		" ^
		varDeclarations ^
		strFunctions ^
  	(* Main header *)
  	"public static void Main()
  	{
  		try
  		{
  			Console.WriteLine(@\"Content-type: text/html\n\n\");
  		" ^
    strStatements ^
		(* Handle runtime errors *)
  	"	} catch (Exception e) {
          Console.WriteLine(\"WarmFusion Error: \" + e.Message);
    	}
  	}
		}"