open Commponents

(*
 * where variable is declared
 *)
type setting = {
  own_func : string ; (* containing function or asynchronous block *)
  (*
   * ID of immediately containing block. Unique in this function, and generated
   * like in dot.ml, ie. pre-order.
   *)
  own_block : int ;
}
(*
 * object pointed to by identifier
 *)
type id_object = {
	name : string;
  obj_type : xvartype ; (* type of value the variable points to *)
  scope_setting : setting ; (* where was the function declared? *)
  (*
   * if this variable needs to be copied from a higher scope for closure,
   * then this is the free variable id that is unique in this function or
   * async block. Otherwise, it is the function-unique bound variable id.
   * the free id and bound variable id follow the same rules, but are in
   * distinct namespaces, eg. we can only have one free variable of id
   * 100 in a function, but we can also have a bound variable of id 100 in
   * that function, as long as there are no other bound variables of id 100
   * in that function.
   *)
  var_id : int ;
  is_free : bool
}
and xfunc_dec_header = { xret_type : xvartype option ; xparams : xvar list }
and xsingle_vartype =
  | XFunkInt
  | XFunkDouble
  | XFunkChar
  | XFunkBool
  (* function instance's value not calculated statically *)
  | XFunkFunc of xfunc_dec_header
and xvartype =
  (*
   * type, size of each level. If integer list is empty, the type is just
   * single
   *)
  xsingle_vartype * (xexpr list)
and xvar = { xbare_type : xvartype }
(* generalize value, which could be known in an expression *)
and funk_value =
  | SingleValue of single_funk_value
  | ArrayValue of xvartype * xrvalue list
(* declared variables already known, so only handle assignments *)
and xassignment = { lvals: xexpr list ; rvals: xrvalue list}
and called_func =
  | PrintCall
  | Double2IntCall
  | Int2DoubleCall
  | Boolean2IntCall
  | Int2BooleanCall
  | GenericCall of xexpr

(*
 * extended version of func_call just uses extended versions of
 * func_call's members
 *)
and xfunc_call = xfunc_dec_header * called_func * (xrvalue list)
(*
 * with some exceptions, the only change is the addition of the xvartype
 * argument, and the use of the extended types of the original arguments,
 * eg. rvalue -> xrvalue
 *)
and xexpr =
  | XSingleConst of xsingle_vartype * single_funk_value
  | XArrayLit of xvartype * xrvalue list
  | XVariable of id_object (* point to object, rather than ID *)
  | XFunkUnExpr of xvartype * unop * xexpr
  | XFunkBinExpr of xvartype * xexpr * binop * xexpr
  | XFunkCallExpr of xvartype * xfunc_call
  | XFunkArrExpr of xvartype * xexpr * xexpr (* array access: array and index *)
  | XFunkAsyncExpr of xvartype * xblock (* return type, async block *)
and xanon = xfunc_dec_header * xblock
and xrvalue =
  | XExprRVal of xvartype * xexpr (* simple expression *)
  | XFuncRVal of xanon (* anonymous function declaration *)
and xstatement =
  | XAssignment of xassignment
  | XFunctionCall of xfunc_call
  | XBlock of xblock
  | XForBlock of (xstatement option) * (xexpr option) * (xstatement option) *
    xblock
  | XIfBlock of xexpr * xblock
  | XIfElseBlock of xexpr * xblock * xblock
  | XWhileBlock of xexpr * xblock
  | XBreak
  | XReturn of xrvalue option
(*
 * unlike the AST block, the SAST block needs to know about the variables
 * that need to be declared in it, and those it has declared
 *)
and xblock = { xstmts : xstatement list ; (* statements in code *)
               (*
                * free variable that needs a copy, paired with the higher-scope
                * variable that needs to be copied
                *)
               need_copy : (id_object * id_object) list ;
               (* variables it has declared, including copies *)
               declared : id_object list
             }

(* same as funcdec, except xbody contains xstatements *)
type xfuncdec = { global_id : id_object ; xfid : string ;
		  xfunc_header: xfunc_dec_header ;
                  xbody : xblock }

type xdeclaration =
  (*
   * we already know which variables are being declared, so code generation
   * only needs to know about assignment declarations
   *)
  | XVardec of xassignment
  | XFuncdec of xfuncdec

(* exactly analogous to xprogram *)
type xprogram = xdeclaration list

(*
 * result of eval on expression, including a possibly pruned expression AST
 * node, a value, if it can be statically determined, and the type
 *)
type evalue = { e_node : xexpr ; result : funk_value option ;
                e_type : xvartype }

val check_types : Commponents.program -> bool

