open Commponents
open Xcommponents
open Printf

(* maps variable name to stack of objects *)
module VarMap = Map.Make(struct
		type t = string
		let compare x y = Pervasives.compare x y
	end)

(* set of variables declared in scope *)
module VarSet = Set.Make(struct
		type t = string
		let compare x y = Pervasives.compare x y
	end)

(*
* Running scope environment, including variable-object stack map,
* variables declared in the block, and current block
*)
type running_env = {
	scope : id_object list VarMap.t ; (* maps name to id_object stack *)
	(*
	* any newly-declared variables in this block that will need to be popped
	* from scope after this block is completed
	*)
	new_objects : VarSet.t ;
	(* function-wide list version of new_objects *)
	new_objects_ordered : id_object list ;
	mutable free_objects_ordered: (id_object* id_object) list ;
	current_block : setting ; (* new variables will use this setting *)
	(*
	* next free id_object will have this value as var_id.
	* unique inside the function or async block, so this value is not passed
	* outside of a function, after it is checked.
	*)
	next_free : int ;
	(*
	* next bound id_object will have this value as var_id.
	* unique inside the function or async block, so this value is not passed
	* outside of a function, after it is checked.
	*)
	next_bound : int ;
	(* function-wide fields for keeping track of return type *)
	(* has a return type been found? If not, then any new return value is OK *)
	has_ret : bool ;
	(* Return type. If it's None, either that means that all return statements don't return anything, or has_ret = false *)
	r_ret_type : xvartype option
}

let filter_out_block new_env old_env =
	{ scope = old_env.scope ; new_objects = old_env.new_objects ;
	  new_objects_ordered = new_env.new_objects_ordered ;
	  free_objects_ordered = new_env.free_objects_ordered ;
	  current_block = old_env.current_block;
	  next_free = new_env.next_free ; next_bound = new_env.next_bound ;
	  has_ret = new_env.has_ret ; r_ret_type = new_env.r_ret_type }
let filter_in_function old_env name_opt =
	let name = match name_opt
		with Some(name) -> name
		| None -> old_env.current_block.own_func ^ "_"
	in { scope = old_env.scope; new_objects = VarSet.empty;
	current_block = { own_func = name ;
			  own_block = 0 };
	next_free = 0; next_bound = 0 ;
	new_objects_ordered = [] ; has_ret = false ; free_objects_ordered = [];
	r_ret_type = None }

(* debugging support *)
let debug_setting setting = 
 		eprintf "setting %s,%d\n" setting.own_func setting.own_block

let debug_single_vartype  typ  =
  eprintf  "#SINGLE_VARTYPE\n";
  match typ with
  | XFunkInt -> eprintf "int"

  | XFunkChar -> eprintf "char"

  | XFunkDouble -> eprintf "double"

  | XFunkBool -> eprintf "bool"

  | XFunkFunc x -> eprintf "function header"

let debug_vartype  var=
  eprintf "#VARTYPE: ";
	debug_single_vartype (fst(var))  
  (*print_expr_list next (next+1) (snd(var))
  List.iter eprint "expr" (snd(var))
*)

let debug_idobject id =
		eprintf "id_object: %s\n" id.name;
		debug_setting id.scope_setting

let debug_idobject_free id =
  eprintf("\027[36;40m[\027[0m");
  debug_idobject(fst(id));
  debug_idobject(snd(id));
  eprintf("\027[36;40m]\027[0m\n")

let debug_env env = 
	eprintf "DEBUGGING ENV\n";
	eprintf "VarMap\n";
	VarMap.iter (fun k v -> eprintf "\tvariable_name: %s\n" k) env.scope;
	eprintf "VarSet\n";
	VarSet.iter (fun k -> eprintf "\tvariable_name: %s\n" k) env.new_objects;
	eprintf "Free Vars\n";                         
	List.iter debug_idobject_free env.free_objects_ordered;
	eprintf "Binded Vars\n";                         
	List.iter debug_idobject env.new_objects_ordered

let debug_xblock xblock =
	eprintf("Declared variables:\n");
	List.iter debug_idobject xblock.declared;
	eprintf("Need to copy variables:\n");
	List.iter (fun (x,y) -> debug_idobject x; debug_idobject y; eprintf("\n")) xblock.need_copy
	
(* end of debugging support *)

(*
 * polymorphic list checker, iterate through a list and returns a list
 * of corresponding xtypes and the new_env
 *)
let list_checker checker env values =
	let env, checked_values = List.fold_left
		(fun (old_env, old_list) value ->
		 let new_env, new_checked = checker old_env value
		 in new_env, new_checked::old_list) (env, [])
		values
	in env, (List.rev checked_values)

let global_scope = "global"

let new_r_env = { scope = VarMap.empty ; new_objects = VarSet.empty ;
		new_objects_ordered = [] ;
		current_block = {own_func = global_scope ; own_block = 0} ;
		next_free = 0 ; next_bound = 0 ; has_ret = false ; r_ret_type = None; 
		free_objects_ordered = []}


(* helper functions for extracting single_vartype values *)
let get_single = function
	| SingleValue(sv) -> sv
	| ArrayValue(_) -> raise (Failure "Expected single value")
let get_int = function
	| IntVal(i_v) -> i_v
	| _ -> raise (Failure "Expected int for get_int")
let get_double = function
	| DoubleVal(d_v) -> d_v
	| _ -> raise (Failure "Expected double for get_double")
let get_bool = function
	| BoolVal(b_v) -> b_v
	| _ -> raise (Failure "Expected bool for get_bool")
let get_char = function
	| CharVal(c_v) -> c_v
	| _ -> raise (Failure "Expected char for get_char")

(* turns single_vartype into vartype *)
let generalize_single st = (st, [])

(* check for duplicate variable in the current scope-block *)

let is_not_duplicate env a =
	if VarMap.mem a env.scope
	then let obj = List.hd (VarMap.find a env.scope) in
	(*need to check if it's in a block at the same depth*)
	if obj.scope_setting = env.current_block
	then false
	else true
	else
		true
	

let check_duplicate env a =
	if is_not_duplicate env a
	then
		true
	else
		raise (Failure ("Variable already declared:" ^ a))

(* must update environment and return y *)
let mapAdd key xtype is_free env =
	(* get list of variables with name key *)
	let old_list = if VarMap.mem key env.scope
		then VarMap.find key env.scope
		else []
		in
	(*check that is not a redeclaration*)
	let _ = check_duplicate env key 
	(* assign a unique id which depends on whether this variable is free or not *)
	in let var_id, next_bound, next_free = if is_free
			then env.next_free, env.next_bound, env.next_free + 1
			else env.next_bound, env.next_bound + 1, env.next_free
	(* create a new id_object using the newly created uid *)		
	in let new_obj = {  name = key;  obj_type = xtype ; scope_setting = env.current_block ;
		var_id = var_id ; is_free = is_free }
	(* create new binded variables list *)
	in let new_obj_list = if is_free then env.new_objects_ordered
		else new_obj::env.new_objects_ordered 
	(* create new free variables list *)
	in let new_free_list = if is_free then
		if old_list = [] 
		then raise(Failure ("Variable " ^ key ^ " has never been declared before"))
		else let old_var = List.hd old_list in 
		(new_obj,old_var)::env.free_objects_ordered
		else env.free_objects_ordered
	(* modify the scope *)	
	in let new_scope =  VarMap.add key (new_obj::old_list) env.scope
	(* return the moloch *)
	in { current_block = env.current_block ; scope = new_scope ;
		next_free = next_free ; next_bound = next_bound ; new_objects =
			VarSet.add key env.new_objects ;
		new_objects_ordered = new_obj_list ; has_ret = env.has_ret ;
		r_ret_type = env.r_ret_type; free_objects_ordered = new_free_list
	}, new_obj


(*
* finds member type of an array
* --simply deletes first level
* (vartype) arr_type: type with at least one entry in sizes list
* returns type that the array stores, which is one level lower
*)
let find_lower arr_type =
	let size_list = snd arr_type
	in if (List.length size_list > 0)
		then fst arr_type, List.tl (snd arr_type)
		else raise (Failure "Array has 0 levels")
(*
* finds member xcommponents type of an array
* --simply deletes first level
* (xvartype) arr_type: type with at least one entry in sizes list
* returns type that the array stores, which is one level lower
*)
let find_lowerx arr_type =
	let size_list = snd arr_type
	in if (List.length size_list > 0)
		then fst arr_type, List.tl (snd arr_type)
		else raise (Failure "Array has 0 levels")
(*
* matches any two vartype objects. Note that in this implementation,
* general and special are effectively interchangeable
* (xvartype) general: the required type, as is specified in the formal
* parameters list, lvalue, or array type
* (xvartype) special: the type of the given value, as specified in the
* rvalue expression
* returns true if special matches general
*)
let rec match_type general special =
	if List.length (snd general) != List.length (snd special) then false
	else match fst general, fst special
	with XFunkFunc(g_f), XFunkFunc(s_f) -> match_func_type g_f s_f
	| g_t, s_t -> g_t == s_t
(*
* matches function types according to their headers.
* Note, again, that switching general and special is alright for now
* (xfunc_dec_header) general: the required function header
* (xfunc_dec_header) special: given function header
* returns true if both return type and formal parameters match
*)
and match_func_type general special =
	(* iteratively matches parameter types *)
	let rec match_param_types = function
		| [], [] -> true
		| g_v::g_tl, s_v::s_tl ->
			let g_vt, s_vt = g_v.xbare_type, s_v.xbare_type
			in match_type g_vt s_vt &&
				(match_param_types (g_tl, s_tl))
		| _, _ -> false
	in
	(* matches optional return type *)
	let match_ret_type = function
		| None, None -> true
		| Some(g_rv), Some(s_rv) -> match_type g_rv s_rv
		| _, _ -> false
	in (match_ret_type (general.xret_type, special.xret_type)) &&
		(match_param_types (general.xparams, special.xparams))
(*
* Converts vartype into xvartype, calling eval on any size expressions
* (vartype) vt: the value to convert
* return xvartype form of vt
*)
let rec extend_vartype vt r_env =
	let extend_sizes sizes r_env =
		let xsizes, r_env = List.fold_left
				(fun (old_xs, old_r_env) size ->
					let r_env, ev = eval old_r_env size
					in (ev.e_node::old_xs, r_env)
				)
				([], r_env) sizes
		in (List.rev xsizes), r_env
	in let extend_single r_env = function
		| FunkInt -> XFunkInt, r_env
		| FunkDouble -> XFunkDouble, r_env
		| FunkChar -> XFunkChar, r_env
		| FunkBool -> XFunkBool, r_env
		| FunkFunc(hdr) -> let xhdr, r_env = extend_func_header hdr
						     r_env
		in XFunkFunc(xhdr), r_env
	in let xsingle, r_env = extend_single r_env (fst vt)
	in let xsizes, r_env = extend_sizes (snd vt) r_env
	in (xsingle, xsizes), r_env


(*
* Checks expression types by traversing its tree
* (running_env) r_env: running environment of variables, as passed from
*                      a higher level
* (expr) expr_nd: AST-node representation of expression
* [? pending decision on whether eval should also generate code] (boolean) wait: Determines if evaluation should wait on an async value.
* returns the updated r_env and the evalue that stores the result of
*         the check
*)
and eval r_env expr_nd =
	(*
	* helper function for checking that unary and binary operations are
	* applied to single_vartype
	*)
	let narrow_single et = match snd et
	with [] -> fst et
		| levels -> raise (Failure
						"Math and logic operations cannot be applied to arrays")
	in match expr_nd
	(* single literal is constant, and always known *)
	with SingleConst(c) ->
		(* extract type from single_vartype *)
		let find_single_type = function
			| IntVal(_) -> FunkInt
			| DoubleVal(_) -> FunkDouble
			| CharVal(_) -> FunkChar
			| BoolVal(_) -> FunkBool
			(*
			* sanity check on compiler: should only have above tokens for constant
			*)
			| _ -> raise (Failure "Improper constant")
		in let st = generalize_single (find_single_type c)
		in let xct, r_env = extend_vartype st r_env
		in let const_xval = XSingleConst(fst xct, c)
		in r_env, { e_node = const_xval; result = Some(SingleValue(c)) ;
			e_type = xct }
	(* Variable value unknown, but can extract type from last declaration *)
	| Variable(v) ->
			let find_var_obj fv r_env=
				let name = fv.id in
				if VarMap.mem name r_env.scope
				then
					let obj = List.hd (VarMap.find name r_env.scope) in
					if obj.scope_setting = r_env.current_block then
						obj, r_env
					else
						if obj.scope_setting.own_func = r_env.current_block.own_func ||
							obj.scope_setting.own_func = global_scope then
							obj, r_env
						else
							let r_env, free_copy = mapAdd name obj.obj_type true r_env
							in 
							free_copy, r_env
				else raise (Failure ("Variable " ^ name ^ " has not been declared"))
			in let v_o, r_env = find_var_obj v r_env in
			r_env, { e_node = XVariable(v_o); result = None ; e_type = v_o.obj_type }
	(*
	* we know array value if all members are constant.
	* but first, we need to eval all the members
	*)
	| ArrayLit(arr_type, elements) ->
			let member_type = find_lower (arr_type) in
			(*
		* evaluates each member, returning the new env_r, filtered rvalue, and
		* if it's constant. The expression or function type is also checked
		* against member_type, the second part of the second argument tuple,
		* but is not returned
		*)
			let eval_member r_env = function
				| ExprRVal(e), mt -> let r_env, expr_member = eval r_env e
				in let xmt, r_env = extend_vartype mt r_env
				in let _ = if not (match_type xmt expr_member.e_type)
					then raise (Failure "expression in array gives wrong type")
				in
				(r_env, XExprRVal(expr_member.e_type, expr_member.e_node),
				expr_member.result != None)
				| FuncRVal(af), (FunkFunc(mtf), []) -> let r_env, fheader, fbody, _ =
						(check_func r_env (fst(af)) (snd(af)) None)
				in let xmtf, r_env = extend_func_header mtf r_env
				in let matched = match_func_type xmtf fheader
				in let _ = if not matched
					then raise (Failure "Function in array has wrong type")
				in (r_env, XFuncRVal(fheader, fbody), false)
				| _, _ -> raise (Failure "Wrong array member")
			(* cumulatively applies eval_member to all members *)
			in let eval_members es = List.fold_left
					(fun (c_env, known, rlist) next ->
						let (n_env, rval, n_known) =
							eval_member c_env
								(next, member_type)
						in
						(n_env, known & n_known, rval::rlist)
					)
					(r_env, true, []) es
			in let r_env, known, rlist = eval_members elements
			in let evaled_list = List.rev rlist
			in let xarr_type, r_env = extend_vartype arr_type r_env
			in let xarr_type = if (List.length evaled_list > 0)
				then let highest = List.hd (snd xarr_type)
				in let have_size = match highest
				with XSingleConst(_, sv) -> (match sv
				with IntVal(si) -> (si > 0)
				| _ -> true
				)
					| _ -> true
				in if have_size then xarr_type
				else let base = fst xarr_type
				in let remainder = List.tl (snd xarr_type)
				in let replacement = (XSingleConst(XFunkInt,
						IntVal(List.length evaled_list)
					)
				)
				in (base, replacement::remainder)
				else xarr_type
			in let tentative = ArrayValue(xarr_type, evaled_list)
			(*
		* "value" is necessary for ArrayLit expression, and if it's fully
		* known, it is put into result
		*)
			in let known_value = if known then Some(tentative) else None
			in r_env, { e_node = XArrayLit(xarr_type, evaled_list) ;
				result = known_value ; e_type = xarr_type }
	(*
	* Prefix unary operator can only work on correct type. If the value of
	* the expression to operate on is known, it is straightforward to calculate
	* the result of the unary expression
	*)
	| FunkUnExpr(op, e) ->
			let r_env, evaled = eval r_env e
			in let unop_func = match narrow_single evaled.e_type
			with XFunkInt ->
				let do_unop_int op_f = fun oprnd -> IntVal(op_f (get_int oprnd))
				in (match op
				with IntNeg -> do_unop_int (fun x -> -x)
				| Positive -> do_unop_int (fun x -> x)
				| BitNot -> do_unop_int (fun x -> lnot x)
				| _ -> raise (Failure ("Integers can only have unary operators \'-\', \'+\' and \'~\'"))
				)
				| XFunkDouble ->
						let do_unop_double op_f = fun oprnd -> DoubleVal(op_f
									(get_double oprnd))
						in (match op
						with IntNeg -> do_unop_double (fun x -> -. x)
						| Positive -> do_unop_double (fun x -> x)
						| _ -> raise (Failure ("Doubles can only have unary operators\'-\', and \'+\'"))
						)
				| XFunkBool ->
						let do_unop_bool op_f = fun oprnd -> BoolVal(op_f (get_bool oprnd))
						in (match op
						with Not -> do_unop_bool (fun x -> not x)
						| _ -> raise (Failure "Booleans can only have unary operators \'!\'")
						)
				| _ -> raise (Failure "Invalid type for unary operator")
			in let conv_eval = match evaled.result
			(* LOOOK HERE THIS IS AN EXAMPLE OF HOW YOU INIT EVALUE *)
			with None -> { e_node = XFunkUnExpr(evaled.e_type, op, evaled.e_node);
				result = None ; e_type = evaled.e_type }
				| Some(rv) -> match rv
				with ArrayValue(_) -> raise (Failure
							"Cannot evaluate array with unary operator")
				| SingleValue(sv) -> let const_result = unop_func(sv)
				in { e_node = XSingleConst(fst evaled.e_type, const_result) ;
					result = Some(SingleValue(const_result)) ;
					e_type = evaled.e_type }
			in r_env, conv_eval
	(*
	* Binary operators can only work on certain pairs of operands.
	* If both operands have known values, calculating the result is possible
	* here. And certain operations can take advantage of the value of only 1
	* operand
	*)
	| FunkBinExpr(e1, op, e2) ->
			let r_env, evaled1 = eval r_env e1
			in let r_env, evaled2 = eval r_env e2
			in let type1 = narrow_single evaled1.e_type
			in let type2 = narrow_single evaled2.e_type
			in let type12 = if type1 == type2 then type1
				else raise (Failure "Binary operands don't match.")
			in let type_out = if (op = LeT) || (op = GrT) || (op = LE) ||
													(op = GE) || (op = Eq) || (op = NEq)
				then XFunkBool else type12
			in let st = generalize_single(type12)
			in let st_out = generalize_single(type_out)
			in let binop_func = match type12
			with XFunkInt ->
				let do_binop_int op_f = fun oprnd1 oprnd2 ->
					IntVal(op_f (get_int oprnd1) (get_int oprnd2))
				in (match op
				with Mult -> do_binop_int (fun a b -> a * b)
				| Div -> do_binop_int (fun a b -> a / b)
				| Mod -> do_binop_int (fun a b -> a mod b)
				| Add -> do_binop_int (fun a b -> a + b)
				| Sub -> do_binop_int (fun a b -> a - b)
				| LSh -> do_binop_int (fun a b -> if b >= 0 then a lsl b
							else a lsr -b)
				| RSh -> do_binop_int (fun a b -> if b >= 0 then a lsr b
							else a lsl -b)
				| LeT -> (fun a b -> BoolVal((get_int a) < (get_int b)))
				| GrT -> (fun a b -> BoolVal((get_int a) > (get_int b)))
				| LE -> (fun a b -> BoolVal((get_int a) <= (get_int b)))
				| GE -> (fun a b -> BoolVal((get_int a) >= (get_int b)))
				| Eq -> (fun a b -> BoolVal((get_int a) = (get_int b)))
				| NEq -> (fun a b -> BoolVal((get_int a) != (get_int b)))
				| BAnd -> do_binop_int (fun a b -> a land b)
				| BXor -> do_binop_int (fun a b -> a lxor b)
				| BOr -> do_binop_int (fun a b -> a lor b)
				| _ -> raise (Failure ("Invalid binary operator for integers"))
				)
				| XFunkDouble ->
						let do_binop_double op_f = fun oprnd1 oprnd2 ->
							DoubleVal(op_f (get_double oprnd1)
									(get_double oprnd2))
						in (match op
						with Mult -> do_binop_double (fun a b -> a *. b)
						| Div -> do_binop_double (fun a b -> a /. b)
						| Add -> do_binop_double (fun a b -> a +. b)
						| Sub -> do_binop_double (fun a b -> a -. b)
						| LeT -> (fun a b -> BoolVal((get_double a) < (get_double b)))
						| GrT -> (fun a b -> BoolVal((get_double a) > (get_double b)))
						| LE -> (fun a b -> BoolVal((get_double a) <= (get_double b)))
						| GE -> (fun a b -> BoolVal((get_double a) >= (get_double b)))
						| Eq -> (fun a b -> BoolVal((get_double a) = (get_double b)))
						| NEq -> (fun a b -> BoolVal((get_double a) != (get_double b)))
						| _ -> raise (Failure ("Invalid binary operator for doubles"))
						)
				| XFunkBool ->
						(match op
						with Eq -> (fun a b -> BoolVal((get_bool a) = (get_bool b)))
						| NEq -> (fun a b -> BoolVal((get_bool a) != (get_bool b)))
						| And -> (fun a b -> BoolVal((get_bool a) && (get_bool b)))
						| Or -> (fun a b -> BoolVal((get_bool a) || (get_bool b)))
						| _ -> raise (Failure ("Invalid binary operator for booleans"))
						)
				| XFunkChar ->
						(match op
						with Eq -> (fun a b -> BoolVal((get_char a) = (get_char b)))
						| NEq -> (fun a b -> BoolVal((get_char a) != (get_char b)))
						| _ -> raise (Failure ("Invalid binary operator for chars"))
						)
				| _ -> raise (Failure "Invalid type for unary operator")
			in let conv_eval = match evaled1.result, evaled2.result
			with Some(r1), Some(r2) ->
				let const_result =
					let v1 = get_single(r1)
					in let v2 = get_single(r2)
					in (binop_func v1 v2)
				in { e_node = XSingleConst(type_out, const_result) ;
					result = Some(SingleValue(const_result)) ;
					e_type = st_out }
				(*
			* some boolean operations can be calculated with only one operand,
			* others, we can find out the result, but we should evaluate them
			* anyways for potential side effects
			*)
				| Some(r1), None ->
						let v1 = get_single r1
						in let first_result = match op
						with And -> (match v1
						with BoolVal(false) -> Some(SingleValue(BoolVal(false)))
						| _ -> None
						)
							| Or -> (match v1
							with BoolVal(true) -> Some(SingleValue(BoolVal(true)))
							| _ -> None
							)
							| _ -> None
						in { e_node = XFunkBinExpr(st, evaled1.e_node, op, evaled2.e_node);
							result = first_result ; e_type = st_out }
				| _, _ ->
						{ e_node =
								XFunkBinExpr(st, evaled1.e_node, op, evaled2.e_node);
							result = None ; e_type = st_out }
			in r_env, conv_eval
	(*
	 * Check that function executes to return correct type, and was given
	 * correct parameters
	 *)
	| FunkCallExpr(fc) ->
		let xfc, r_env = check_func_call fc r_env
		in let ret_type =
			let header, _, _ = xfc
			in match header.xret_type
			with None -> raise (Failure "Function call expression has to return a value")
			| Some(rtype) -> rtype
		in r_env, { e_node = XFunkCallExpr(ret_type , xfc) ; result = None ; e_type = ret_type}
	(*
	 * Check that an integer index is given. Have constant result if both
	 * tuple members are constant
	 *)
	| FunkArrExpr(array_expr, index_expr) ->
		(* evaluate both the array and the indexes *)
		let r_env,array_evalue = eval r_env array_expr in
		let r_env,index_evalue = eval r_env index_expr in
		let _ =
			let index_type = index_evalue.e_type in
			match index_type
			with XFunkInt, [] -> true
			| _ -> raise (Failure "Array index must be an integer.")
		in let xvartype = find_lowerx array_evalue.e_type in
		let xarrexpr = XFunkArrExpr(xvartype, array_evalue.e_node , index_evalue.e_node) in
		(* running_env evalue *)
		r_env, { e_node = xarrexpr ; result = None ; e_type = xvartype }
	(*
	 * Similar to FunkCallExpr, but parameters do not need to be checked, but
	 * entire body needs to be checked here
	 *)
	| FunkAsyncExpr(block) ->
		let async_env = filter_in_function r_env None in
		let xblocco, async_env = check_block block async_env in
		let xvartype = match async_env.r_ret_type
			with None -> raise (Failure "async block must return a value")
			| Some(xrt) -> xrt
		in let asyncExpr =  XFunkAsyncExpr(xvartype,xblocco)
		in let evalue = { e_node = asyncExpr ; result = None ;
				  e_type = xvartype }
		in r_env ,evalue

(* Evaluate the lvalues of assignment *)
and eval_lvalues id_list var_type env =
	let xvt, env = extend_vartype var_type env in
	let new_env, variables = List.fold_left (fun (original_env, original_list) next_id ->
		let new_env, new_obj = mapAdd next_id xvt false original_env in
		new_env, new_obj::original_list
	) (env, []) id_list in
	let variables = List.rev variables in
	let var_exprs = List.map (fun v -> XVariable(v)) variables in
	var_exprs, new_env

(* Evaluate the rvalues of assignment	 *)
and eval_rvalue env rvalue =
	match rvalue with
	| ExprRVal x ->
		let new_env,evalue = eval env x
		in new_env, XExprRVal(evalue.e_type, evalue.e_node)
	| FuncRVal x ->
		let func_header, func_body = x in
		let env, xheader, xbody, _ = check_func env func_header func_body None in
		env, XFuncRVal(xheader, xbody)

(* evalute a variable declaration against the environment *)
and eval_vardec env dec =
	let env, rvalues = list_checker eval_rvalue env dec.actual_list in
	let lvalues, env = eval_lvalues dec.id_list dec.var_type env in
	let xa =
		if (List.length rvalues) = 0
		then
			{lvals=[];rvals=[]}
		else
			{lvals=lvalues;rvals=rvalues}
	in
	let _ = check_xassignment xa in
	env, xa

and check_func_call fc env =
	let (f_expr, f_rvalues) = fc in 
	(* evaluate rvalues *)
	let env, evaluated_rvalues = list_checker eval_rvalue env f_rvalues in 
	(* helper function: takes an X function call*)
	let check_param_types xfc =
		(*made of x function header, called func and rvals *)
		let xhdr, _, rvals = xfc in
		let param_types = 
			List.map (fun p_v -> p_v.xbare_type) xhdr.xparams
		in let n_formal = List.length xhdr.xparams
		in let n_actual = List.length rvals
		(* check if the number of parameters is the same *)
		in let _ = if (n_formal = n_actual) then ()
			else raise (Failure ("Expected " ^ string_of_int(n_formal) ^
				    " parameters, but " ^ string_of_int(n_actual) ^ " passed in\n"))
		in
		if (check_rvalues_types param_types rvals) then xfc
		else raise (Failure "Function parameters are of wrong type.")
	in
	let handle_generic env generic_func =
		let env, fevalue = eval env generic_func
		in let function_vartype = fevalue.e_type
		in let function_header = (match (function_vartype) with
			| XFunkFunc(header), [] -> header
			| _ -> raise (Failure "This is not a function")
		)
		in check_param_types (function_header, GenericCall(fevalue.e_node), evaluated_rvalues), env
	in
	(match f_expr with
	| Variable v ->
		(match v.id with
		| "print" -> ({xret_type=None; xparams = []}, PrintCall, evaluated_rvalues), env
		| "double2int" -> check_param_types ({xret_type=Some(generalize_single XFunkInt) ; xparams = [{ xbare_type = (XFunkDouble,[]) }]}, 
			Double2IntCall, evaluated_rvalues), env
		| "int2double" -> check_param_types ({xret_type=Some(generalize_single XFunkDouble); xparams = [{ xbare_type = (XFunkInt,[]) }]}, 
			Int2DoubleCall, evaluated_rvalues), env
		| "bool2int"-> check_param_types ({xret_type=Some(generalize_single XFunkInt); xparams = [{ xbare_type = (XFunkBool,[]) }]}, 
			Boolean2IntCall, evaluated_rvalues), env
		| "int2bool"->  check_param_types ({xret_type=Some(generalize_single XFunkBool); xparams = [{ xbare_type = (XFunkInt,[]) }]}, 
			Int2BooleanCall, evaluated_rvalues), env
		| _ -> handle_generic env f_expr
		)
	| generic_func -> handle_generic env generic_func
	)

and check_rvalues_types types rvalues =
	let get_rtype = function
		| XExprRVal(rt, _) -> rt
		| XFuncRVal(hdr, _) -> generalize_single (XFunkFunc hdr)
	in
	(* Match lvalues with rvalues in semantics *)
	let ret_value =
		if (List.length types) = (List.length rvalues) 
		then
		List.fold_left2
			(fun _ t rv ->
				if (match_type t (get_rtype rv)) 
				then
					true
				else
					raise (Failure ("Left" ^ "right value types do not match."))
			) true types rvalues
		else
			false
	in
		ret_value

(** check the assignment statment types*)
and check_xassignment xa =
	let get_ltype = function
		| XVariable(v) -> v.obj_type
		| XFunkArrExpr(xvt, _, _) -> xvt
		| _ -> raise (Failure "Left value must be variable or array.")
	in
	let ltypes = List.map get_ltype xa.lvals
	in
	check_rvalues_types ltypes xa.rvals

and check_statement stmt env =
	match (stmt) with
	| Break -> XBreak, env
	| Assignment (expressions, rvalues) ->
		let right_checker = list_checker eval_rvalue
		in let env, xrvalues = right_checker env rvalues
		in let left_checker = list_checker eval
		in let env, lvalues = left_checker env expressions
		in let xexprs = List.map (fun x -> x.e_node) lvalues
		in let xa = { lvals = xexprs ; rvals = xrvalues }
		in let _ = check_xassignment xa
		in XAssignment(xa), env
	| Declaration x ->
		let env, xvardec = eval_vardec env x in
		XAssignment(xvardec), env
	| FunctionCall x ->
		let call, env = check_func_call x env
		in XFunctionCall(call), env
	| Block x ->
			let block, env = check_block x env 
				in XBlock(block), env
	| ForBlock (a,b,c,d) ->
			let am, env = (match a with
				| None -> None, env
				| Some(stmt) -> let as1, as2 = check_statement stmt env in Some(as1), as2) in
			let bm, env = (match b with
			| None -> None, env
			| Some(expr) -> let bs1, bs2 = eval env expr in Some(bs2.e_node), bs1) in
			let cm, env = (match c with
				| None -> None, env
				| Some(stmt) -> let cs1, cs2 = check_statement stmt env in Some(cs1), cs2) in
			let dm, env = check_block d env in
			XForBlock(am, bm, cm, dm), env
	| IfBlock (x,y) ->
			let env, xexpr = eval env x in
			let xblock, env = check_block y env in
			XIfBlock(xexpr.e_node, xblock), env
	| IfElseBlock (x,y,z) ->
			let env,xm  = eval env x in
			let ym, env = check_block y env in
			let zm, env = check_block z env in
			XIfElseBlock(xm.e_node, ym,zm) , env
	| WhileBlock(x,y) ->
			let env, xexpr = eval env x in
			let xblock, env = check_block y env in
			XWhileBlock(xexpr.e_node, xblock), env
	| Return x ->
			let replace_ret_type old_env new_has_ret new_type = { scope = old_env.scope ;
									      new_objects = old_env.new_objects ;
									      new_objects_ordered = old_env.new_objects_ordered ;
									      current_block = old_env.current_block ;
									      next_free = old_env.next_free ;
									      next_bound = old_env.next_bound ;
									      has_ret = new_has_ret ; r_ret_type = new_type;
									      free_objects_ordered = old_env.free_objects_ordered }
			in match (x) with
			(* Check return values for consistency *)
			(* Does return a value. Must be consistent with any old type *)
			| Some(rval) ->
					let new_env, new_xvalue = eval_rvalue env rval
					(* figure out returned type *)
					in let ret_type = (match new_xvalue
						with XExprRVal(xvt, rexpr) -> xvt
						| XFuncRVal(anon) -> generalize_single (XFunkFunc(fst anon)))
					in let has_ret, r_ret_type = if (env.has_ret) then
							(* match running return type to new return type *)
							let type_matched = match env.r_ret_type
								with None -> raise (Failure ("Function does not need to return a value," ^
										    " according to previous return statements."))
								| Some(r_ret) -> match_type r_ret ret_type
							in (if (type_matched) then true, env.r_ret_type
						 		else raise (Failure "Return statements do not match in type."))
							(* always alright if no previous return type *)
						else (true, Some(ret_type))
					in XReturn(Some(new_xvalue)), replace_ret_type new_env has_ret r_ret_type
			| None ->
					(* if there is a previous return type, it must be None *)
					let has_ret, r_ret_type = if (env.has_ret) then
						(if env.r_ret_type = None then env.has_ret, env.r_ret_type
						 else raise (Failure "Function needs to return a value, according to previous return statements."))
					(* always alright if no previous return type *)
					else (true, None)
					in XReturn(None), replace_ret_type env has_ret r_ret_type
					
and check_block stmts env =
	let deeper_env = { scope = env.scope; new_objects = env.new_objects;
		current_block = { own_func =
				env.current_block.own_func;
			own_block = env.current_block.own_block + 1 };
		next_free = env.next_free;
		next_bound = env.next_bound ;
		new_objects_ordered = env.new_objects_ordered ;
		has_ret = env.has_ret ; r_ret_type = env.r_ret_type;
		free_objects_ordered = env.free_objects_ordered}
	in let old_env = env
	in let xstmts, new_env = List.fold_left
			(fun (old_xstmts, old_env) stmt ->
				let xstmt, env = check_statement stmt old_env
				in (xstmt::old_xstmts, env)) ([], deeper_env) stmts

	in let env = filter_out_block new_env old_env
	in
	{ xstmts = List.rev xstmts ; need_copy = List.rev env.free_objects_ordered ;
		declared = List.rev env.new_objects_ordered }, env
		
and extend_func_header fh r_env =
	(*TO_XRET: helper function to extend function header to XFH *)
	let to_xret r_env = function
		| None -> None, r_env
		| Some(rt) -> let xrt, r_env = extend_vartype rt r_env
				in Some(xrt), r_env
	(* TO_XPARAM: helper function that should check parameters of a function and put them in the scope *)
	in let to_xparam prms r_env =
		let xprms, r_env = List.fold_left
				(* ok so this is the function that does the job *)
				(fun (old_prms, r_env) prm ->
					(* first extend the vartype of the parameter *)
					let xbt, r_env = match prm.bare_type
					with None -> raise (Failure
								"Missing paramter type in function header")
					| Some(bt) -> extend_vartype bt r_env
					(* then create this object which is just a wrapper of the type ?!*)
					in let xprm = { xbare_type = xbt }
					in (xprm::old_prms, r_env)
				)
				([], r_env) prms
		in (List.rev xprms), r_env
	in let xparams, r_env = to_xparam fh.params r_env in 
	let xrt, r_env = to_xret r_env fh.ret_type
	in { xret_type = xrt ; xparams = xparams }, r_env



(** Check function body and find the return type
		(running_env) r_env: the most recent environment, as passed by caller
		(statement list) body: list of statements in the body of the function level
		returns updated r_env, updated body, which will be futher used in
		check_func, return type, and id_objects that need to be copied from a
		higher scope *)
and check_func_body r_env body =
	let func_env, xbody = List.fold_left
				(fun (old_env, old_xbody) stmt ->
					let xstmt, r_env = check_statement stmt
								old_env
					in (r_env, xstmt::old_xbody)
				) (r_env, []) body
	in
		
	r_env, { xstmts = List.rev xbody ; 
	need_copy = List.rev func_env.free_objects_ordered;
	declared = List.rev func_env.new_objects_ordered }, func_env.r_ret_type

(**
* Check entire function, including header and body USED by Rvalue and FuncDec
* (running_env) r_env: the most recent environment, as passed by caller
* (func_dec_header) header: header of function, which gives the type that
* return statements should have, and types of the formal parameters, which
* will be used as variables in the body
* (statement list) body: list of statements in the body of the function
* returns updated r_env, tuple containing header and updated body, which
* will be futher used as an anon value, and id_objects that need to be
* copied from higher scope
*)
and check_func r_env header body name_opt =
	let xheader, r_env = extend_func_header header r_env in
	let r_env, global_obj, name_opt = match name_opt
		with None -> r_env, None, None
		| Some(name) ->
			let r_env, global_obj = mapAdd name (XFunkFunc(xheader),[]) false
						r_env in
			r_env, Some(global_obj), Some(name)
	in
	let func_env = filter_in_function r_env name_opt in
	let func_env = List.fold_left2 (fun old_env prm xprm ->
		(* add name to the env *)
		fst (mapAdd prm.id xprm.xbare_type false old_env))
		func_env header.params xheader.xparams
	in
	let func_env, body, rtype = check_func_body func_env body in
	let hdr_rtype = xheader.xret_type in
	let _ = match rtype
		with None -> (match hdr_rtype
				with None -> ()
				| _ -> raise (Failure
					"Function returns no value, but the header says it does")
			     )
		| Some(rt) -> (match hdr_rtype
			with Some(hrt) -> if (match_type hrt rt) then ()
				else raise (Failure "Return type of function does not match header")
			| None -> raise (Failure
				"Function returns value, but the header says it does not")
			     )
	in
	let check_if_copy_required_variables_in_scope r_env xblock =
		let r_body_need_copy = ref [] in
		let r_r_env = ref r_env in
		List.iter (fun (dest,org) ->
			if (List.mem org r_env.new_objects_ordered) || not (is_not_duplicate r_env org.name)
			then
				r_body_need_copy := (dest,org)::!r_body_need_copy
			else
				let n_env, n_variable = mapAdd org.name org.obj_type true !r_r_env in
				r_r_env := n_env;
				r_body_need_copy:= (dest,n_variable)::!r_body_need_copy;
			) xblock.need_copy;
		!r_r_env, {xstmts = xblock.xstmts; need_copy = (!r_body_need_copy); declared = xblock.declared}
	in 
	let new_r_env, new_r_body = check_if_copy_required_variables_in_scope r_env body in
	new_r_env, xheader, new_r_body, global_obj

	

let check_declaration declaration env =
	match declaration with
	| Newline x -> None, env
	| Funcdec f ->
		(* invoke the check_func keeping the resulting env for the variables,
		 * but not propagating it directly *) 
		let env, xheader, xbody, obj_opt = (check_func env f.func_header f.body (Some f.fid)) in
		let global_obj = match obj_opt
			with Some(g_o) -> g_o
			| None -> raise (Failure "check_func should have returned global_object")
		in
		Some(XFuncdec({ global_id = global_obj ; xfid = f.fid ; xfunc_header = xheader ; xbody = xbody })), env 

	| Vardec x ->
		let env, xvardec = eval_vardec env x in
		Some(XVardec(xvardec)), env

let check_ast_type prog =
	let check_dec_foldable (old_list, old_env) line =
		let xdecl, n_env = check_declaration line old_env in
			match xdecl with 
			| Some(xdecl) -> (xdecl::old_list), n_env
			| None -> old_list, n_env
	in let prog_list, env = List.fold_left (check_dec_foldable) ([], new_r_env) prog
	in (List.rev prog_list), List.rev env.new_objects_ordered
