open Commponents
open Xcommponents

(* maps parameter types to the the C struct *)
module ParamTable = Map.Make(String)

(*
 * aggregator for global program code
 *)
type struct_table = {
  struct_decs : string ; (* struct declarations *)
  prog_decs : string ; (* C function declarations *)
  (* paramater structs that have already been declared *)
  params : string ParamTable.t
}

(*
 * helper function for writing C-style function calls
 * (string) func_name: name or C-style expression for C function or macro
 * (string list) params: handlers for expressions
 * returns C function call
 *)
let c_func_call func_name params =
  let params_code = String.concat ", " params
  in func_name ^ "(" ^ params_code ^ ")"

let default_env = "self->scope"

let global_scope = "global"

let is_global iobj = iobj.scope_setting.own_func = global_scope

(*
 * get C expression to acces free variable
 * (string) struct_id: the function instance struct
 * (int) free_id: var_id of a free id_object
 * (string) estruct: type of the environment structure for the function
 * returns C expression that accesses free variable in environment of
 *   function
 *)
let get_free_name struct_id free_id estruct =
  "((" ^ estruct ^ " *) " ^ struct_id ^ ")->v_" ^ string_of_int(free_id)

let get_bound_name id = "bv_" ^ string_of_int(id)

let get_global_name id = "gv_" ^ string_of_int(id)

let get_non_free_name iobj = if iobj.scope_setting.own_func = global_scope
  then get_global_name iobj.var_id
  else get_bound_name iobj.var_id

(*
 * get C expression for free or bound variable inside a function
 * (id_object) iobj: ocaml representation of the variable to access
 * (string) estruct: type of the environment structure for the function, and
 *                   contains iobj, if it is a free variable
 *)
let get_name iobj estruct estruct_var = if iobj.is_free
  then get_free_name estruct_var iobj.var_id estruct
  else get_non_free_name iobj

(*
 * extract single_funk_value from from funk_value
 * (funk_value): assumed to be SingleValue
 * returns the single_funk_value inside the single_value
 *)
let get_single = function
  | SingleValue(sv) -> sv
  | ArrayValue(_) -> raise (Failure "Expected single value")

(*
 * Helper functions to extract primitive values from single_funk_value
 * Each function, "get_[type]" takes the following form:
 * (single_funk_value): assumed to contain a value of type [type]
 * returns value of type [type]
 *)
let get_int = function
  | IntVal(i_v) -> i_v
  | _ -> raise (Failure "Expected int for get_int")
let get_double = function
  | DoubleVal(d_v) -> d_v
  | _ -> raise (Failure "Expected double for get_double")
let get_bool = function
  | BoolVal(b_v) -> b_v
  | _ -> raise (Failure "Expected bool for get_bool")
let get_char = function
  | CharVal(c_v) -> c_v
  | _ -> raise (Failure "Expected char for get_char")

(*
 * Returns string of how type would appear in one of the C-library functions
 * (xsingle_vartype): type to convert to string
 * returns lower-case string corresponding to type for primitive functions only
 *)
let get_type_name = function
  | XFunkInt -> "int"
  | XFunkDouble -> "double"
  | XFunkChar -> "char"
  | XFunkBool -> "bool"
  (* there are no families of C functions that can directly access a function *)
  | _ -> raise (Failure "Cannot access functions like this")

(*
 * generate C function name for setting a primitive value
 * (xsingle_vartype) st: type of value to set
 * returns name of type-appropriate function to write to var struct
 *)
let set_func_type st = "set_" ^ (get_type_name st)

(*
 * generate C function name for reading a primitive value
 * (xsingle_vartype) st: type of value to read
 * returns name of type-appropriate function to read from var struct
 *)
let get_func_type st = "get_" ^ (get_type_name st)

(*
 * Given non-pointer variable, returns C expression for its reference
 * (string) var: C variable of a non-pointer type
 * returns reference to var
 *)
let point var = "(&" ^ var ^ ")"

(*
 * returns C expression that returns raw, primitive value of a constant or
 * a non-pointer variable
 * (string) sv: value could be constant or non-pointer C variable
 * (xsingle_vartype) st: type of primitive value
 * (bool) is_var: is sv a variable? If false, it's a constant
 * return C expression that returns value contained in sv
 *)
let get_single_raw sv st is_var =
  let bare = if is_var then
      let getter_func = get_func_type st
      in c_func_call getter_func [point sv]
    else sv
  in "(" ^ bare ^ ")"

(*
 * helper function for creating variable name
 * (string) prefix: alphabetic prefix, eg. "t"
 * (int) id: index of variable in function
 * returns C-style variable
 *)
let get_field_name prefix id = prefix ^ "_" ^ (string_of_int id)

(*
 * generate structure
 * (string) name: struct's name, including "struct"
 * (string) prefix: alphabetic prefix, eg. "t"
 * (int) count: number. indexes are expected to include 0 to count - 1
 * returns C-style struct declaration
 *)
let gen_struct name prefix count =
  let rec get_member_codes member_codes current =
    if current < count then
      (get_member_codes
        (("\tstruct var " ^
          (get_field_name prefix current) ^ ";\n")::member_codes)
         (current + 1))
    else member_codes
  in name ^ "{ \n" ^
    (String.concat "\n" (List.rev (get_member_codes [] 0))) ^
    "};\n"

(*
 * get the C-style struct name
 * (string) sname: bare name of struct
 * returns
 *)
let get_struct_type sname = "struct " ^ sname

(*
 * get string encoding for list of types for parameters. The string is
 * used to look up the param structure in a ParamTable
 * params
 * (xvar list): the list of parameters from xfunc_dec_header
 * returns string representation of parameters
 *)
let param_encode params = string_of_int (List.length params)

(*
 * get C struct type for param struct
 * (xvar list): the list of parameters from xfunc_dec_header
 * returns C code for the param struct type
 *)
let get_pstruct_type params = get_struct_type ("param_" ^ (param_encode params))

(*
 * helper function for getting param struct name
 * (var list) params: list of parameter variables
 * (struct_table) struct_tbl: current struct declarations and map of structures 
 *                            returns
 *   struct name (including "struct ")
 *   the updated struct_tbl, which will have the new name added if it was not
 *     there before
 *   the code for declaring the struct, if the type is new, and contains
 *     at least 1 parameter. Otherwise, return empty string. Also return
 *     empty string if there are no parameters.
 *)
let get_param_struct params struct_tbl =
  if ((List.length params) = 0) then "", struct_tbl else
    let encoding = param_encode params in
    if ParamTable.mem encoding struct_tbl.params
      then (ParamTable.find encoding struct_tbl.params, struct_tbl)
      else
        let param_struct = get_pstruct_type params in
        let param_dec = if (List.length params >= 1)
          then gen_struct param_struct "p" (List.length params)
          else ""
        in (param_struct,
          { params = ParamTable.add encoding param_struct struct_tbl.params ;
            struct_decs = struct_tbl.struct_decs ^ param_dec ;
            prog_decs = struct_tbl.prog_decs })

(*
 * get name for per-function free-variable struct
 * (string) fname: function name
 * returns "struct [fname]_env"
 *)
let get_estruct_type fname = get_struct_type fname ^ "_env"

(*
 * Generate struct for free variables
 * (string) fname: function name
 * (id_object list) needed: needed sources for free variables, ie. the need_copy
 *                          field of the function body's xblock
 * return C-style declaration of free-variable struct
 *)
let get_env_struct fname needed =
  if (List.length needed = 0) then ("", "")
    else let env_struct = get_estruct_type fname in
      (env_struct, gen_struct env_struct "v" (List.length needed))

(*
 * get pointer to array member of var struct
 * (string) name: name of the struct var value that is an array
 * (string) index: array index. Could be a constant or variable
 * returns C-style expression for getting the pointer to the arrat member
 *)
let get_arr_memb name index = name ^ "->val.array[" ^ index ^ "]"

(*
 * generic helper for traversing a var that could be an array
 * (string) name: name of the struct var value that is an array or single value
 * (int) depth: number of levels in array. a single value has depth 0
 * (fun string -> string) pre_iter: if the object is an array, the action to
 *                                  take before iterating through the array
 * (fun string -> string) single_handler : if the object is a single value,
 *                                         the action to take
 * returns C-style nested for loops for handling all the single values the
 * variable contains
 *)
let rec trav_arrs name depth pre_iter single_handler = match depth
  with 0 -> single_handler name
  | d -> let memb_access = point (get_arr_memb name "arr_i")
         in let memb_name = "member_" ^ string_of_int(depth)
         in "{\n" ^
         "int arr_i;\n" ^
         (pre_iter name) ^
         "for (arr_i = 0; arr_i < " ^ name ^ "->arr_size; arr_i++) {\n" ^
         "struct var *" ^ memb_name ^ " = " ^ memb_access ^ ";\n" ^
         (trav_arrs memb_name (depth - 1) pre_iter single_handler) ^
         "\n}\n" ^
         "\n}\n"


(* "struct function_instance" *)
let func_inst_type = get_struct_type "function_instance"

(*
 * string to cast a variable to a C pointer type
 * (string) tn: C type to cast an expression to
 * (string) castee: the expression to cast as [tn] *
 *)
let cast_ptr tn castee = "(" ^ tn ^ " *) (" ^ castee ^ ")"

(*
 * output temporary function and its initialization code
 * (int) id: the temporary id of variable, derived from temp_cnt
 * returns temporary variable, and its initialization code
 *)
let gen_temp id =
  let t_name = "t_" ^ string_of_int(id)
  in let t_dec = "struct var " ^ t_name ^ ";\n" ^
    (c_func_call "init_var" [point t_name]) ^ ";\n"
  in t_name, t_dec

(*
 * allocates function instance
 * (string) temp: base name for instance to which "_inst" is appended.
 *                Could be variable for var struct.
 * returns
 *   pointer variable to function instance
 *   code to malloc function instance 
 *)
let alloc_inst temp = let inst_var = temp ^ "_inst"
  in let init_inst = func_inst_type ^ " *" ^ inst_var ^
    " = malloc(sizeof(" ^ func_inst_type ^ "));\n"
  in inst_var, init_inst

(*
 * wrapper for alloc_inst that also has code for setting the instance of temp
 * (string) temp: base name and variable of type struct var
 * returns
 *   pointer variable to function instance
 *   code to malloc function instance, and set it to ptr of temp
 *)
let set_alloc_inst temp =
  let inst_var, init_inst = alloc_inst temp
  in let set_alloced = temp ^ ".val.ptr = " ^ inst_var ^ ";\n"
  in inst_var, init_inst ^ set_alloced

(*
 * like set_alloc_inst, but given a pointer, and bare variable is not available
 * (string) dst_ptr: C expression for pointer to struct var
 * returns
 *   pointer variable to function instance
 *   code to malloc function instance, and set it to ptr of temp
 *   updated temp_cnt
 *)
let set_alloc_inst_ptr dst_ptr =
  let temp_var, _ = gen_temp 0
  in let inst_var, init_inst = alloc_inst temp_var
  in let set_alloced = dst_ptr ^ "->val.ptr = " ^ inst_var ^ ";\n"
  in inst_var, init_inst ^ set_alloced

(*
 * Helper function for wrapping C statements inside block braces
 * (string) bare_blocks: lines of C code
 * returns bare_blocks inside braces
 *)
let enclose bare_block = "{\n" ^ bare_block ^ "\n}\n"

(*
 * generic helper function for creating a function definition out of a function
 * body
 * (string) fname: function name
 * (string) fbody_code: C-style function body code
 * (string) ret_type: C return type, which could be "void"
 * (string list) params: parameter types and names
 * returns C-style function definition
 *)
let wrap_c_func fname fbody_code ret_type params =
  let param_strs = String.concat ", " params
  in "static " ^ ret_type ^ fname ^ "(" ^ param_strs ^ ")\n" ^
  (enclose fbody_code)

(*
 * helper functions for generating the names of the C functions for each
 * funk function that will be the members of struct function_instance
 * (string) fname: internal name of function
 *)
(* returns name of .function member *)
let get_inst_function fname = fname ^ "_function"
(* returns name of .init member *)
let get_inst_init fname = fname ^ "_init"
(* returns name of .copy member *)
let get_inst_copy fname = fname ^ "_copy"

(*
 * Creates C expression that returns cast pointer to function instance
 *)
let get_func_inst var = "(" ^ var ^ "->val.ptr" ^ ")"

(*
 * calls initialization function from a function var struct on an instance
 * (string) dst_inst_ptr: C pointer to function instance to initialize
 * (string) initer_ptr: C pointer to var struct to initialize as function
 * returns call of init function in variables function instance on dst_ptr
 *)
let gen_init_func dst_inst_ptr initer_ptr =
  c_func_call ((get_func_inst initer_ptr) ^ "->init") [dst_inst_ptr]

(*
 * copy function instance from var struct pointer to var struct
 * (string) dst_ptr: C expression that is of type "struct var *"
 * (string) src_ptr: C expression that is of type "struct var *"
 * returns C code to allocate instance at destination, code to initialize it
 *   copy source's env to it
 *)
let copy_func dst_ptr src_ptr =
  let copy_line = (c_func_call "copy_funcvar"
    [dst_ptr ; src_ptr]) ^ ";\n"
  in copy_line

(*
 * helper function that puts function code into a C function
 * (string) fname: function name
 * (string) fbody_code: function's C code
 *)
let wrap_func_body fname fbody_code =
  let fbody_code =
    func_inst_type ^ " *self = (" ^ func_inst_type ^ " *)data;\n" ^
    fbody_code
  in wrap_c_func fname fbody_code "void *" ["void *data"]

(*
 * generates default copy function output by gen_expr
 * (string) src: variable of type struct var
 * returns function for generating 1-level copy variable function
 *)
let default_copy src =
  fun dst -> (c_func_call "copy_primitive" [point dst ; point src]) ^ ";\n"

(*
 * generates copy function output by gen_expr for non-primitive types
 * (string) src: variable of type struct var
 * returns function for generating 1-level copy variable function
 *)
let shallow_copy src =
  fun dst -> (c_func_call "shallow_copy" [point dst ; point src]) ^ ";\n"

(*
 * generates internal name for anonymous function. Can also be used for async
 * blocks
 * (string) fname: function that contains this function. Could be anonymous,
 *                 in which case its name was also created using this function
 * (int) temp_cnt: id used to name this function
 * returns name of anon and possibly-updated temp_cnt
 *)
let get_anon_name fname temp_cnt =
  fname ^ "anon" ^ string_of_int(temp_cnt), temp_cnt

(*
 * Generates C code for a function body, which could be an anonymous block
 * (string) fname: internal version of name
 * (xbody) fbody: function body or anonymous block
 * (xvar list) params: function's paramaters. always empty for async blocks
 * (struct_table) struct_tbl: current struct declarations and map of structures 
 *                            returns
 * returns
 *   the function to instantiate the function
 *   the updated struct_tbl
 *)
let rec gen_func_body fname fbody params struct_tbl =
  let estruct = get_estruct_type fname
  in let c_body, _, struct_tbl = gen_block fbody estruct fname 0
    struct_tbl
  in let n_params = List.length params
  in let pstruct_var = "my_params"
  in let pstruct, dec_pstruct, struct_tbl = if n_params > 0 then
    let pstruct, struct_tbl = get_param_struct params struct_tbl
    in let dec_pstruct = pstruct ^ " *" ^ pstruct_var ^ " = " ^
      (cast_ptr pstruct "self->params") ^
      ";\n"
    in pstruct, dec_pstruct, struct_tbl
    else "", "", struct_tbl
  in let get_param = fun i -> point (pstruct_var ^ "->p_" ^ string_of_int(i))
  in let bound_decs, _ = List.fold_left
    (fun (old_code, id) dec -> 
      let bv_name = "bv_" ^ string_of_int(id)
      in let param_copy = if id < n_params
        then (c_func_call "shallow_copy" [point bv_name ; get_param id]) ^ ";\n"
        else ""
      in let bv_dec = "struct var " ^ bv_name ^ ";\n" ^
        (c_func_call "init_var" [point bv_name]) ^ ";\n" ^ param_copy
      in (old_code ^ bv_dec, id + 1)
    ) ("", 0) fbody.declared
  in let initer = (fun inst_var env_var temp_cnt struct_tbl ->
    gen_new_func inst_var fbody fname estruct env_var temp_cnt struct_tbl)
  in let estruct_name, estruct_dec =
    get_env_struct fname fbody.need_copy
  in let func_def = wrap_func_body (get_inst_function fname)
                    (dec_pstruct ^ bound_decs ^ c_body ^
		     "\nRETURN: return NULL;\n")
  in let func_init = gen_init fname fbody estruct_name pstruct
  in let func_copy = gen_copy fname fbody estruct_name
  in let struct_tbl = { struct_decs = struct_tbl.struct_decs ^ estruct_dec ;
                        prog_decs = struct_tbl.prog_decs ^ func_def ^
                        func_copy ^ (fst func_init) ;
                        params = struct_tbl.params }
  in initer, struct_tbl
(*
 * generates code to evaluate expression
 * (xexpr) expr_nd: expression node of SAST
 * (string) estruct: environment structure
 * (string) fname: function name
 * (int) temp_cnt:-k 0123456789abcdef cd running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 *                            structures returns
 * returns
 *   C representation of value
 *   is value a temporary variable, rather than raw constant?
 *   function to generate code to copy value into a var struct
 *   C code to generate value
 *   the value's type
 *   new temp_cnt
 *   new struct_tbl
 *)
and gen_expr expr_nd estruct fname temp_cnt struct_tbl =
  match expr_nd
  (* different code and copier for different types of expressions *)
  (* a simple constant; no C variable needed *)
  with XSingleConst(vt, sv) ->
    let const_copy type_pre = fun src -> fun dst ->
      (c_func_call ("set_" ^ type_pre)
       [point dst ; src]) ^ ";\n"
    in let val_code, val_copy = (match sv
    with IntVal(iv) -> string_of_int(iv), const_copy "int"
    | DoubleVal(dv) -> string_of_float(dv), const_copy "double"
    | BoolVal(dv) -> (if dv then "1" else "0"), const_copy "bool"
    | CharVal(cv) -> "\'" ^ (Char.escaped cv) ^ "\'", const_copy "char"
    | _ -> raise (Failure "Only support primitives as XSingleConst")
    )
    in val_code, false, val_copy val_code, "", (vt, []), temp_cnt, struct_tbl
  (* array has its own temp variable, and needs to evaluate each member *)
  | XArrayLit(xvt, xrvs) ->
      let arr_temp, arr_gen = gen_temp temp_cnt
      in let arr_ptr = point arr_temp
      in let temp_cnt = temp_cnt + 1
      in let size_expr = List.hd (snd xvt)
      in let size_temp, size_is_temp, _, size_gen, size_xvt, temp_cnt,
        struct_tbl = gen_expr size_expr estruct fname temp_cnt struct_tbl
      in let arr_gen = arr_gen ^ size_gen ^
        (c_func_call "init_array"
         [arr_ptr ; get_single_raw size_temp XFunkInt size_is_temp]) ^ ";\n"
      in let _, arr_fill, temp_cnt, struct_tbl = List.fold_left
        (fun (i, old_code, old_cnt, old_tbl) xrv ->
          let memb_code, handle, _, temp_cnt, struct_tbl =
            gen_rvalue xrv estruct fname old_cnt old_tbl
          in let fill_memb = (c_func_call "set_element" [arr_ptr;
                                                        (string_of_int i) ;
                                                        (point handle)]) ^ ";\n"
          in i + 1, old_code ^ memb_code ^ fill_memb, temp_cnt, struct_tbl
        )
        (0, "", temp_cnt, struct_tbl) xrvs
      in let arr_gen = arr_gen ^ arr_fill
      in arr_temp, true, shallow_copy arr_temp, arr_gen, xvt, temp_cnt,
        struct_tbl
  (* variable object. Access free or bound variable *)
  | XVariable(iobj) -> let name = get_name iobj estruct default_env
    in let temp_cnt = temp_cnt + 1
    (* do allow inside effects due to anonymous functions *)
    in let is_function = match iobj.obj_type
      with XFunkFunc(_), [] -> true
      | _ -> false
    in let fetch_var, var_temp, temp_cnt =
      if (is_global iobj || is_function) then "", name, temp_cnt
      else
        let var_temp, var_gen = gen_temp temp_cnt
        in let temp_cnt = temp_cnt + 1
        in let copy_var = gen_copy_var_wrap (point var_temp) (point name)
          iobj.obj_type estruct
        in let fetch_var = var_gen ^ copy_var
        in fetch_var, var_temp, temp_cnt
    in var_temp, true, default_copy var_temp, fetch_var, iobj.obj_type,
      temp_cnt, struct_tbl
    (*in name, true, default_copy name, "", iobj.obj_type, temp_cnt,
      struct_tbl*)
  (*
   * Unary expression. Evaluate expression first, then apply a primitive C
   * operation on it
   *)
  | XFunkUnExpr(xvt, op, xpr) ->
    let un_temp, un_gen = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let opstring = match op
      with IntNeg -> "-"
      | BitNot -> "~"
      | Not -> "!"
      | Positive -> ""
    in let st = fst xvt
    in let xpr_temp, xpr_is_temp, _, xpr_gen, xpr_xvt, temp_cnt,
      struct_tbl = gen_expr xpr estruct fname temp_cnt struct_tbl
    in let set_result = c_func_call (set_func_type st) [point un_temp ;
                        opstring ^
                        (get_single_raw xpr_temp st xpr_is_temp)] ^ ";\n"
    in let un_gen = xpr_gen ^ un_gen ^ set_result
    in un_temp, true, default_copy un_temp, un_gen, xvt, temp_cnt,
      struct_tbl
  (*
   * binary function. Generate code for evaluating both expressions,
   * but depending on the operation and the result of one of the expressions,
   * the other expression may not be executed during runtime, or a different
   * operator is used
   *)
  | XFunkBinExpr(xvt, xpr0, op, xpr1) ->
    let bin_temp, bin_gen = gen_temp temp_cnt
    in let bin_ptr = point bin_temp
    in let temp_cnt = temp_cnt + 1
    in let st = fst xvt
    in let t0, is_temp0, _, gen0, xvt0, temp_cnt,
      struct_tbl = gen_expr xpr0 estruct fname temp_cnt struct_tbl
    in let t1, is_temp1, _, gen1, xvt1, temp_cnt,
      struct_tbl = gen_expr xpr1 estruct fname temp_cnt struct_tbl
    (* most operations always need to evaluate both expressions *)
    in let eval_both gen0 gen1 = gen0 ^ gen1
    (* for most operations, always use the same C operator *)
    in let single_op_raw op = fun val0 val1 ->
        c_func_call (set_func_type st) [bin_ptr ; val0 ^ op ^ val1 ] ^ ";\n"
    (* wrapper for single_op that figures out how to get the input values *)
    in let single_op op xvt0 xvt1 = fun t0 t1 is_temp0 is_temp1 ->
      single_op_raw op (get_single_raw t0 (fst xvt0) is_temp0)
        (get_single_raw t1 (fst xvt1) is_temp1)
    (* condition on the expressions' results determines what operator to use *)
    in let cond_op op0 op1 xvt0 xvt1 cond = fun t0 t1 is_temp0 is_temp1
      -> let val0 = get_single_raw t0 (fst xvt0) is_temp0
      in let val1 = get_single_raw t1 (fst xvt1) is_temp0
      in "if (" ^ (cond val0 val1) ^ ") {\n" ^
        single_op_raw op0 val0 val1 ^
        "} else {\n" ^
        single_op_raw op1 val0 val1 ^
        "}\n"
    (* most operations always evaluate both expressions and use one operand *)
    in let default_code op xvt0 xvt1 = fun gen0 gen1 t0 t1 is_temp0
      is_temp1 ->
      (eval_both gen0 gen1) ^
      (single_op op xvt0 xvt1 t0 t1 is_temp0 is_temp1)
    (* evaluate second expression if first condition is met *)
    in let cond_exec1 op xvt0 xvt1 cond only0 = fun gen0 gen1 t0 t1 is_temp0
      is_temp1 ->
      let val0 = (get_single_raw t0 (fst xvt0) is_temp0)
      in let val1 = (get_single_raw t1 (fst xvt1) is_temp1)
      in gen0 ^
        "if (" ^ (cond val0) ^ ") {\n" ^
        gen1 ^
        single_op_raw op val0 val1 ^
        "} else {\n" ^
        only0 val0 ^
        "}\n"
    (*
     * for And and Or, if condition is not met, we just use the first
     * expression
     *)
    in let get0 val0 = c_func_call (set_func_type st) [bin_ptr ; val0] ^
      ";\n"
    (*
     * evaluate both expressions, but condition determines operation
     *)
    in let both_cond op0 op1 xvt0 xvt1 cond = fun gen0 gen1 t0 t1 is_temp0
      is_temp1 ->
      (eval_both gen0 gen1) ^ (cond_op op0 op1 xvt0 xvt1 cond t0 t1 is_temp0
                               is_temp1)
    (*
     * condition function that compares expression output to some value
     *)
    in let comp_cond eq target =
      fun cmpval -> cmpval ^ " " ^ eq ^ " " ^ target
    (*
     * condition for non-negative shift
     *)
    in let nonneg val0 val1 = comp_cond ">=" "0" val1
    (*
     * wrapper for both_cond with shift operations, which reverses shift for
     * negative shifts
     *)
    in let shift_cond op0 op1 xvt0 xvt1 = both_cond op0 op1 xvt0 xvt1
      nonneg
    (*
     * get code generator depending on the operation
     *)
    in let gen_bin_code = match op
      with Mult -> default_code "*" xvt0 xvt1
      | Div -> default_code "/" xvt0 xvt1
      | Mod -> default_code "%" xvt0 xvt1
      | Add -> default_code "+" xvt0 xvt1
      | Sub -> default_code "-" xvt0 xvt1
      | LSh -> shift_cond "<<" ">>" xvt0 xvt1
      | RSh -> shift_cond ">>" "<<" xvt0 xvt1
      | LeT -> default_code "<" xvt0 xvt1
      | GrT -> default_code ">" xvt0 xvt1
      | LE -> default_code "<=" xvt0 xvt1
      | GE -> default_code ">=" xvt0 xvt1
      | Eq -> default_code "==" xvt0 xvt1
      | NEq -> default_code "!=" xvt0 xvt1
      | BAnd -> default_code "&" xvt0 xvt1
      | BXor -> default_code "^" xvt0 xvt1
      | BOr -> default_code "|" xvt0 xvt1
      | And -> cond_exec1 "&&" xvt0 xvt1 (comp_cond "!=" "0")
               get0
      | Or -> cond_exec1 "||" xvt0 xvt1 (comp_cond "==" "0")
               get0
    in let bin_code = bin_gen ^ (gen_bin_code gen0 gen1 t0 t1 is_temp0
                                 is_temp1)
    in bin_temp, true, default_copy bin_temp, bin_code, xvt, temp_cnt,
      struct_tbl
  (* call function, using the generic gen_func_call *)
  | XFunkCallExpr(xvt, xfc) ->
    let ret_var, call_code, temp_cnt, struct_tbl = gen_func_call xfc estruct
                                                   fname temp_cnt struct_tbl
    in let ret_temp, ret_gen = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let store_ret = gen_copy_var_wrap (point ret_temp) (point ret_var)
      xvt estruct
    in let get_call_code = ret_gen ^ call_code ^ store_ret
    in ret_temp, true, default_copy ret_var, get_call_code, xvt, temp_cnt,
      struct_tbl
  (*
   * evaluating an async expressions is like declaring a function, and then
   * running it as a thread
   *)
  | XFunkAsyncExpr(xvt, ab) ->
    let out_temp, out_gen = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let async_fname, temp_cnt = get_anon_name fname temp_cnt
    in let async_var, init_async_inst = alloc_inst async_fname
    in let temp_cnt = temp_cnt + 1
    in let st = fst xvt
    in let initer, struct_tbl = gen_func_body async_fname ab [] struct_tbl
    in let init_func, temp_cnt, struct_tbl = initer async_var default_env
                                             temp_cnt struct_tbl
    in let call_func = c_func_call "run_async" [(point out_temp) ; async_var] ^
      ";\n"
    in let async_code = out_gen ^ init_async_inst ^ init_func ^ call_func
    in out_temp, true, default_copy out_temp, async_code, xvt, temp_cnt,
      struct_tbl
  | XFunkArrExpr(xvt, arr_expr, i_expr) ->
    let arr_temp, arr_gen = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let arr_temp, arr_is_temp, _, gen_arr, arr_xvt, temp_cnt,
      struct_tbl = gen_expr arr_expr estruct fname temp_cnt struct_tbl
    in let i_temp, i_is_temp, _, gen_i, i_xvt, temp_cnt,
      struct_tbl = gen_expr i_expr estruct fname temp_cnt struct_tbl
    in let i_val = get_single_raw i_temp XFunkInt i_is_temp
    in let arr_gen = gen_arr ^ gen_i
    in let el_temp, el_gen = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let arr_get = (c_func_call "get_element" [(point arr_temp) ;
                                                i_val])
    in let el_copy = gen_copy_var_wrap (point el_temp) arr_get xvt estruct
    in let el_copy = el_gen ^ el_copy
    in el_temp, true, default_copy arr_temp, arr_gen ^ el_copy, xvt, temp_cnt,
      struct_tbl
(*
 * Generates code to copy between variables
 * (string) dst_ptr: C pointer variable of output
 * (string) src_ptr: C pointer variable of input
 * (int) depth: depth of array. 0 for single type
 * (xsingle_vartype) base_type: the first member of the variable's type tuple
 * (string) estruct: env struct
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 *                            structures returns
 * returns
 *   code to copy variable
 *   updated temp_cnt
 *   updated struct_tbl
 *)
and gen_copy_var dst_ptr src_ptr depth base_type estruct_var =
  if depth = 0 then
    let try_copy = match base_type
      with XFunkFunc(_) -> copy_func dst_ptr src_ptr
      | _ -> (c_func_call "shallow_copy" [dst_ptr ; src_ptr]) ^ ";\n"
    in try_copy
  else
    let src_memb_access = point (get_arr_memb src_ptr "arr_i")
    in let dst_memb_access = point (get_arr_memb dst_ptr "arr_i")
    in let src_memb = "member_src_" ^ string_of_int(depth)
    in let dst_memb = "member_dst_" ^ string_of_int(depth)
    in let alloc_code = (c_func_call "init_array" [dst_ptr ; src_ptr ^
                                                   "->arr_size"]) ^ ";\n"
    in let gen_members =
      let gen_member = gen_copy_var dst_memb
        src_memb (depth - 1) base_type estruct_var
      in "{\n" ^
      "int arr_i;\n" ^
      "for (arr_i = 0; arr_i < " ^ dst_ptr ^ "->arr_size; arr_i++) {\n" ^
      "struct var *" ^ dst_memb ^ " = " ^ dst_memb_access ^ ";\n" ^
      "struct var *" ^ src_memb ^ " = " ^ src_memb_access ^ ";\n" ^
      gen_member ^
      "\n}\n" ^
      "\n}\n"
    in alloc_code ^ gen_members
(*
 * wrapper for gen_copy_var, that takes a full type, from which it finds the
 * array depth and the single base type
 * (string) dst_ptr: C variable of output
 * (string) src_ptr: C variable of input
 * (xvartype) full_type: contains base type, and depth is length of its second
 *                       part, the list of sizes
 * (string) estruct: env struct
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 *                            structures returns
 * returns
 *   code to copy variable
 *   updated temp_cnt
 *   updated struct_tbl
 *)
and gen_copy_var_wrap dst_ptr src_ptr full_type estruct_name =
  gen_copy_var dst_ptr src_ptr (List.length (snd full_type))
    (fst full_type) estruct_name
(*
 * generates C code that initializes a new function instance
 * (string) inst_var: pointer to instance
 * (xblock) fbody: contains free variables that need copies
 * (string) dst_estruct: type of instance's env
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 *                            structures returns
 * returns
 *   code to initialize function
 *   updated temp_cnt
 *   updated struct_tbl
 *)
and gen_new_func inst_var fbody fname dst_estruct dst_env temp_cnt struct_tbl =
  let setup_env, temp_cnt, struct_tbl = if ((List.length fbody.need_copy) > 0)
    then
      let alloc_env = c_func_call "SETUP_FUNC" [inst_var ; fname] ^ ";\n"
      in let env_var = "t_" ^ string_of_int(temp_cnt) ^ "_env"
      in let temp_cnt = temp_cnt + 1
      in let src_estruct = get_estruct_type fname
      in let create_env_var = dst_estruct ^ " *" ^ env_var ^ " = (" ^
        dst_estruct ^ "*) " ^ inst_var ^ "->scope;\n"
      in let copy_code, temp_cnt, struct_tbl =
        List.fold_left (fun (old_code, old_cnt, old_tbl)
        (dst, src) ->
        let dst_var = get_name dst dst_estruct env_var
        in let src_var = get_name src src_estruct dst_env
        in let sdt = dst.obj_type
        in let new_code =
          gen_copy_var (point dst_var) (point src_var) (List.length (snd sdt))
          (fst sdt) dst_estruct
        in old_code ^ new_code, temp_cnt, struct_tbl)
        ("", temp_cnt, struct_tbl) fbody.need_copy
      in alloc_env ^ create_env_var ^ copy_code, temp_cnt, struct_tbl
    else "", temp_cnt, struct_tbl
  in let func_init_name = get_inst_init fname
  in (c_func_call func_init_name [inst_var]) ^ ";\n" ^ setup_env,
    temp_cnt, struct_tbl

(*
 * generate code for xrvalue
 * (xrvalue) rv: the xrvalue for which to generate code
 * (string) estruct: environment structure
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 * returns:
 *   C code to calculate xrvalue
 *   the C-code representation of the variable handler of the xrvalue
 *   the type of the rvalue
 *   the updated temp_cnt
 *   the updated struct_tbl
 *)
and gen_rvalue rv estruct fname temp_cnt struct_tbl =
  let rcode, temp, rtype, temp_cnt, struct_tbl = match rv
    with XExprRVal(vt, ex) ->
      let handle, have_temp, copier, excode, _, temp_cnt, struct_tbl =
        gen_expr ex estruct fname temp_cnt struct_tbl
      in if have_temp then excode, handle, vt, temp_cnt, struct_tbl
      else
        let temp, temp_code = gen_temp temp_cnt
        in let temp_cnt = temp_cnt + 1
        in let cpcode = match vt
          with (st, []) -> copier temp
          (* sanity check *)
          | (_, _) -> raise (Failure "Arrays must have a temporary variable")
        in excode ^ temp_code ^ cpcode, temp, vt, temp_cnt, struct_tbl
    | XFuncRVal(anf) ->
      let header, body = anf
      in let anon_name, temp_cnt = get_anon_name fname temp_cnt
      in let temp, temp_code = gen_temp temp_cnt
      in let temp_cnt = temp_cnt + 1
      in let inst_var, init_inst = set_alloc_inst temp
      in let gen_vars = temp_code ^ init_inst
      in let initer, struct_tbl =
        gen_func anon_name header body struct_tbl
      in let init_code, temp_cnt, struct_tbl = initer inst_var default_env
        temp_cnt struct_tbl
      in gen_vars ^ init_code, temp, (XFunkFunc(header), []), temp_cnt,
        struct_tbl
  in rcode, temp, rtype, temp_cnt, struct_tbl
(*
 * generate code for a list of xrvalues
 * (xrvalue list) rvalues: the xrvalues for which to generate code
 * (string) estruct: environment structure
 * (string) fname: function name
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 * returns:
 *   C code to calculate xrvalue
 *   list of C-code representations of the variable handlers of the xrvalue
 *   the type of the rvalue
 *   the updated temp_cnt
 *   the updated struct_tbl
 *)
and gen_rvalues rvalues estruct fname temp_cnt struct_tbl =
  let rvalue_code, handles, types, temp_cnt, struct_tbl = List.fold_left
    (fun (old_code, old_handles, old_types, old_cnt, old_tbl) rv ->
      let next_piece, handle, rtype, temp_cnt, struct_tbl =
        gen_rvalue rv estruct fname old_cnt old_tbl
      in old_code ^ next_piece, handle::old_handles, rtype::old_types,
        temp_cnt, struct_tbl
    )
    ("", [], [], temp_cnt, struct_tbl) rvalues
  in rvalue_code ^ "", List.rev handles, List.rev types,
    temp_cnt, struct_tbl
(*
 * generate code for function calls
 * (xfunc_call) fc: the function call SAST node
 * (string) estruct: environment structure
 * (string) fname: function name
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 * returns:
 *   a variable handle for the return value
 *   code to call or execute the function
 *   the updated temp_cnt
 *   the updated struct_tbl
 *)
and gen_func_call fc estruct fname temp_cnt struct_tbl =
  let hdr, callee, params = fc
  in let param_code, handles, types, temp_cnt, struct_tbl =
    gen_rvalues params estruct fname temp_cnt struct_tbl
  in let conv_func dst_st src_st conv = fun src temp_cnt ->
    let dst_temp, dst_gen = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let get_src_func = get_func_type src_st
    in let set_dst_func = set_func_type dst_st
    in let get_src = c_func_call get_src_func [point src]
    in let set_dst = c_func_call set_dst_func [point dst_temp ; conv get_src] ^
      ";\n"
    in let gen_conv = dst_gen ^ set_dst
    in dst_temp, gen_conv, temp_cnt
  in let ret_var, exec_code, temp_cnt, struct_tbl = match callee
    (* built-in functions, and generic function *)
    with PrintCall ->
      let print_single stype = fun sname ->
        let print_type = match stype
          with XFunkInt -> "INT"
          | XFunkDouble -> "DOUBLE"
          | XFunkChar -> "CHAR"
          | XFunkBool -> "BOOL"
          | _ -> raise (Failure "Only primitives can be printed directly")
        in (c_func_call ("PRINT_" ^ print_type) [sname]) ^ ";\n"
      in let print_code = List.fold_left2
        (fun old_code rhand rtype ->
          let new_code = trav_arrs (point rhand) (List.length (snd rtype))
            (fun _ -> "") (print_single (fst rtype))
          in old_code ^ new_code
        )
        "" handles types
      in "", print_code, temp_cnt, struct_tbl
    | Double2IntCall ->
      let conv_ret, conv_code, temp_cnt =
        conv_func XFunkInt XFunkDouble (fun x -> x) (List.hd handles) temp_cnt
      in conv_ret, conv_code, temp_cnt, struct_tbl
    | Int2DoubleCall ->
      let conv_ret, conv_code, temp_cnt =
        conv_func XFunkDouble XFunkInt (fun x -> x) (List.hd handles) temp_cnt
      in conv_ret, conv_code, temp_cnt, struct_tbl
    | Boolean2IntCall ->
      let conv_ret, conv_code, temp_cnt =
        conv_func XFunkInt XFunkBool (fun x -> x) (List.hd handles) temp_cnt
      in conv_ret, conv_code, temp_cnt, struct_tbl
    | Int2BooleanCall ->
      let conv_ret, conv_code, temp_cnt =
        conv_func XFunkInt XFunkBool (fun x -> x ^ " ? 1 : 0")
          (List.hd handles) temp_cnt
      in conv_ret, conv_code, temp_cnt, struct_tbl
    (* generic case, where we use the struct var containing ptr *)
    | GenericCall(xpr) ->
      let func_var, _, _, func_gen, func_type, temp_cnt, struct_tbl =
        gen_expr xpr estruct fname temp_cnt struct_tbl
      (* passing in parameters *)
      in let pstruct, struct_tbl = get_param_struct hdr.xparams struct_tbl
      in let pstruct_name = "params_" ^ string_of_int(temp_cnt)
      in let temp_cnt = temp_cnt + 1
      in let func_var_ptr = point func_var
      in let func_ptr = func_var_ptr ^ "->val.ptr"
      in let rec check_global = function
        | XVariable(iobj) -> is_global iobj
        | XFunkArrExpr(_, arr_xpr, _) -> check_global arr_xpr
        | _ -> false
      (* have to declare own function instance if global *)
      in let func_ptr, dec_structs, read_pstruct, run_func, temp_cnt =
        if (check_global xpr)
        then let fsname = "fi_" ^ string_of_int(temp_cnt)
          in let set_pstructs = if (String.length pstruct = 0) then "" else
            (c_func_call (func_ptr ^ "->init") [(point fsname)]) ^ ";\n"
          in let dec_structs = "struct function_instance " ^ fsname ^ ";\n" ^
            set_pstructs
          in let temp_cnt = temp_cnt + 1
          in let read_pstruct = (cast_ptr pstruct (fsname ^ ".params"))
          in let run_func = (c_func_call (func_ptr ^ "->function")
            [(point fsname)]) ^ ";\n"
          in (point fsname), dec_structs, read_pstruct, run_func, temp_cnt
        else let run_func = (c_func_call "run_funcvar" [func_var_ptr]) ^ ";\n"
         in func_ptr, "",
           (cast_ptr pstruct (cast_ptr func_inst_type (func_ptr)) ^
           "->params"), run_func, temp_cnt
      in let get_pstruct =
        if ((String.length pstruct) = 0) then "" else
        pstruct ^ " *" ^ pstruct_name ^ " = " ^
        read_pstruct ^ ";\n"
      (* set parameters *)
      in let _, set_params =
        let set_param (count, old_code) src src_type =
          let new_code = gen_copy_var
            (point (pstruct_name ^ "->p_" ^ string_of_int(count))) (point src)
            (List.length (snd src_type)) (fst src_type) default_env
          in count + 1, old_code ^ new_code
        in List.fold_left2 set_param (0, "") handles
          types
      (* value is in ret_field *)
      in let ret_field = func_ptr ^ "->ret_val"
      in ret_field, func_gen ^ dec_structs ^ get_pstruct ^ set_params ^
        run_func, temp_cnt, struct_tbl
  in ret_var, param_code ^ ";\n" ^ exec_code, temp_cnt, struct_tbl
(*
 * generate code for statement
 * (string) estruct: environment structure
 * (string) fname: function name
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 * (xstatement): the statement from which to generate code
 * returns:
 *   code to execute the statement
 *   the updated temp_cnt
 *   the updated struct_tbl
 *)
and gen_stmt estruct fname temp_cnt struct_tbl =
  (*
   * generic loop code generation --implemented as C while loop
   * (xexpr option) cond_expr: expression for calculation condition. default to
   *                           1
   * (xstmt option) start: statement before the first iteration
   * (xstmt option) progress: statement at end of iteration
   * (xbody) body: loop body
   *)
  let gen_loop cond_expr start progress body =
    (*
     * declaration of condition variable
     * calculation of condition value
     * C expression that returns condition value
     * updated temp_cnt and struct_tbl
     *)
    let gen_check, calc_cond, get_cond, temp_cnt, struct_tbl = match cond_expr
      (* None expression defaults to 1, and no calculation needed *)
      with None -> "", "", "1", temp_cnt, struct_tbl
      (*
       * Some expression requires calculation before the while check, which
       * is done before entering the while loop, and at the end of each
       * iteration
       *)
      | Some(cond_expr) ->
        (*
         * declare temporary variable that is used for "while" check. We need
         * a single variable that can be accessed before and inside the loop
         * body, and in the while check
         *)
        let check_temp, gen_check = gen_temp temp_cnt
        in let temp_cnt = temp_cnt + 1
        (*
         * generate code for calculating condition result
         * this code is used before and at the end of the loop body, so
         * redeclaration is alright
         *)
        in let cond_temp, cond_is_temp, copy_cond, cond_gen, _, temp_cnt,
          struct_tbl = gen_expr cond_expr estruct fname temp_cnt struct_tbl
        (*
         * full condition value calculation also includes moving the value
         * to the common condition check variable
         *)
        in let calc_cond = cond_gen ^ (copy_cond check_temp)
        (*
         * the condition check variable is a variable of type bool, so we need
         * to use a C function to access it
         *)
        in let get_cond = (get_single_raw check_temp XFunkBool true)
        in gen_check, calc_cond, get_cond, temp_cnt, struct_tbl
    (* evaluate start and progression statements if they exist *)
    in let run_start, temp_cnt, struct_tbl = match start
      with None -> "", temp_cnt, struct_tbl
      | Some(xstmt) -> gen_stmt estruct fname temp_cnt struct_tbl xstmt
    in let run_progress, temp_cnt, struct_tbl = match progress
      with None -> "", temp_cnt, struct_tbl
      | Some(xstmt) -> gen_stmt estruct fname temp_cnt struct_tbl xstmt
    (* the loop body block code *)
    in let body_code, temp_cnt, struct_tbl = gen_block body estruct fname
      temp_cnt struct_tbl
    (* code before "while", outside the loop body *)
    in let pre_code = gen_check ^ run_start ^ calc_cond
    (* the C loop header *)
    in let while_header = "while (" ^ get_cond ^ ")"
    (*
     * each body includes loop body, as well as progress and
     * calculation of next condition
     *)
    in let full_body_code = body_code ^ run_progress ^ calc_cond
    in pre_code ^ while_header ^ (enclose full_body_code),
      temp_cnt, struct_tbl
  and gen_if_block cond_expr body =
    let check_temp, gc = gen_temp temp_cnt
    in let temp_cnt = temp_cnt + 1
    in let cond_temp, cond_is_temp, copy_cond, cond_gen, _, temp_cnt,
    	   struct_tbl = gen_expr cond_expr estruct fname temp_cnt struct_tbl
    in let calc_cond = cond_gen ^ (copy_cond check_temp)
    in let get_cond = get_single_raw check_temp XFunkBool true
    in let pre_code = gc ^ calc_cond
    in let if_header = "if (" ^ get_cond ^ ")"
    in let new_code, temp_cnt, struct_tbl = gen_block body estruct fname temp_cnt
    					    struct_tbl
    in pre_code ^ if_header ^ (enclose new_code), temp_cnt, struct_tbl
  in function
  | XAssignment(xa) ->
    let assign_code, temp_cnt, struct_tbl = gen_assign xa estruct fname temp_cnt
                                            struct_tbl
    in assign_code, temp_cnt, struct_tbl
  | XFunctionCall(fc) ->
    let _, call_code, temp_cnt, struct_tbl = gen_func_call fc estruct fname
                                             temp_cnt struct_tbl
    in call_code, temp_cnt, struct_tbl
  | XBlock(blk) ->
    let new_code, temp_cnt, struct_tbl = gen_block blk estruct fname temp_cnt
                                         struct_tbl
    in enclose new_code, temp_cnt, struct_tbl
  | XForBlock(start, cond_expr, progress, body) ->
    gen_loop cond_expr start progress body
  | XIfBlock(cond_expr, body) -> gen_if_block cond_expr body
  | XIfElseBlock(cond_expr, ifbody, elsebody) ->
    let codei, temp_cnt, struct_tbl = gen_if_block cond_expr ifbody
    in let codee, temp_cnt, struct_tbl = gen_block elsebody estruct fname
    					 temp_cnt struct_tbl
    in codei ^ "else" ^ (enclose codee), temp_cnt, struct_tbl
  | XWhileBlock(cond_expr, body) -> gen_loop (Some cond_expr) None None body
  | XBreak -> "break;", temp_cnt, struct_tbl
  | XReturn(rvo) ->
    (* if there is return value, copy it to ret_val of the function instance *)
    let put_out, temp_cnt, struct_tbl = match rvo
      with Some(rv) ->
        let out_var = "(&self->ret_val)"
        in let rcode, temp, rtype, temp_cnt, struct_tbl =
          gen_rvalue rv estruct fname temp_cnt struct_tbl
        in let copy_code = (gen_copy_var_wrap
                            out_var (point temp) rtype estruct)^ ";\n"
        in rcode ^ copy_code, temp_cnt, struct_tbl
      | None -> "", temp_cnt, struct_tbl
    in put_out ^ "goto RETURN;\n", temp_cnt, struct_tbl
(*
 * generate code for block
 * (xblock): a block that contains statements to turn into code, and variables
 *           to declare
 * (string) estruct: environment structure
 * (string) fname: function name 
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of param
 *                            structs
 * returns:
 *   code to execute the block
 *   the updated temp_cnt
 *   the updated struct_tbl
 *)
and gen_block xb estruct fname temp_cnt struct_tbl = List.fold_left
  (fun (old_code, old_cnt, old_tbl) stmt ->
    let new_code, new_cnt, new_tbl = gen_stmt estruct fname old_cnt old_tbl stmt
    in (old_code ^ new_code, new_cnt, new_tbl)
  )
  ("", temp_cnt, struct_tbl) xb.xstmts
(*
 * generates access to generates C expression to access array
 * (string) arr_ptr: variable of type struct var * that is an array
 * (xexpr) i_expr: integer expression
 * (string) estruct: environment structure
 * (string) fname: function name
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of param
 *                            structs
 * returns
 *   array element, of type struct var
 *   updated temp_cnt
 *   updated struct_tbl
 *)
and gen_arr_access arr_ptr i_expr estruct fname temp_cnt struct_tbl =
  let i_val, i_is_temp, _, i_code, i_xvt, temp_cnt, struct_tbl =
    gen_expr i_expr estruct fname temp_cnt struct_tbl
  in let e_var = c_func_call "get_element" [arr_ptr ;
    (get_single_raw i_val (fst i_xvt) i_is_temp) ]
  in e_var, i_code, temp_cnt, struct_tbl
(*
 * generate code for assignment or assigning declaration
 * (xassignment): an assignment, that contains rvalues to evaluate, and
 *                lvalues to fill
 * (string) estruct: environment structure
 * (string) fname: function name
 * (int) temp_cnt: running count of temporary variables
 * (struct_table) struct_tbl: current struct declarations and map of
 * returns:
 *   code to execute the assignment
 *   the updated temp_cnt
 *   the updated struct_tbl
 *)
and gen_assign assign estruct fname temp_cnt struct_tbl =
  let rvalue_code, handles, types, temp_cnt, struct_tbl =
    gen_rvalues assign.rvals estruct fname temp_cnt struct_tbl
  in let rec get_lvalue (temp_cnt, struct_tbl) = function
    | XVariable(iobj) ->
      if ((is_global iobj) && fname != global_scope)
      then raise (Failure
                  ("You are attempting to change a global constant:\n" ^
                   iobj.name))
      else (point (get_name iobj estruct default_env)), "", temp_cnt, struct_tbl
    | XFunkArrExpr(xvt, a_expr, i_expr) ->
      let arr_var, arr_code, temp_cnt, struct_tbl = get_lvalue
                                                    (temp_cnt, struct_tbl)
                                                    a_expr
      in let e_var, arr_access, temp_cnt, struct_tbl = gen_arr_access arr_var
                                                       i_expr estruct fname
                                                       temp_cnt struct_tbl
      in e_var, arr_code ^ arr_access, temp_cnt, struct_tbl
    | _ -> raise (Failure "Invalid lvalue")
  in let lvars, lcode, temp_cnt, struct_tbl = List.fold_left
    (fun (old_vars, old_code, old_cnt, old_tbl) lexpr ->
     let new_var, new_code, temp_cnt, param_tbl = get_lvalue
                                                  (old_cnt, old_tbl) lexpr
     in new_var::old_vars, old_code ^ new_code, temp_cnt, struct_tbl)
     ([], "", temp_cnt, struct_tbl) assign.lvals
  in let lvars = List.rev lvars
  in let gen_pair_code lvar (rval, rvar) temp_cnt struct_tbl =
    match rval
      with XExprRVal(xvt, xpr) -> (match xpr
        with XFunkAsyncExpr(_) -> ((c_func_call "run_async_assign"
                                  [lvar ; rvar])) ^ ";\n",
                                  temp_cnt, struct_tbl
        | _ -> let direct_copy =
            gen_copy_var lvar rvar (List.length (snd xvt)) (fst xvt) default_env
          in direct_copy, temp_cnt, struct_tbl
      )
      | XFuncRVal(hdr, _) ->
        let ft = XFunkFunc(hdr)
        in let direct_copy =
          gen_copy_var lvar rvar 0 ft default_env
        in direct_copy, temp_cnt, struct_tbl
  in let rvals_vars =
    let rev = List.fold_left2 (fun old_list rval rvar ->
      (rval, point rvar)::old_list
    ) [] assign.rvals handles
    in List.rev rev
  in let pair_copies, temp_cnt, struct_tbl = List.fold_left2
    (fun (old_copies, old_cnt, old_tbl) lvar rva ->
     let new_copy, temp_cnt, struct_tbl = gen_pair_code lvar rva old_cnt
       struct_tbl
     in old_copies ^ new_copy, temp_cnt, struct_tbl)
    ("", temp_cnt, struct_tbl) lvars rvals_vars
  in let assign_code = rvalue_code ^ lcode ^ pair_copies
  in assign_code, temp_cnt, struct_tbl
(*
 * Generates C code for a function
 * (string) fname: the name for the function. If the function is anonymous,
 *        the name was declared automatically
 * (xfunc_header) fheader: the parameter and return header of the function
 * (xbody) fbody: the body of the function
 * (struct_table) struct_tbl: current struct declarations and map of
 *                            structures returns
 * returns:
 *   code that declares the env struct, if needed
 *   code that declares param struct, if needed
 *   C version of function
 *   (code for init function, its name)
 *   code for copy function
 *   updated struct_tbl
 *)
and gen_func fname fheader fbody struct_tbl =
  let func_initer, struct_tbl = gen_func_body fname fbody
    fheader.xparams struct_tbl
  in let pstruct_name, struct_tbl =
    get_param_struct fheader.xparams struct_tbl
  in func_initer, struct_tbl
(*
 * Wrapper function for generating code for global function
 * (xfuncdec) g_func: global function declaration, which contains all
 *                    the information that is needed for the function itself.
 *                    the main difference is that the C code function will
 *                    append "_global" to the name
 * (struct_table) struct_tbl: current struct declarations and map of
 *                            structures returns
 * returns:
 *   the static variable for the global function's instance
 *   the declaration of that static variable
 *   the function for initializing the function instance
 *   updated struct_tbl
 *)
and gen_global_func g_func struct_tbl =
  let global_name = g_func.xfid ^ "_global"
  in let global_inst = global_name ^ "_instance"
  in let global_inst_dec = "static " ^ func_inst_type ^ " " ^ global_inst ^
    ";\n"
  in let initer, struct_tbl =
    gen_func global_name g_func.xfunc_header g_func.xbody struct_tbl
  in global_inst, global_inst_dec, initer, struct_tbl
(*
 * defines .init member of function
 * (string) fname: internal name of function
 * (xblock) fbody: function body
 * (string) estruct_name: name of free-variable struct type
 * returns C function for initializing a function instance
 *)
and gen_init fname fbody estruct_name pstruct_name =
  let ni = "new_inst"
  in let ne = ni ^ "->scope"
  in let ni_memb = ni ^ "->"
  in let f_func, f_init, f_copy = get_inst_function fname, get_inst_init fname,
                                  get_inst_copy fname;
  in let inst_funcs = ni_memb ^ "function = " ^ f_func ^ ";\n" ^
                      ni_memb ^ "init = " ^ f_init ^ ";\n" ^
                      ni_memb ^ "copy = " ^ f_copy ^ ";\n"
  in let dec_dst = if ((String.length estruct_name) > 0)
    then estruct_name ^ " *dst_env = malloc(sizeof(" ^
      estruct_name ^ "));\n" ^
      ne ^ " = dst_env;\n"
    else ""
  in let dec_param = if ((String.length pstruct_name) > 0)
    then pstruct_name ^ " *params = malloc(sizeof(" ^
      pstruct_name ^ "));\n" ^
      ni ^ "->params = params;\n"
    else ""
  in let rec init_vars count old_code =
    let init_this = c_func_call "init_var"
      [(point (get_free_name ne count estruct_name))] ^ ";\n"
    in if count < (List.length fbody.need_copy)
      then init_vars (count + 1) (old_code ^ init_this)
      else old_code
  in let init_vars_code = init_vars 0 ""
  in let init_ret = (c_func_call "init_var" [point (ni ^ "->ret_val")]) ^ ";\n"
  in let fbody_code = inst_funcs ^ dec_dst ^ dec_param ^ init_vars_code ^
    init_ret
  in (wrap_c_func f_init fbody_code "void " [func_inst_type ^ " *" ^ ni],
    f_init)
(*
 * defines .copy member of function
 * (string) fname: internal name of function
 * (xblock) fbody: function body
 * (string) estruct_name: name of free-variable struct type
 * returns C function for copying a function instance
 *)
and gen_copy fname fbody estruct_name =
  let fbody_code = if (List.length fbody.need_copy > 0) then
    let dec_src = estruct_name ^ " *src_env = (" ^ estruct_name ^
      " *) src->scope;\n"
    in let dec_dst = estruct_name ^ " *dst_env = (" ^ estruct_name ^
      " *) dst->scope;\n"
    in let copy_vars, _ = List.fold_left
      (fun (old_code, i) (free_src, _) ->
        let copy_code = (gen_copy_var_wrap (point (get_free_name "dst_env" i
                                            estruct_name))
                         (point (get_free_name "src_env" i estruct_name))
                         free_src.obj_type estruct_name)
        in old_code ^ copy_code, i + 1
      ) ("", 0) fbody.need_copy
    in dec_src ^ dec_dst ^ copy_vars
    else ""
  in wrap_c_func (get_inst_copy fname) fbody_code "void "
    [func_inst_type ^ " *" ^ "dst" ; func_inst_type ^ " *" ^ "src"]


(*
 * main function for generating C code
 * (xprogram) prog: root of the SAST
 * returns the complete C code
 *)
let gen_prog prog globals =
  let build_dec temp_cnt struct_tbl = function
    | XVardec(xa) ->
      let new_code, temp_cnt, struct_tbl = gen_assign xa "" global_scope
        temp_cnt struct_tbl
      in new_code, "", temp_cnt, { struct_decs = struct_tbl.struct_decs ;
                         prog_decs = struct_tbl.prog_decs ;
                         params = struct_tbl.params }
    | XFuncdec(xf) ->
      let global_inst, global_inst_dec, initer, struct_tbl =
        gen_global_func xf struct_tbl
      in let try_start = if xf.xfid = "main" then
        c_func_call (global_inst ^ ".function") ["&" ^ global_inst] ^ ";\n"
        else ""
      in let init_code, temp_cnt, struct_tbl = initer (point global_inst)
        (global_inst ^ "->env") temp_cnt struct_tbl
      in let copy_code = (get_global_name xf.global_id.var_id) ^
        ".val.ptr = " ^ (point global_inst) ^ ";\n"
      in init_code ^ copy_code, try_start, temp_cnt,
        { struct_decs = struct_tbl.struct_decs ^ global_inst_dec ;
          prog_decs = struct_tbl.prog_decs ; params = struct_tbl.params }
  in let main_init, main_start, _, struct_tbl =
    List.fold_left
    (fun (old_main, old_start, old_cnt, old_tbl) dec ->
      let new_main, new_start, temp_cnt, struct_tbl =
        build_dec old_cnt old_tbl dec
      in old_main ^ new_main,
      old_start ^ new_start, temp_cnt, struct_tbl)
    ("", "", 0, { struct_decs = "" ; prog_decs = "" ;
                          params = ParamTable.empty}) prog
  in let global_decs, global_inits, _ = List.fold_left
    (fun (old_dec, old_code, id) dec -> 
      let bv_name = get_global_name id
      in let bv_dec = "struct var " ^ bv_name ^ ";\n"
      in let bv_init = (c_func_call "init_var" [point bv_name]) ^ ";\n"
      in (old_dec ^ bv_dec, old_code ^ bv_init, id + 1)
    ) ("", "", 0) globals
  in "#include <complete.h>\n" ^
    struct_tbl.struct_decs ^
    global_decs ^
    struct_tbl.prog_decs ^
    "int main(void)\n" ^
    "{\n" ^
    global_inits ^ main_init ^ main_start ^
    "return 0;\n" ^
    "}"
