/*
 * Runtime handling of function instances
 */
#ifndef FUNC_H
#define FUNC_H

#include <funk_error.h>
#include <env.h>

/* function, with pointers to parameters, in-scope functions, return values */
struct function_instance {
	void *params; /* explicit parameters given in Funk language */
	void *scope; /* variables manipulated inside scope of function */
	struct var ret_val; /* placeholder for return values */
	/*
	 * backend implementation of function in Funk. Note that it takes in
	 * all the previous members of this struct
	 * self: void form of this function_instance
	 */
	void *(*function)(void *self);
	/* automatically generated function for initializing environments */
	void (*init)(struct function_instance *new_inst);
	/* automatically generated function for copying function instances */
	void (*copy)(struct function_instance *dst,
		     struct function_instance *src);
};

struct async_data {
	struct var *out;
	struct function_instance *async_block;
};

void *exec_async_block(void *data);

static inline void run_async(struct var *out,
			     struct function_instance *async_block)
{
	pthread_t t;
	struct async_data *ad = malloc(sizeof(struct async_data));

	ad->out = out;
	ad->async_block = async_block;

	LOCK_VAR(out);
	if (pthread_create(&t, NULL, exec_async_block, ad)) {
		perror(ASYNC_ERROR);
		exit(-1);
	}
}

static inline void check_funcvar(struct var *func_var)
{
	if (func_var->val.ptr == NULL) {
		fprintf(stderr, ERROR "Using uninitialized function value\n");
		exit(-1);
	}
}

static inline void run_funcvar(struct var *func_var)
{
	check_funcvar(func_var);

	LOCK_VAR(func_var);
	func_var->val.ptr->function(func_var->val.ptr);
	UNLOCK_VAR(func_var);
}

static inline void copy_funcvar(struct var *dst, struct var *src)
{
	struct function_instance *inst;

	check_funcvar(src);
	inst = malloc(sizeof(struct function_instance));

	LOCK_VAR(src);
	src->val.ptr->init(inst);
	src->val.ptr->copy(inst, src->val.ptr);
	UNLOCK_VAR(src);

	LOCK_VAR(dst);
	dst->val.ptr = inst;
	UNLOCK_VAR(dst);

}

#define SETUP_FUNC_0(inst_ptr, base_name) \
	base_name##_init(inst_ptr);
#define SETUP_FUNC(inst_ptr, base_name) \
	(inst_ptr)->scope = malloc(sizeof(struct base_name##_env)); \
	SETUP_FUNC_0(inst_ptr, base_name)
#define CREATE_FUNC(inst_ptr, base_name) \
	inst_ptr = malloc(sizeof(struct function_instance)); \
	SETUP_FUNC(inst_ptr, base_name)
#define CREATE_FUNC_0(inst_ptr, base_name) \
	inst_ptr = malloc(sizeof(struct function_instance)); \
	SETUP_FUNC_0(inst_ptr, base_name)

#endif /* FUNC_H */
