/*
 * Runtime handling of variables in and between scopes
 */
#ifndef ENV_H
#define ENV_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>

#include <funk_error.h>

struct function_instance;

union value {
	long int int_v;
	int bool_v;
	double double_v;
	char char_v;
	struct var *array; /* member objects */
	struct function_instance *ptr; /* function pointer */
};

struct var {
	size_t arr_size; /* number of elements */
	/* primitive value, or pointer to array or function */
	union value val;
	pthread_mutex_t await_lock;
	pthread_cond_t await_cond;
};

/*
 * Holds assignment pair for pthread function to handle
 */
struct async_assignment {
	struct var *lval;
	struct var *rval;
};

void *assign_async(void *data);

#define LOCK_VAR(var) pthread_mutex_lock(&(var)->await_lock)
#define UNLOCK_VAR(var) pthread_mutex_unlock(&(var)->await_lock)

static inline void run_async_assign(struct var *lval, struct var *rval)
{
	pthread_t t;
	struct async_assignment *aa = malloc(sizeof(struct async_assignment));

	aa->lval = lval;
	aa->rval = rval;

	//Lock a global lock
	LOCK_VAR(lval);
	if (pthread_create(&t, NULL, assign_async, aa)) {
		perror(ASYNC_ERROR);
		exit(-1);
	}
	pthread_join(t, NULL);
	UNLOCK_VAR(lval);
}

static inline void copy_primitive(struct var *dst, struct var *src)
{
	union value temp;

	LOCK_VAR(src);
	memcpy(&temp, &src->val, sizeof(dst->val));
	UNLOCK_VAR(src);

	LOCK_VAR(dst);
	memcpy(&dst->val, &temp, sizeof(dst->val));
	UNLOCK_VAR(dst);
}

static inline void init_var(struct var *new_var)
{
	memset(&new_var->val, 0, sizeof(union value));
	pthread_mutex_init(&new_var->await_lock, NULL);
	pthread_cond_init(&new_var->await_cond, NULL);
}

static inline void init_array(struct var *new_array, int size)
{
	int i;

	if (size <= 0) {
		fprintf(stderr, ERROR "Array size must be positive");
		exit(-1);
	}
	new_array->arr_size = (size_t) size;
	new_array->val.array = malloc(sizeof(struct var) * size);
	for (i = 0; i < size; i++) {
		init_var(&(new_array->val.array[i]));
	}
}

static inline void check_bounds(struct var *array, int index)
{
	if (array->val.array == NULL) {
		fprintf(stderr, ERROR
			"Array has not been initialized\n");
		exit(-1);
	}
	if (index >= array->arr_size || index < 0) {
		fprintf(stderr, ERROR
			"Array out of bounds: Size is %d, but %d was requested",
			(int) array->arr_size, index);
		exit(-1);
	}
}

static inline struct var *get_element(struct var *array, int index)
{
	check_bounds(array, index);
	return &array->val.array[index];
}

static inline void shallow_copy(struct var *dst, struct var *src)
{
	union value temp;
	size_t arr_size;

	LOCK_VAR(src);
	memcpy(&temp, &src->val, sizeof(src->val));
	arr_size = src->arr_size;
	UNLOCK_VAR(src);

	LOCK_VAR(dst);
	memcpy(&dst->val, &temp, sizeof(dst->val));
	dst->arr_size = arr_size;
	UNLOCK_VAR(dst);
}

static inline void _shallow_copy(struct var *dst, struct var *src)
{
	union value temp;
	size_t arr_size;

	memcpy(&temp, &src->val, sizeof(src->val));
	arr_size = src->arr_size;

	memcpy(&dst->val, &temp, sizeof(dst->val));
	dst->arr_size = arr_size;
}

static inline void set_element(struct var *out, int index,
				      struct var *new_member)
{
	struct var *out_slot = get_element(out, index);

	shallow_copy(out_slot, new_member);
}

/* getting primitives */
static inline int get_int(struct var *int_var)
{
	int int_val;
	LOCK_VAR(int_var);
	int_val = (int_var)->val.int_v;
	UNLOCK_VAR(int_var);
	return int_val;
}

static inline int get_bool(struct var *bool_var)
{
	int bool_val;
	LOCK_VAR(bool_var);
	bool_val = (bool_var)->val.bool_v;
	UNLOCK_VAR(bool_var);
	return bool_val;
}

static inline double get_double(struct var *double_var)
{
	double double_val;
	LOCK_VAR(double_var);
	double_val = (double_var)->val.double_v;
	UNLOCK_VAR(double_var);
	return double_val;
}

static inline char get_char(struct var *char_var)
{
	char char_val;
	LOCK_VAR(char_var);
	char_val = (char_var)->val.char_v;
	UNLOCK_VAR(char_var);
	return char_val;
}

/* setting primitives */
static inline void set_int(struct var *int_var, int int_val)
{
	LOCK_VAR(int_var);
	(int_var)->val.int_v = int_val;
	UNLOCK_VAR(int_var);
}

static inline void set_bool(struct var *bool_var, int bool_val)
{
	LOCK_VAR(bool_var);
	(bool_var)->val.bool_v = bool_val;
	UNLOCK_VAR(bool_var);
}

static inline void set_double(struct var *double_var, double double_val)
{
	LOCK_VAR(double_var);
	(double_var)->val.double_v = double_val;
	UNLOCK_VAR(double_var);
}

static inline void set_char(struct var *char_var, char char_val)
{
	LOCK_VAR(char_var);
	(char_var)->val.char_v = char_val;
	UNLOCK_VAR(char_var);
}

#endif /* ENV_H */
