{
  open Parser
  let translate_escape = function
    | 'n' -> '\n'
    | 'r' -> '\r'
    | 't' -> '\t'
    | 'a' -> '\007'
    (*
     * the rest are printable characters that had special meaning, but 
     * the backslash removes it
     *)
    | '\"' -> '\"'
    | '\\' -> '\\'
    | x -> raise (Failure("illegal escape sequence: \\" ^ String.make 1 x))
  let rec translate_escapes s =
    if (String.contains s '\\') then
      let esc_i = String.index s '\\' in
      let before = String.sub s 0 esc_i in
      let after_start = esc_i + 2 in
      let after = String.sub s after_start ((String.length s) - after_start) in
      let replaced = translate_escape(s.[esc_i + 1]) in
      String.concat "" [before ; String.make 1 replaced ;
			translate_escapes(after)]
    else
      s
}

let fp1 = ['0'-'9']* '.' ['0'-'9']+
let fp2 = ['0'-'9']+ '.' ['0'-'9']*
let exp = 'e' ['+' '-']? ['0'-'9']+

(* code reference from ocaml lex *)
(* http://caml.inria.fr/svn/ocaml/trunk/lex/lexer.mll *)

let back_slash = '\\'
let back_slash_escapes = back_slash ['n' 'r' 't' 'a' '"' '\\']
let string_char = ([^ '"'] | back_slash_escapes)*

rule free_form = parse
| [' ' '\t'] { free_form lexbuf }
| ['\n' '\r']+ { NEWLINE }
| "//" { linecomment lexbuf }
| "/*" { blockcomment lexbuf }
| "||" { OR }
| "&&" { AND }
| '!' { NOT }
| '+' { PLUS }
| '*' { MULT }
| '-' { MINUS }
| '/' { DIV }
| '=' { ASSIGN }
| '<' { LT }
| '>' { GT }
| "<=" { LTE }
| ">=" { GTE }
| "==" { EQ }
| "!=" { NE }
| ';' { SEMI }
| ',' { COMMA }
| '(' { LPAREN }
| ')' { RPAREN }
| '{' { LBRACE }
| '}' { RBRACE }
| '[' { LBRACKET }
| ']' { RBRACKET }
| '~' { BNOT }
| '&' { BAND }
| '|' { BOR }
| '%' { MOD }
| '^' { BXOR }
| "<<" { LSHIFT }
| ">>" { RSHIFT }
| "func" { FUNC }
| "var" { VAR }
| "async" { ASYNC }
| "bool" { BOOL }
| "double" { DOUBLE }
| "char" { CHAR }
| "int" { INT }
| "if" { IF }
| "else" { ELSE }
| "for" { FOR }
| "break" { BREAK }
| "return" { RETURN }
| "true" | "false" as tf { BOOL_LIT (bool_of_string tf) }
| fp1 as fp { DOUBLE_LIT (float_of_string fp) }
| fp2 as fp { DOUBLE_LIT (float_of_string fp) }
| fp1 exp as fp { DOUBLE_LIT (float_of_string fp) }
| fp2 exp as fp { DOUBLE_LIT (float_of_string fp) }
| ['0'-'9']+ exp as fp { DOUBLE_LIT (float_of_string fp) }
| ['0'-'9']+ as lxm { INT_LIT (int_of_string lxm) }
| ['a'-'z' 'A'-'Z' '_'] ['0'-'9' 'a'-'z' 'A'-'Z' '_']* as id { ID (id) }
| eof { EOF }
| '"' (string_char as content) '"' { STRING (translate_escapes content) }
(* escaped character *)
| '\'' back_slash (_ as esc_char) '\''
    { CHAR_LIT (translate_escape(esc_char)) }
(* unescaped single character *)
| '\'' (_ as char_match) '\'' { CHAR_LIT (char_match) }
| _ as char { raise (Failure("illegal character:[" ^ Char.escaped char ^ "]")) }

and linecomment = parse
| ['\r' '\n'] { NEWLINE }
| _ {linecomment lexbuf}

and blockcomment = parse
| "*/" { free_form lexbuf }
| _ {blockcomment lexbuf}
