{ open Parser }

rule token = parse
  [' ' '\t' '\r' '\n']	{ token lexbuf } 			(* White space characters *)
  | "/*"     		{ comment lexbuf }           		(* Multiline Comments *)
  | "//"     		{ linecomment lexbuf }       		(* Inline Comments *)
  | '('      		{ LPAREN }				(* Left parenthesis *)
  | ')'      		{ RPAREN }				(* Right parenthesis *)
  | '{'      		{ LBRACE }				(* Left brace *)
  | '}'      		{ RBRACE }				(* Left brace *)
  | ';'      		{ SEMI }				(* Semicolon, to separate statements *)
  | ','      		{ COMMA }				(* Comma, to separate arguments within a function *)
  | '+'      		{ PLUS }				(* Addition mathematical operator *)
  | '-'      		{ MINUS }				(* Subtraction mathematical operator *)
  | '*'      		{ TIMES }				(* Multiplication mathematical operator *)
  | '/'      		{ DIVIDE }				(* Division mathematical operator *)
  | '^'      		{ POW }					(* Exponentiation mathematical operator *)
  | '='      		{ ASSIGN }				(* Assignment operator *)
  | '.'      		{ DEREFERENCE }				(* Dereference an attribute of an object *)
  | "=="     		{ EQ }					(* Equality relational operator *)
  | "!="     		{ NEQ }					(* Inequality relational operator *)
  | '<'      		{ LT }					(* Less than relational operator *)
  | "<="     		{ LEQ }					(* Less than or equal to relational operator *)
  | '>'      		{ GT }					(* Greater than relational operator *)
  | ">="     		{ GEQ }					(* Greater than or equal to relational operator *)
  | "!"	   		{ NOT }					(* Not logical operator *)
  | "&&"	   	{ AND }					(* And logical operator *)
  | "||"	   	{ OR }					(* Or logical operator *)
  | "++"     		{ CONCAT }				(* String concatenation operator *)
  | "if"     		{ IF }					(* If statement *)
  | "else"   		{ ELSE }				(* Part of if statement *)
  | "for"    		{ FOR }					(* For loop *)
  | "while"  		{ WHILE }				(* While Loop *)
  | "function"		{ FUNCTION }				(* To declare user-defined funtions *)
  | "vehicle" 		{ VEHICLE }				(* Vehicle object *)
  | "pedestrian"  	{ PEDESTRIAN }				(* Pedestrian object *)
  | "trafficsignal"  	{ TRAFFICSIGNAL }			(* Traffic Signal object *)
  | "stopsign" 	 	{ STOPSIGN }				(* Stop sign object *)
  | "object" 		{ OBJECT }				(* Simple object *)
  | "boolean" 		{ BOOLEAN }				(* Boolean data type *)
  | "int" 		{ INT }					(* Interger data type *)
  | "decimal" 		{ DECIMAL }				(* Float data type *)
  | "string"   		{ STRING }				(* String data type *)
  | "print"  		{ PRINT }				(* Built-in print function *)
  | "random" 		{ RANDOM }				(* Built-in random function *)
  | "sqrt"   		{ SQRT }				(* Built-in sqrt function *)
  | "streq"   		{ STREQ }				(* Built-in streq function *)

  | "true"		as lxm { LITERAL(lxm) }  		(* Boolean *)

  | "false"		as lxm { LITERAL(lxm) }  		(* Boolean *)

  | ['0'-'9']+ 		as lxm { LITERAL(lxm) }  		(* Integers *)
  
  | ['0'-'9']+ '.' ['0'-'9']* ('e' ('+'|'-')? ['0'-'9']+)?
	 		as lxm { LITERAL(lxm) }  		(* Floats *)

  |            '.' ['0'-'9']+ ('e' ('+'|'-')? ['0'-'9']+)?
	 		as lxm { LITERAL(lxm) }  		(* Floats *)

  | ['0'-'9']+                 'e' ('+'|'-')? ['0'-'9']+
	 		as lxm { LITERAL(lxm) }  		(* Floats *)
  
  | ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']* 
  			as lxm { ID(lxm) }  			(* Identifiers *)
  
  | '\"' [^ '\"']* '\"' 
  			as lxm { STR(lxm) }  			(* Strings *)
  
  | eof 		{ EOF }

  | _ as char 		{ raise (Failure("illegal character " ^ Char.escaped char)) }
  
and comment = parse
  "*/" 			{ token lexbuf }
  | _    		{ comment lexbuf }

and linecomment = parse
  ['\r' '\n'] 		{ token lexbuf }
  | _    		{ linecomment lexbuf }
