open Ast

module NameMap = Map.Make(struct
  type t = string
  let compare x y = Pervasives.compare x y
end)

exception ReturnException of string NameMap.t * string NameMap.t NameMap.t

(* Main entry point: run a program *)

let run (vars, funcs) =

  (* Put function declarations in a symbol table *)

  let func_decls = List.fold_left
    (fun funcs fdecl -> 
      if NameMap.mem fdecl.fname funcs then 
        raise (Failure ("function " ^ fdecl.fname ^ 
        	" is defined more than once!"));
        NameMap.add fdecl.fname fdecl funcs ) 
    NameMap.empty funcs

  in    

  (* Put variable declarations in a symbol table *)

  let var_decls = List.fold_left
    (fun globals vdecl -> 
      let attrs = List.fold_left  
        (fun attr_map attr_decl -> 
        NameMap.add attr_decl.key attr_decl.value attr_map)
        NameMap.empty vdecl.vattrs in
          if NameMap.mem vdecl.vname globals then 
            raise (Failure ("variable " ^ vdecl.vname ^ 
            		" is defined more than once!"));
            NameMap.add vdecl.vname attrs globals;)

  in

  (* Invoke a function and return an updated global symbol table *)

  let rec call fdecl actuals globals =

    (* Evaluate an object expression and return (value, updated environment) *)

    let rec objeval env = function

      Literal(i) -> i, env

      | Noexpr -> "1", env 				(* must be a string; 
      							   must be non-zero for the for loop predicate *)

      | Str(str) -> (String.sub str 1 (( String.length str)-2)), env

      | Neg(e) ->
	  let v, env = objeval env e in 
	    string_of_float(-.(float_of_string(v))), env

      | Binop(e1, op, e2) ->
          let v1, env = objeval env e1 in
          let v2, env = objeval env e2 in
          let boolean_to_string i = if i then "true" else "false" in
            (match op with

              Add -> string_of_float(float_of_string(v1) +. float_of_string(v2))

              | Sub -> string_of_float(float_of_string(v1) -. float_of_string(v2))

              | Mult -> string_of_float(float_of_string(v1) *. float_of_string(v2))

              | Div -> string_of_float(float_of_string(v1) /. float_of_string(v2))

              | Pow -> string_of_float(float_of_string(v1) ** float_of_string(v2))

              | Equal -> boolean_to_string(float_of_string(v1) = float_of_string(v2))

              | Neq -> boolean_to_string(float_of_string(v1) <> float_of_string(v2))

              | Less -> boolean_to_string(float_of_string(v1) < float_of_string(v2))

              | Leq -> boolean_to_string(float_of_string(v1) <= float_of_string(v2))

              | Greater -> boolean_to_string(float_of_string(v1) > float_of_string(v2))

              | Geq -> boolean_to_string(float_of_string(v1) >= float_of_string(v2))

              | Concat -> String.concat "" (v1::(v2::[]))

              | And -> if (0 = (String.compare v1 "true") 
              		&& 0 = (String.compare v2 "true")) then "true" else "false"

              | Or -> if (0 = (String.compare v1 "true") 
              		|| 0 = (String.compare v2 "true")) then "true" else "false"), env

      | Streq(e1, e2) ->
          let v1, env = objeval env e1 in
          let v2, env = objeval env e2 in
	  let int_to_string i = if (i = 0) then "true" else "false" in
	    int_to_string (String.compare v1 v2), env

      | Not(e) ->
	  let v, env = objeval env e in
	    let not_e i = if (0 = (String.compare i "true")) then "false" else "true" in
	    not_e (v), env

      | Sqrt(e) ->
	  let v, env = objeval env e in
	    string_of_float(sqrt(float_of_string(v))), env

      | Random(e) ->
	  let v, env = objeval env e in
	    string_of_int(Random.self_init (); Random.int (int_of_string(v))), env

      | ObjAssign(var, attr, e) ->
          let v, (locals, globals) = objeval env e in
            if NameMap.mem var locals then
              v, (NameMap.add var 
              	 (NameMap.add attr v (NameMap.find var locals) ) locals, globals)
            else if NameMap.mem var globals then
              v, (locals, NameMap.add var 
              	 (NameMap.add attr v (NameMap.find var globals) ) globals) 
            else raise (Failure ("undeclared identifier " ^ var))

      | Attribute(var, attr) ->
          let (locals, globals) = env in
            if NameMap.mem var locals then
              if NameMap.mem attr (NameMap.find var locals) then
                NameMap.find attr (NameMap.find var locals),(locals, globals)
              else raise (Failure ("undeclared attribute " ^ attr ^ 
              			" for local variable " ^ var))
            else if NameMap.mem var globals then
              if NameMap.mem attr (NameMap.find var globals) then
                NameMap.find attr (NameMap.find var globals),(locals, globals)
              else raise (Failure ("undeclared attribute " ^ attr ^ 
              			" for global variable " ^ var))
            else raise (Failure ("undeclared identifier " ^ var))

    in

    (* Evaluate an expression and return (value, updated environment) *)

    let rec eval env = function

        Id(var) ->
          let locals, globals = env in
            if NameMap.mem var locals then
             (NameMap.find var locals), env
            else if NameMap.mem var globals then
              (NameMap.find var globals), env 
            else raise (Failure ("undeclared identifier " ^ var))

        | Assign(var, attr, e) ->
            let v, (locals, globals) = objeval env e in
              if NameMap.mem var locals then
                NameMap.empty, (NameMap.add var (NameMap.add attr v 
                	       (NameMap.find var locals) ) locals, globals)
              else if NameMap.mem var globals then
                NameMap.empty, (locals, NameMap.add var 
                	       (NameMap.add attr v (NameMap.find var globals) ) globals) 
              else raise (Failure ("undeclared identifier " ^ var))

        | Call(f, actuals) ->
            let fdecl =
              try NameMap.find f func_decls
              with Not_found -> raise (Failure ("undefined function " ^ f))

            in

            let actuals, env = List.fold_left
                (fun (actuals, env) actual ->
                  let v, env = eval env actual in v :: actuals, env)
                ([], env) actuals

            in

            let (locals, globals) = env in

              try

                let globals = call fdecl actuals globals in 
                  NameMap.empty, (locals, globals) 
              
              with ReturnException(v, globals) -> v, (locals, globals)

    in

    (* Execute a statement and return an updated environment *)

    let rec exec env = function

      Block(stmts) -> List.fold_left exec env stmts

      | Expr(e) -> let _, env = eval env e in env

      | If(e, s1, s2) ->
          let v, env = objeval env e in
            exec env (if (0 = (String.compare v "true")) then s1 else s2)

      | While(e, s) ->
          let rec loop env =
 	    let v, env = objeval env e in
	      if (0 = (String.compare v "true")) then loop (exec env s) else env
	  in loop env

      | For(e1, e2, e3, s) ->
          let _, env = objeval env e1 in
            let rec loop env =
              let v, env = objeval env e2 in
                if (0 = (String.compare v "true")) then
                  let _, env = objeval (exec env s) e3 in
                  loop env
                else
                  env
            in loop env

      | Print(e) ->
          let str, env = objeval env e in
            print_endline str; env

    in

    (* Enter the function: bind actual values to formal arguments *)

    let locals =
      try List.fold_left2
        (fun locals formal actual -> NameMap.add formal actual locals)
        NameMap.empty fdecl.formals actuals
      with Invalid_argument(_) ->
        raise (Failure ("wrong number of arguments passed to " ^ fdecl.fname))

    in

    (* Add local variables to the symbol table *)

    let locals = var_decls locals fdecl.locals in 

    (* Execute each statement in sequence, return updated global symbol table *)

    snd (List.fold_left exec (locals, globals) fdecl.body)

  in 

  (* Run a program: add global variables to the symbol table, find and run "main"  *)

  let globals = var_decls NameMap.empty vars in 

  try
    call (NameMap.find "main" func_decls) [] globals
  with Not_found -> raise (Failure ("did not find the main() function"))
