(* type checking exceptions *)
exception Symbol_not_found of string

exception Symbol_redefinition of string * Types.mdl_type

exception Type_mismatch of Types.mdl_type * Types.mdl_type

(* symbol table *)
type symbol_table =
	{ parent : symbol_table option; 
		context : Jast.scope;
		mutable scope : Jast.var_decl list;
		mutable funcs : Jast.method_decl list
	}

let string_of_scope = function
	| Jast.Global -> "*GLOBAL*"
	| Jast.Method -> "*METHOD*"
	| Jast.Formal -> "*Formal*"
	| Jast.ForLoop -> "*FORLOOP*"

let dump env =
	print_string ("context=" ^ ((string_of_scope env.context) ^ "\n"));
	List.iter
		(fun symbol ->
					Printf.printf "{sname=%s, st=%s}\n" symbol.Jast.vname
						(Types.string_of_type symbol.Jast.vtype))
		env.scope;
	List.iter
		(fun symbol ->
					Printf.printf "{fname=%s, ft=%s}\n" symbol.Jast.fname
						(Types.string_of_type symbol.Jast.rettype))
		env.funcs

(* add symbol to current environment *)
let add_sym env (t, n) =
	if not (List.exists (fun sy -> sy.Jast.vname = n) env.scope)
	then
		{
			parent = env.parent;
			context = env.context;
			scope = { Jast.vname = n; Jast.vtype = t; Jast.vscope = env.context; } :: env.scope;
			funcs = env.funcs;
		}
	else raise (Symbol_redefinition (n, t))

let same_sig (l1 : Jast.var_decl list) (l2 : Types.mdl_type list) =
	List.fold_left2
		(fun b v1 t2 -> b && (v1.Jast.vtype = t2))
		true l1 l2

let add_func env (fdecl : Jast.method_decl) =
	try
		let signiture = List.map (fun v -> v.Jast.vtype) fdecl.Jast.formals
		and sameName =
			List.find (fun f -> f.Jast.fname = fdecl.Jast.fname) env.funcs in
		if (same_sig sameName.Jast.formals signiture)
		then
			raise(Symbol_redefinition (fdecl.Jast.fname, fdecl.Jast.rettype))
		else
			{
				parent = env.parent;
				context = env.context;
				scope = env.scope;
				funcs = fdecl :: env.funcs;
			}
	with Not_found ->
			{
				parent = env.parent;
				context = env.context;
				scope = env.scope;
				funcs = fdecl :: env.funcs;
			}

let rec find_env_sym_rec env n =
	try
	(* look in current scope ignore (dump env); *)
		(env, (List.find (fun sy -> sy.Jast.vname = n) env.scope))
	with
	| Not_found ->
			(match env.parent with
				| (* in top scope - symbol not found *) None ->
						raise (Symbol_not_found n)
				| (* search parent scope *) Some parent_env ->
						find_env_sym_rec parent_env n)

(* find a type for a given symbol - search current and then parent         *)
(* environments                                                            *)
let find_env_sym env n =
	(* print_string ("\nlooking for variable ***" ^ (n ^ "***\n")); *)
	snd (find_env_sym_rec env n)

let rec find_env_func_rec env fname (types : Types.mdl_type list) =
	try
	(* look in current scope ignore (dump env); *)
		(List.find
				(fun f ->
							f.Jast.fname = fname &&
							(same_sig f.Jast.formals types)) env.funcs)
	with
	| Not_found ->
			(match env.parent with
				| (* in top scope - symbol not found *) None ->
						raise (Types.MDLException ("Undeclared function: " ^ fname ^ "(" ^
									(String.concat "," (List.map Types.string_of_type types)) ^
									")"))
				| (* search parent scope *) Some parent_env ->
						find_env_func_rec parent_env fname types)

let find_env_func env fname (types : Types.mdl_type list) =
(*  print_string ("\nlooking for function ###" ^ (fname ^ "()###\n"));*)
	find_env_func_rec env fname types

let symbol_names env = List.map (fun symbol -> symbol.Jast.vname) env.scope

(* create a new subordinate scope - keep reference to parent scope *)
let newscope c env = { parent = Some env; context = c; scope = []; funcs = []; }

