/* Parser for Markush Description Language */

%{ open Ast  %}

%token LBRACK RBRACK LPAREN RPAREN LBRACE RBRACE 
%token COLON SEMICOLON COMMA 
%token PLUS MINUS TIMES DIVIDE
%token EQ NEQ LT LEQ GT GEQ NOT ASSIGN
%token INT MOL ATOM BOND STRING BOOLEAN TRUE FALSE
%token IF NOELSE ELSE FOR RETURN WHILE COVERS
%token <string> ID
%token EOF
%token <int> INT_LITERAL
%token <string> CHEM_LITERAL
%token <string> STRING_LITERAL

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left COVERS
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE
%right NOT

%start program
%type < Ast.program> program

%%

program:
  | /* nothing */ { [], [] }
  | program vdecl { ($2 :: fst $1), snd $1 }
  | program fdecl { fst $1, ($2 :: snd $1) }

fdecl:
  | types ID LPAREN formals_opt RPAREN LBRACE vdecl_list stmt_list RBRACE
      { { rettype = $1; 
	  fname = $2;
	  formals = $4; (* a list of tuples *)
	  locals = List.rev $7; (* a list of tuples *)
	  body =  List.rev $8 (* a list of statements *) } }

types:
  | INT { Types.Int }
  | MOL { Types.Mol }
  | ATOM { Types.Atom }
  | BOND { Types.Bond }
	| STRING { Types.String }
	| BOOLEAN { Types.Boolean }
	| INT LBRACK RBRACK { Types.Tuple(Types.Int) }
  | MOL LBRACK RBRACK { Types.Tuple(Types.Mol) }
	| STRING LBRACK RBRACK { Types.Tuple(Types.String) }
	| BOOLEAN LBRACK RBRACK { Types.Tuple(Types.Boolean) }

formals_opt:
  | /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
  | types ID                   { [($1 , $2)] }
  | formal_list COMMA types ID { ($3, $4) :: $1 }

vdecl_list:
  | /*nothing*/ { [] }
  | vdecl_list vdecl { $2::$1 }
   
vdecl:
  | types ID SEMICOLON { ($1, $2) }

stmt_list:
  | /* nothing */ { [] }
  | stmt_list stmt { $2::$1 }

stmt:
	/* Expression statement */
  | expr SEMICOLON { Expr($1) }
	
	/* Compound statement */
  | LBRACE stmt_list RBRACE { Block(List.rev $2) } 

	/* Conditional statements */
  | IF LPAREN expr RPAREN stmt %prec NOELSE { IfThen($3, $5) }
  | IF LPAREN expr RPAREN stmt ELSE stmt { IfThenElse($3, $5, $7) } 

	/* While statement */
  | WHILE LPAREN expr RPAREN stmt { While($3, $5) }

	/* For statement */
  | FOR LPAREN types ID COLON expr RPAREN stmt { For($3, $4, $6, $8) }

	/* Return statement */
  | RETURN expr SEMICOLON { Return($2) } 

expr:
	/* Primary expressions */
  | ID { Id($1) }
  | INT_LITERAL          { IntLiteral($1) }
  | chem_expr { ChemLiteral($1) }
	| STRING_LITERAL       { StringLiteral($1) }
	| TRUE      { BooleanLiteral(true) }
	| FALSE      { BooleanLiteral(false) }
  | LPAREN expr RPAREN { $2 }
  | ID LPAREN actuals_opt RPAREN { Call($1, $3) }

	/* Unary operators */
  | NOT expr { Unop(Types.Not, $2) }

	/* Binary operators */
  | expr PLUS   expr { Binop($1, Types.Add, $3) }
  | expr MINUS  expr { Binop($1, Types.Sub, $3) }
  | expr TIMES  expr { Binop($1, Types.Mult, $3) }
  | expr DIVIDE expr { Binop($1, Types.Div, $3) }
  | expr EQ     expr { Binop($1, Types.Equal, $3) }
  | expr NEQ    expr { Binop($1, Types.Neq,   $3) }
  | expr LT     expr { Binop($1, Types.Less,  $3) }
  | expr LEQ    expr { Binop($1, Types.Leq,   $3) }
  | expr GT     expr { Binop($1, Types.Greater,  $3) }
  | expr GEQ    expr { Binop($1, Types.Geq,   $3) }
	| expr COVERS expr { Covers($1, $3) }

	/* Assignment operator */
  | ID ASSIGN expr   { Assign($1, $3) }

	/* List operator */
  | LBRACK actuals_opt RBRACK { List($2) }

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
  | expr                    { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }

chem_expr:
  | CHEM_LITERAL { $1 }
