(* Pretty-prints a Jast into Java code *)

let default_keyword = "public static"

let default_implicit_param = "null"

let rec string_of_expr ((expr : Jast.expr_detail), (etype : Types.mdl_type)) =
	match expr with
	| Jast.Id vdecl -> vdecl.Jast.vname
	| Jast.IntLiteral l -> string_of_int l
	| Jast.ChemLiteral s ->
			"new Object() {
			public OEGraphMol evalChemLiteral() {
			OEGraphMol temp = new OEGraphMol();
			oechem.OEParseSmiles(temp, \""
			^ s ^ "\");
	return temp;
	}
	}.evalChemLiteral()"
	| Jast.StringLiteral s -> "\"" ^ s ^ "\""
	| Jast.BooleanLiteral b ->
			if b then "true" else "false"
	| Jast.Call (fdecl, te_list) ->
			(match fdecl.Jast.fname with
				| "getWeight" ->
						"RxnUtils.getAtomicWeight(" ^
						(String.concat "," (List.map string_of_expr te_list)) ^
						")"
				| "print" -> (
							let arg = List.hd te_list in
							let typ = snd arg in
							match typ with
							| Types.Int | Types.String | Types.Boolean ->
									"System.out.print(" ^ (string_of_expr arg) ^ ")"
							| Types.Mol ->
									"System.out.print(oechem.OECreateSmiString(" ^
									(string_of_expr arg) ^
									", OESMILESFlag.DEFAULT & ~OESMILESFlag.Canonical))"
							| Types.Atom | Types.Bond | Types.Tuple(_) ->
									raise (Types.MDLException ("Unsupported type for print(): " ^
												(Types.string_of_type typ)))
						)
				| "getAtoms" ->
						(String.concat "," (List.map string_of_expr te_list)) ^
						".GetAtoms()"
				| "getBonds" ->
						(String.concat "," (List.map string_of_expr te_list)) ^
						".GetBonds()"
				| "getIdx" ->
						(String.concat "," (List.map string_of_expr te_list)) ^
						".GetIdx()"
				| "getNbr" ->
						"(" ^ (string_of_expr (List.hd te_list)) ^ ")" ^ ".GetNbr(" ^
						(string_of_expr (List.nth te_list 1)) ^ ")"
				| "setMarked" ->
						"(" ^ (string_of_expr (List.hd te_list)) ^ ")" ^
						".SetBoolData(\"marked\", true)"
				| "isMarked" ->
						"(" ^ (string_of_expr (List.hd te_list)) ^ ")" ^
						".GetBoolData(\"marked\")"
				| "newBond" ->
						let createSmi id = "oechem.OECreateSmiString(" ^ id ^ ")" in
						"new Object() {
						public OEGraphMol evalChemLiteral() {
						OEGraphMol temp = new OEGraphMol();
						oechem.OEParseSmiles(temp, "
						^ (String.concat "+"
								(List.map createSmi
										(List.map string_of_expr te_list))) ^ ");
				return temp;
				}
				}.evalChemLiteral()"
				| "joinFrags" ->
						"RxnUtils.joinFrags" ^
						"(" ^
						(String.concat "," (List.map string_of_expr te_list)) ^ ")"
				| _ ->
						fdecl.Jast.fname ^
						("(" ^
							((String.concat "," (List.map string_of_expr te_list)) ^ ")")))
	| Jast.Unop (Types.Not, e) ->
			"!" ^ "(" ^ (string_of_expr e) ^ ")"
	| Jast.Unop (_, e) ->
			raise (Types.MDLException "Unsupported unary operator")
	| Jast.Binop (e1, o, e2) ->
			(string_of_expr e1) ^	" " ^ (Types.string_of_operator o) ^ " " ^
			(string_of_expr e2)
	| Jast.Assign (vdecl, e) ->
			(vdecl.Jast.vname) ^ " = " ^ (string_of_expr e)
	| Jast.List l ->
			let typeDetect fstElem =
				match fstElem with
				| Types.Int _ -> "int"
				| Types.Mol -> "OEGraphMol"
				| Types.Atom -> "OEAtomBase"
				| Types.Bond -> "OEBondBase"
				| Types.String -> "String"
				| Types.Boolean -> "boolean"
				| Types.Tuple(t) -> raise (Types.MDLException "Unsupported list type")
			in
			"new " ^ (typeDetect (snd (List.hd l))) ^ " [] {" ^
			(String.concat "," (List.map string_of_expr l)) ^ "}"
	| Jast.Covers (chem1, chem2) ->
			"RxnUtils.covers(" ^
			(string_of_expr chem1) ^ ", " ^ (string_of_expr chem2) ^ ")"

let rec string_of_type =
	function
	| Types.Int -> "int "
	| Types.Mol -> "OEGraphMol "
	| Types.Atom -> "OEAtomBase "
	| Types.Bond -> "OEBondBase "
	| Types.String -> "String "
	| Types.Boolean -> "boolean "
	| Types.Tuple(t) -> (string_of_type t) ^ "[] "

let string_of_vdecl vdecl =
	match vdecl.Jast.vscope with
	| Jast.Global ->
			"static " ^ (string_of_type vdecl.Jast.vtype) ^ vdecl.Jast.vname ^ ";\n"
	| Jast.Method ->
			(match vdecl.Jast.vtype with
				| Types.Mol | Types.Atom | Types.Bond | Types.String ->
						(string_of_type vdecl.Jast.vtype) ^ vdecl.Jast.vname ^ " = null;\n"
				| _ ->
						(string_of_type vdecl.Jast.vtype) ^ vdecl.Jast.vname ^ ";\n")
	| Jast.Formal | Jast.ForLoop ->
			string_of_type vdecl.Jast.vtype ^ vdecl.Jast.vname

let rec string_of_stmt =
	function
	| Jast.Block stmts ->
			"{\n" ^ ((String.concat "" (List.map string_of_stmt stmts)) ^ "}\n")
	| Jast.Expr expr -> (string_of_expr expr) ^ ";\n"
	| Jast.Return e -> "return " ^ ((string_of_expr e) ^ ";\n")
	| Jast.While (e, s) ->
			"while (" ^ ((string_of_expr e) ^ (")\n" ^ (string_of_stmt s)))
	| Jast.IfThenElse (e, s1, s2) ->
			"if (" ^ (string_of_expr e) ^
			")\n" ^ (string_of_stmt s1) ^ "else\n" ^ (string_of_stmt s2)
	| Jast.IfThen (e, s1) ->
			"if (" ^	(string_of_expr e) ^ ")\n" ^ (string_of_stmt s1)
	| Jast.For (vdecl, e, stmt) ->
			let loopType typ = match typ with
				| Types.Atom -> "OEAtomBase "
				| Types.Bond -> "OEBondBase "
				| Types.Mol -> "OEGraphMol "
				| Types.Int -> "int "
				| Types.String -> "String "
				| Types.Boolean -> "boolean "
				| Types.Tuple(t) ->
						raise (Types.MDLException ("Invalid loop variable " ^
									(Types.string_of_type typ)))
			in
			"for (" ^
			((loopType vdecl.Jast.vtype) ^ vdecl.Jast.vname ^
				(" : " ^
					((string_of_expr e) ^ (")\n" ^ ((string_of_stmt stmt) ^ "\n")))))

let string_of_method_decl f =
	let mainDetect f =
		match f.Jast.fname with
		| "main" ->
				if (List.length f.Jast.formals) = 0
				then " void main(String [] args)"
				else
					raise (Types.MDLException ("Entry point of a MDL program must have " ^
								"the signiture of int main()"))
		| _ ->
				" " ^
				((string_of_type f.Jast.rettype) ^
					(" " ^
						(f.Jast.fname ^
							("(" ^
								((String.concat ","
											(List.map string_of_vdecl f.Jast.formals))
									^ ")")))))
	in
	if (List.exists (fun fdecl -> f = fdecl) !(Semantics.genv).Sym.funcs)
	then ""
	else
		default_keyword ^
		(mainDetect f) ^
		"\n{\n\n" ^
		(String.concat "" (List.map string_of_vdecl f.Jast.locals)) ^
		(String.concat "" (List.map string_of_stmt f.Jast.body)) ^ "}\n"

let string_of_jclass =
	fun (Jast.JClass (vars, methods)) ->
			"\npublic class " ^ "MarkushDL" ^ "\n{\n" ^
			String.concat "\n" (List.map string_of_vdecl vars) ^
			"\n\n" ^
			String.concat "" (List.map string_of_method_decl methods) ^
			"}\n"

(* e.g., import lib.*; *)
let string_of_imports import_list = (String.concat ";\n" import_list) ^ ";\n"

let translate program =
	(string_of_imports [ "import openeye.oechem.*" ]) ^
	"\n" ^ (string_of_jclass program) ^ "\n"
