(* Scanner for Markush Description Language *)

{ open Parser }

let letter = ['a'-'z' 'A'-'Z']
let digit = ['0'-'9']
let num = digit+ | digit*'.'digit+

rule token = parse
| "/*"    { comment lexbuf }
| [' ' '\t' '\r' '\n'] { token lexbuf }
(* 1 *)
| '(' { LPAREN }   
| ')' { RPAREN }
| '{' { LBRACE }
| '}' { RBRACE }
| '[' { LBRACK }
| ']' { RBRACK }

(* 2 *)
| ':' { COLON }
| ';' { SEMICOLON }
| ',' { COMMA }

(* 3 *)
| '+' { PLUS }
| '-' { MINUS }
| '*' { TIMES }
| '/' { DIVIDE }

(* 4 *)
| "==" { EQ }
| "!=" { NEQ }  
| '<'  { LT }
| "<=" { LEQ }
| '>'  { GT }
| ">=" { GEQ }
| '!'  { NOT } 
| '='  { ASSIGN }

(* 5 *)
| "int"          { INT }
| "Mol"          { MOL }
| "Atom"         { ATOM }
| "Bond"         { BOND }
| "String"       { STRING }
| "boolean"      { BOOLEAN }
| "true"         { TRUE }
| "false"        { FALSE }

(* 6 *)
| "if"           { IF }
| "else"         { ELSE }
| "for"          { FOR }
| "return"       { RETURN }
| "while"        { WHILE }
| "covers"       { COVERS }

| digit+ as s    { INT_LITERAL(int_of_string(s)) }
| letter (letter | digit | '_')* as identi { ID(identi) }
| eof            { EOF }

| "'''"          { chem_type "" lexbuf}

| '"'            { string_type "" lexbuf}

(* comment *)
and comment = parse
  "*/" { token lexbuf }
| _ { comment lexbuf }

and chem_type str = parse
   "'''" { CHEM_LITERAL(str) }
|  "\n" {   let pos = lexbuf.Lexing.lex_curr_p in
            raise (Failure("Unclosed string literal, found beginning of" 
            ^ " a new line without closure of string; "
            ^ " in line #" ^ (string_of_int pos.Lexing.pos_lnum))) }
|  _ { chem_type (str^(Lexing.lexeme lexbuf)) lexbuf }

and string_type str = parse
   '"' { STRING_LITERAL(str) }
|  "\n" {   let pos = lexbuf.Lexing.lex_curr_p in
            raise (Failure("Unclosed string literal, found beginning of" 
            ^ " a new line without closure of string; "
            ^ " in line #" ^ (string_of_int pos.Lexing.pos_lnum))) }
|  _ { string_type (str^(Lexing.lexeme lexbuf)) lexbuf }
