import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import openeye.oechem.OEAtomBase;
import openeye.oechem.OEExprOpts;
import openeye.oechem.OEGraphMol;
import openeye.oechem.OEHasAtomicNum;
import openeye.oechem.OEQMol;
import openeye.oechem.OESubSearch;
import openeye.oechem.oechem;

/**
 * @author yun03
 *
 */
public final class RxnUtils {

	static final String R_PATTERN = "\\[R\\d+\\]";
//	private static final String Z_PATTERN = "\\[Z\\d+\\]";

	public static OEGraphMol [] joinFrags(OEGraphMol [] lhs, OEGraphMol [] rhs) {
		ArrayList<OEGraphMol> result = new ArrayList<OEGraphMol>();
		for (OEGraphMol m1 : rhs) {
			ArrayList<Integer> rgs = new ArrayList<Integer>();
			for (OEAtomBase atom : m1.GetAtoms( new OEHasAtomicNum(0))) {
				rgs.add( atom.GetMapIdx() );
			}
			
			String s1 = oechem.OECreateSmiString(m1);
			s1 = replaceRZ(s1, R_PATTERN, rgs);
			for (OEGraphMol m2 : lhs) {
				String s2 = oechem.OECreateSmiString(m2);
				s2 = replaceRZ(s2, R_PATTERN, rgs);
				OEGraphMol mol = new OEGraphMol();
				oechem.OEParseSmiles(mol, s1+"."+s2);
				
				oechem.OEAssignMDLHydrogens( mol );
				oechem.OEFindRingAtomsAndBonds( mol );
				oechem.OEAssignAromaticFlags( mol );

				result.add(mol);
			}
		}
		return result.toArray(new OEGraphMol [0]);
	}

	static String replaceRZ(String smiWithRZ, String regex, List<Integer> rgs) {
		Pattern p = Pattern.compile("^"+regex);		
		Matcher m = p.matcher(smiWithRZ);
		if (m.find()) {
			String head = m.group();
			int end = m.end();
			String atom = smiWithRZ.substring(end, end+1);
			smiWithRZ = atom + "(" + head + ")" + smiWithRZ.substring(end+1);
		}
		
		p = Pattern.compile(regex);		
		m = p.matcher(smiWithRZ);
		StringBuffer sb = new StringBuffer();
		while (m.find()) {
			String num = smiWithRZ.substring(m.start()+2, m.end()-1);
			int n = Integer.parseInt(num);
			if (rgs.contains(n)) {
				m.appendReplacement(sb, "%"+(80+n));
			}
		}
		m.appendTail(sb);
		return sb.toString();
	}
	
	public static boolean covers(OEGraphMol[] markush, OEGraphMol test) {
		for (OEGraphMol mol : markush) {
			oechem.OESuppressHydrogens(mol);
			OEQMol qmol = new OEQMol(mol);
			qmol.BuildExpressions(OEExprOpts.DefaultAtoms, OEExprOpts.DefaultBonds);
			OESubSearch sss = new OESubSearch();
			sss.Init(qmol);
			if (sss.SingleMatch(test)) {
				return true;
			}
		}
		return false;		
	}
	
	/**
	 * Note that only selected elements frequently seen in organic molecules
	 * are supported. Result is rounded to the nearest integer.
	 * 
	 * @param atom
	 * @return weight of the input atom 
	 */
	public static int getAtomicWeight(OEAtomBase atom) {
		switch (atom.GetAtomicNum()) {
		case 0: // R-group
			return 0;
		case 1: // Hydrogen
			return 1;
		case 6: // Carbon
			return 12;
		case 7: // Nitrogen
			return 14;
		case 8: // Oxygen
			return 16;
		case 9: // Fluorine
			return 19;
		case 11: // Sodium
			return 23;
		case 12: // Magnesium
			return 23;
		case 16: // Sulfur
			return 32;
		case 17: // Chlorine
			return 35;
		default:
			throw new RuntimeException("Unsupported atom: " + atom);
		}
	}
}
