open Ast

let rec string_of_expr = function
    Literal(l) -> string_of_int l
  | Price(f) -> string_of_float f
  | Symbol(s, c) -> "api.lookup(\"" ^ s ^ "\")" ^ c 
  | Id(s) -> s
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^
      (match o with
	Add -> "+" | Sub -> "-" | Mult -> "*" | Div -> "/"
      | Equal -> "==" | Neq -> "!="
      | Less -> "<" | Leq -> "<=" | Greater -> ">" | Geq -> ">=" 
      | And -> "&&" | Or -> "||" | Not -> "!") ^ " " ^
      string_of_expr e2
  | Assign(v, e) -> v ^ " = " ^ string_of_expr e
  | True -> "true"
  | False -> "false"
  | Noexpr -> ""

let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return -> "break;\n";
  | Wait(i) -> "sleep(" ^ string_of_int i ^ ");"
  | Waitexe(s) -> "\nnew Thread( \n new Runnable() { \npublic void run() {" ^
   " try { " ^ string_of_stmt s ^ " } \ncatch (Exception e) { \ne.printStackTrace();" ^
   " } \n } \n}).start();"
  | Exit -> "System.exit(0);"
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s
  | Buy(e, s) -> "api.buy(\"" ^ e ^ "\", " ^ string_of_int s ^ ");\n"
  | Sell(e, s) -> "api.sell(\"" ^ e ^ "\", " ^ string_of_int s ^ ");\n"
  | Lookup(s, c) -> "api.lookup(\"" ^ s ^ "\")" ^ c
  | Print(s) -> "api.print(" ^ s ^ ");"
  

let string_of_vdecl id = "private double " ^ id ^ ";\n"

let string_of_fdecl fdecl =
  " class " ^ fdecl.fname ^ " extends Thread {\n" ^
  "	private boolean running = true; \n" ^
  String.concat "" (List.map string_of_vdecl fdecl.locals) ^
  "   public void run() { \n" ^
  " while(running) {\n" ^
  " try { \n" ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "} catch (Exception e) { \n System.out.println(e); \n running = false; " ^
  "}\n   }\n  }\n }\n"

let string_of_threads tdecl =
  let rec helper i = function 
	[] -> "\n";
      | h::t -> (helper(i - 1) t) ^
  "     threads[" ^ string_of_int i ^ "] = new " ^ h.fname ^ "();\n"
  in helper ((List.length tdecl) - 1) tdecl
  

let string_of_program (vars, funcs) =
  "package hootie.generated;\n" ^
  "import hootie.connector.StockTradeAPI;\n\n" ^
  "public class LanguageStructure { \n" ^
  "    private StockTradeAPI api;\n" ^
  "    public Thread[] threads = new Thread[" ^ string_of_int (List.length funcs) ^ "];\n" ^
  "    public LanguageStructure(StockTradeAPI api) {\n" ^
  (string_of_threads funcs) ^
  "         this.api = api; "^
  "\n    }" ^
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl funcs) ^
  "}\n"
