%{ open Ast %}

%token SEMICOLON COMMA
%token LEFT_PAREN RIGHT_PAREN LEFT_BRACE RIGHT_BRACE LEFT_BRACKET RIGHT_BRACKET
%token PLUS MINUS TIMES DIVIDE ASSIGN
%token EQUAL NOT_EQUAL LESS_THAN GREATER_THAN LESS_EQUAL GREATER_EQUAL
%token RETURN IF ELSE WHILE INT VOID
%token <int> LITERAL
%token <string> ID
%token EOF

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left EQUAL NOT_EQUAL
%left LESS_THAN GREATER_THAN LESS_EQUAL GREATER_EQUAL
%left PLUS MINUS
%left TIMES DIVIDE

%start program
%type <Ast.program> program

%%

program:
    /* nothing */     { [], [] }
  | program variable_declaration  { ($2 :: fst $1), snd $1 }
  | program function_definition   { fst $1, ($2 :: snd $1) }


function_definition:
    ID LEFT_PAREN parameter_opt RIGHT_PAREN
      LEFT_BRACE declaration_opt statement_opt RIGHT_BRACE
      { {
        fname = $1;
        formals = $3;
        locals = $6;
        body = $7;
      } }

parameter_opt:
    /* nothing */    { [] }
  | parameter_list   { List.rev $1 }

parameter_list:
    parameter_declaration                        { [$1] }
  | parameter_list COMMA parameter_declaration   { $3 :: $1 }

parameter_declaration:
    INT ID    { Int($2) }

variable_declaration:
    INT ID SEMICOLON                                  { Int($2) }
  | INT ID LEFT_BRACKET LITERAL RIGHT_BRACKET SEMICOLON   { Array($2, $4) }

variable_reference:
    ID                                       { IntRef($1) }
  | ID LEFT_BRACKET expression RIGHT_BRACKET { ArrayRef($1, $3) }

declaration_opt:
    /* nothing */     { [] }
  | declaration_list  { List.rev $1 }

declaration_list:
    variable_declaration                    { [$1] }
  | declaration_list variable_declaration   { $2 :: $1 }

statement_opt:
    /* nothing */   { [] }
  | statement_list  { List.rev $1 }

statement_list:
    statement                 { [$1] }
  | statement_list statement  { $2 :: $1 }

statement:
    expression SEMICOLON { Expression($1) }
  | RETURN expression SEMICOLON { Return($2) }
  | LEFT_BRACE statement_opt RIGHT_BRACE { Block($2) }
  | IF LEFT_PAREN expression RIGHT_PAREN statement %prec NOELSE { If($3, $5, Block([])) }
  | IF LEFT_PAREN expression RIGHT_PAREN statement ELSE statement { If($3, $5, $7) }
  | WHILE LEFT_PAREN expression RIGHT_PAREN statement { While($3, $5) }

expression:
    LITERAL                    { Literal($1) }
  | variable_reference         { VarRef($1) }
  | expression PLUS expression { Binop($1, Add, $3) }
  | expression MINUS expression { Binop($1, Sub, $3) }
  | expression TIMES expression { Binop($1, Mult, $3) }
  | expression DIVIDE expression { Binop($1, Div, $3) }
  | expression EQUAL expression { Binop($1, Equal, $3) }
  | expression NOT_EQUAL expression { Binop($1, NotEqual, $3) }
  | expression LESS_THAN expression { Binop($1, LessThan, $3) }
  | expression GREATER_THAN expression { Binop($1, GreaterThan, $3) }
  | expression LESS_EQUAL expression { Binop($1, LessEqual, $3) }
  | expression GREATER_EQUAL expression { Binop($1, GreaterEqual, $3) }
  | variable_reference ASSIGN expression { Assign($1, $3) }
  | ID LEFT_PAREN argument_opt RIGHT_PAREN
	  { match $1 with
	    "print" -> Call("printf", Data("$output") :: $3)
	  | _ -> Call($1, $3) }
  | LEFT_PAREN expression RIGHT_PAREN { $2 }

argument_opt:
    /* nothing */  { [] }
  | argument_list  { List.rev $1 }

argument_list:
    expression                     { [$1] }
  | argument_list COMMA expression { $3 :: $1 }
