type operator = Add | Sub | Mult | Div | Equal | NotEqual | LessThan | GreaterThan
        | LessEqual | GreaterEqual

type return_type =
    IntRet
  | Void

type var =
    Int of string
  | Array of string * int

type vref =
    IntRef of string
  | ArrayRef of string * expression
and expression =
    Literal of int
  | VarRef of vref
  | Binop of expression * operator * expression
  | Assign of vref * expression
  | Call of string * expression list
  | Data of string
  | Noexpr

type statement =
    Block of statement list
  | Expression of expression
  | Return of expression
  | If of expression * statement * statement
  | While of expression * statement

type function_definition = {
    fname : string;
    formals : var list;
    locals : var list;
    body : statement list;
  }

type program = var list * function_definition list

let rec string_of_var = function
    Int(s) -> s
  | Array(s, i) -> s ^ "[" ^ string_of_int i ^ "]"
and string_of_vref = function
    IntRef(s) -> s
  | ArrayRef(s, e) -> s ^ "[" ^ string_of_expr e ^ "]"
and string_of_expr = function
    Literal(l) -> string_of_int l
  | VarRef(v) -> string_of_vref v
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^
      (match o with
	    Add -> "+" | Sub -> "-" | Mult -> "*" | Div -> "/" | Equal -> "=="
      | NotEqual -> "!=" | LessThan -> "<" | LessEqual -> "<="
      | GreaterThan -> ">" | GreaterEqual -> ">=") ^ " " ^ string_of_expr e2
  | Assign(v, e) -> string_of_vref v ^ " = " ^ string_of_expr e
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Data(s) -> "\"" ^ s ^ "\""
  | Noexpr -> ""

let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expression(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s

let string_of_vdecl = function
    Int(id) -> "int " ^ id ^ ";\n"
  | Array(id, size) -> "int "^ id ^ "[" ^ string_of_int size ^ "];\n"

let string_of_farg = function
    Int(id) -> "int " ^ id
  | Array(id, size) -> "int "^ id ^ "[" ^ string_of_int size ^ "]"

let string_of_fdecl fdecl =
  fdecl.fname ^ "(" ^
  String.concat ", " (List.map string_of_farg fdecl.formals) ^ ")\n{\n" ^
  String.concat "" (List.map string_of_vdecl fdecl.locals) ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"
    
let string_of_program (vars, funcs) =
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl funcs)
