(* compiler driver
  - invokes each compiler pass in sequence *)

open Unix

let cpp_compiler = "g++"
let pcre_include = ""
let pcre_lib = ""
let pcre_name = "-lpcrecpp"

type action = Ast | Compile | Assemble | Help | Version

let version (n:unit) =
	"strlang version 0.1 (Green Eggs 'n Ham) 12/22/11"

let usage (name:string) =
	"usage:\n" ^ name ^ "\n" ^
		"        -a file.str                (dump AST of source)\n" ^
		"        -c file.str [file.cpp]     (compile to C++)\n" ^
		"        -e file.str file.exe       (compile to executable)\n" ^
		"        -h                         (display this message)\n" ^
		"        -v                         (display version number)\n"

let get_compiler_path (path:string) =
	try
		let i = String.rindex path '/' in
		String.sub path 0 i
	with _ -> "."

let _ =
	let action =
		if Array.length Sys.argv > 1 then
			(match Sys.argv.(1) with
				"-a" -> if Array.length Sys.argv == 3 then Ast else Help
				| "-c" ->
					if (Array.length Sys.argv == 3) ||
						(Array.length Sys.argv == 4) then Compile else Help
				| "-e" -> if Array.length Sys.argv == 4 then Assemble else Help
				| "-v" -> Version
				| _ -> Help)
		else Help in
	match action with
		Help -> print_endline (usage Sys.argv.(0))
		| Version -> print_endline (version ())
		| (Ast | Compile | Assemble) ->
			let input = open_in Sys.argv.(2) in
			let lexbuf = Lexing.from_channel input in
			let ast = Parser.program Scanner.token lexbuf in
			if action = Ast then print_endline (Ast.string_of_program ast)
			else
				let env = Symtab.symtab_of_program ast in
				let checked = Check.check_program ast env in
				let simple = Simple.simplify_program checked in
				let program = Output.c_of_simple simple in
				if action = Compile then
					if Array.length Sys.argv == 3 then print_endline program
					else
						let out = open_out Sys.argv.(3) in
						output_string out program; close_out out
				else
					let outfilename = Sys.argv.(3) ^ "_strlang.cpp" in
					let out = open_out outfilename in
					output_string out program; close_out out;
					execvp cpp_compiler [|cpp_compiler; pcre_include; "-I" ^ (get_compiler_path Sys.argv.(0));
						outfilename; "-o"; Sys.argv.(3); pcre_lib; pcre_name|]
					