%{ open Ast

let block_id = ref 1

let gen_block_id (u:unit) =
	let x = block_id.contents in
		block_id := x + 1; x
	
%}

%token LPAREN RPAREN LBRACE RBRACE LBRACKET RBRACKET EOF
%token PLUS MINUS TIMES DIVIDE MOD
%token OR AND
%token LESS GREATER LESSEQUAL GREATEREQUAL EQUAL UNEQUAL NOT
%token CARET AT TILDE
%token LARROW RARROW

%token SEMI

%token STR NUM

%token <string> STR_LITERAL
%token <int> NUM_LITERAL

%token <string> ID

%nonassoc NOELSE

%right LARROW

%left OR 
%left AND 

/* theoretically EQUAL and UNEQUAL should have different precedence, but this breaks other things */
%left LESS LESSEQUAL GREATER GREATEREQUAL EQUAL UNEQUAL

%left PLUS MINUS
%left TIMES DIVIDE MOD
%left TILDE

%left NEG NOT CARET
%left LBRACKET

%right AT

%start program
%type <Ast.program> program

%%

program:
	var_list func_list { { globals = List.rev $1; functions = List.rev $2; block_count = gen_block_id ()} }

func_list:
	{ [] }
	| func_list func { $2 :: $1 }

func:
	ID LPAREN formals_list_opt RPAREN RARROW var_type_opt block
		{ { name = $1; ret_type = $6; body = $7; formals = $3} }

block:
	LBRACE var_list stmt_list RBRACE
		{ {locals = List.rev $2; statements = $3; block_id = gen_block_id ()} }

stmt_list:
	{ [] }
	| stmt stmt_list { $1 :: $2 }

stmt:
	block { CodeBlock($1) }
	| expr SEMI { Expression($1) }
	| LESS expr GREATER block { Loop($2, $4) }
	| LBRACKET expr RBRACKET block NOT LBRACKET RBRACKET block { Conditional($2, $4, $8) }
	| LBRACKET expr RBRACKET block %prec NOELSE { Conditional($2, $4, {locals = []; statements = []; block_id = -1}) }
	| RARROW expr SEMI { Return($2) }
	| RARROW SEMI { Return(NoExpr) }

expr:
	/* binary arithmetic operators */
	expr PLUS expr { Binop($1, Plus, $3) }
	| expr MINUS expr { Binop($1, Minus, $3) }
	| expr TIMES expr { Binop($1, Times, $3) }
	| expr DIVIDE expr { Binop($1, Divide, $3) }
	| expr MOD expr { Binop($1, Mod, $3) }

	/* comparison */
	| expr LESS expr { Binop($1, Less, $3) }
	| expr GREATER expr { Binop($1, Greater, $3) }
	| expr LESSEQUAL expr { Binop($1, Leq, $3) }
	| expr GREATEREQUAL expr { Binop($1, Geq, $3) }

	| expr TILDE expr TILDE expr { Replace($1, $3, $5) }

	/* equality */
	| expr EQUAL expr { Binop($1, Eq, $3) }
	| expr UNEQUAL expr { Binop($1, Neq, $3) }
	
	/* logical or/and */
	| expr OR expr { Binop($1, Or, $3) }
	| expr AND expr { Binop($1, And, $3) }

	/* assignment */	
	| lvalue LARROW expr { Assign($1, $3) }
	
	/* arithmetic and logical negation */
	| MINUS expr %prec NEG { Unop($2, Neg) }
	| NOT expr { Unop($2, Not) }
	
	/* length operator */
	| CARET expr { Unop($2, Len) }
	
	/* keys operator */
	| AT MOD expr { Unop($3, Keys) }
	
	/* values operator */
	| AT AT expr { Unop($3, Vals) }
	
	/* grouping */
	| LPAREN expr RPAREN { $2 }

	/* function call */
	| ID LPAREN actuals_list_opt RPAREN { FuncCall($1, $3) }
	
	/* variable */
	| lvalue { Rvalue($1) }

	/* literals */
	| STR_LITERAL { StrLiteral($1) }
	| NUM_LITERAL { NumLiteral($1) }

formals_list_opt:
	formals_list { $1 }
	| CARET { [] }
	
formals_list:
	var { [$1] }
	| var SEMI formals_list { $1 :: $3 }

actuals_list_opt:
	{ [] }
	| actuals_list { $1 }

actuals_list:
	expr { [$1] }
	| expr SEMI actuals_list { $1 :: $3 }

var_list:
	{ [] }
	| var_list var SEMI { $2 :: $1 }

var:
	var_type ID { ($2, $1) }

var_type_opt:
	CARET { Simple(None) }
	| var_type {$1}

var_type:
	STR { Simple(Str) }
	| NUM { Simple(Num) }
	| MOD LBRACKET STR SEMI STR RBRACKET { Map(Str, Str) }
	| MOD LBRACKET NUM SEMI STR RBRACKET { Map(Num, Str) }
	| MOD LBRACKET STR SEMI NUM RBRACKET { Map(Str, Num) }
	| MOD LBRACKET NUM SEMI NUM RBRACKET { Map(Num, Num) }

lvalue:
	ID { ($1, NoExpr) }
	| ID LBRACKET expr RBRACKET { ($1, $3) }
