(* output
 	-takes simple IR produced by the simlification pass
	-outputs C++ code
*)

open Ast
open Check
open Simple

let type_of_var (v:simple_var) =
	let (_,t,_) = v in
	t

let c_of_bop = function
	Plus -> "+" | Minus -> "-" | Times -> "*" | Divide -> "/" | Mod -> "%"
	| Less -> "<" | Greater -> ">" | Leq -> "<=" | Geq -> ">=" | Eq -> "=="
	| Neq -> "!=" | Or -> "||" | And -> "&&"
	
let c_of_uop = function
	Neg -> "-" | Not -> "!" | _ -> raise(Failure("internal error"))

let c_of_var_type = function
	Simple(Str) -> "string"
	| Simple(Num) -> "int"
	| Simple(None) -> "void"
	| Map(k,v) ->
		"map <" ^ (match (k,v) with
			(Str,Str) -> "string,string"
			| (Str,Num) -> "string,int"
			| (Num,Num) -> "int,int"
			| (Num,Str) -> "int,string"
			| _ -> raise(Failure("internal error"))) ^ ">"

let c_of_param_type t =
	if t = Simple(None) then "void"
	else (c_of_var_type t) ^ "&"

let c_of_var_name (decl:simple_var) =
	let (name,_,id) = decl in
	let suffix = if id = -1 then "" else string_of_int id in
	name ^ "_" ^ suffix

let c_of_var_init = function
	Simple(Num) -> "(0)"
	| Simple(Str) -> "(\"\")"
	| Map(_,_) -> ""
	| _ -> raise(Failure("internal error"))

let c_of_var_decl (v:simple_var) =
	let (_, t, _) = v in
	(c_of_var_type t) ^ " " ^ c_of_var_name v

let c_of_param (p:simple_var) =
	let (_, t, _) = p in
	(c_of_param_type t) ^ " " ^ c_of_var_name p

let c_of_formals_list = function
	[] -> "void"
	| formals -> String.concat ", " (List.map c_of_param formals)

let c_of_actual_list = function
	[] -> ""
	| actuals -> String.concat ", " (List.map c_of_var_name actuals)

let c_of_var_decl_def (v:simple_var) =
	let (_,t,_) = v in
	(c_of_var_decl v) ^ (c_of_var_init t)

let c_of_var_decl_list = function
	[] -> ""
	| vars-> (tabs 0 ^ (String.concat (";\n" ^ tabs 0) (List.map c_of_var_decl_def vars)) ^ ";\n")

let c_of_func_decl_formals = function
	[] -> "void"
	| vars -> String.concat (", ") (List.map c_of_param_type vars)

let c_of_func_decl (f:simple_fdecl) =
	let (name, ret, formals, id) = f in
	c_of_var_type ret ^ " " ^ name ^ "(" ^ c_of_func_decl_formals formals ^ ");\n"

let c_of_func_decl_list = function
	[] -> ""
	| fdecls -> String.concat ("\n") (List.map c_of_func_decl fdecls) ^ "\n\n"

let c_of_method_call (result:simple_var) (fname:string) (receiver:simple_var) args =
	let(_,t,_) = result in
	(if t <> Simple(None) then (c_of_var_name result) ^ " = " else "") ^ (c_of_var_name receiver) ^ "." ^
		fname ^ "(" ^ (c_of_actual_list args) ^ ")"

let c_of_func_call (result:simple_var) (fname:string) (args:simple_var list) =
	let (_,t,_) = result in
	(if t <> Simple(None) then (c_of_var_name result) ^ " = " else "") ^ fname ^ "(" ^
		(c_of_actual_list args) ^ ")"

let c_of_func_name = function
	(name,_,_,_) -> name

let c_of_unop (result:simple_var) (op:Ast.uop) (expr:simple_var) =
		(c_of_var_name result) ^ " = " ^ c_of_uop op ^ " " ^ (c_of_var_name expr)

let c_of_binop (result:simple_var) (op:Ast.bop) (e1:simple_var) (e2:simple_var) =
	(c_of_var_name result) ^ " = " ^ (c_of_var_name e1) ^ " " ^
		c_of_bop op ^ " " ^ (c_of_var_name e2)

let c_of_expr = function
	Bin(r, op, e1, e2) -> (c_of_var_name r) ^ " = " ^ (c_of_var_name e1) ^ " " ^
		c_of_bop op ^ " " ^ (c_of_var_name e2)
	| Un(r, op, e) -> c_of_unop r op e
	| Lit(r, v) -> (c_of_var_name r) ^ " = " ^
		(match v with
		StrLit(s) -> s
		| NumLit(n) -> string_of_int n)
	| Deref(result, base, accessor) ->
		(c_of_var_name result) ^ " = " ^ (c_of_var_name base) ^ "[" ^ c_of_var_name accessor ^ "]"
	| Alias(base, accessor, input) ->
		let (_,t,_) = accessor in
		let extra = if t <> Simple(None) then "[" ^ (c_of_var_name accessor) ^ "]" else "" in
		(c_of_var_name base) ^ extra ^ " = " ^ (c_of_var_name input)
	| Call(r, f, el) ->
		c_of_func_call r (c_of_func_name f) el

let rec c_of_stmt = function
	Decl(v) -> if type_of_var v <> Simple(None) then tabs 0 ^ (c_of_var_decl_def v) ^ ";\n"
	else ""
	| If(r, l) -> tabs 0 ^ "if(" ^ (c_of_var_name r) ^ ") goto " ^ l ^ ";\n"
	| Label(s) -> s ^ ": ;\n"
	| Jmp(s) -> tabs 0 ^ "goto " ^ s ^";\n"
	| Ret(e) -> let (_,t,_) = e in tabs 0 ^ "return" ^ (if t <> Simple(None) then " " ^ c_of_var_name e else "") ^ ";\n"
	| Expr(e) -> tabs 0 ^ c_of_expr e ^ ";\n"

and c_of_stmt_list = function
	[] -> ""
	| head::tail -> (c_of_stmt head) ^ (c_of_stmt_list tail)

and c_of_block (sl:simple_stmt list) =
	let s = "\n" ^ tabs 0 ^ "{\n" in
	depth := depth.contents + 1;
	let s = s ^ String.concat "" (List.map c_of_stmt sl) in
	depth := depth.contents - 1; s ^ tabs 0 ^ "}\n\n"


and c_of_header (header:simple_fdecl) (args:simple_var list) =
	let (name, ret, _, id) = header in
	tabs 0 ^ c_of_var_type ret ^ " " ^ name ^ "(" ^ c_of_formals_list args ^ ")"

and c_of_funclist = function
	[] -> ""
	| head::tail -> c_of_header head.header head.args ^ (c_of_block head.code) ^
		(c_of_funclist tail)

let rec c_of_simple (p:simple_program) =
	depth := 0; "#include \"strlib.h\"\n\n" ^ (c_of_func_decl_list p.fdecls) ^
	c_of_var_decl_list p.gvars ^ c_of_funclist p.funcs
