(* Written by Erica Sponsler and Nate Weiss *)

type bstmt =
    LitI of int   (* Push a literal *)
  | LitC of char  (* Push a character *)
  | LitB of bool  (* Push a boolean *)
  | LitNull       (* Push null *)
  | Drp of int    (* Drop a certain number of bytes from the stack *)
  | Bin of Ast.op (* Perform arithmetic on top of stack *)
  | Lod           (* Fetch global variable *)
  | Ldh           (* Fetch from the heap *)
  | Str           (* Store global variable *)
  | Sth           (* Store value to the heap *)
  | Lfp           (* Load frame pointer relative *)
  | Sfp           (* Store frame pointer relative *)
  | Cnd          (* Create a node on the node-heap with value at top of stack *)
  | Jsr of int    (* Call function by absolute address *)
  | Ssp
  | Rsp
  | Ent of int    (* Push FP, FP -> SP, SP += i *)
  | Rts of int    (* Restore FP, SP, consume formals, push result *)
  | Beq of int    (* Branch relative if top-of-stack is zero *)
  | Bne of int    (* Branch relative if top-of-stack is non-zero *)
  | Bra of int    (* Branch relative *)
  | Hlt           (* Terminate *)

type prog = {
    num_globals : int;   (* Number of global variables *)
    text : bstmt array; (* Code for all the functions *)
  }

let string_of_stmt = function
    LitI(i) -> "LitI " ^ string_of_int i
  | LitC(c) -> "LitC " ^ string_of_int (int_of_char c)
  | LitB(b) -> "LitB " ^ string_of_bool b
  | LitNull -> "LitNull"
  | Drp(i) -> "Drp " ^ string_of_int i
  | Bin(Ast.Add) -> "Add"
  | Bin(Ast.Sub) -> "Sub"
  | Bin(Ast.Mult) -> "Mul"
  | Bin(Ast.Div) -> "Div"
  | Bin(Ast.Equal) -> "Eql"
  | Bin(Ast.Neq) -> "Neq"
  | Bin(Ast.Less) -> "Lt"
  | Bin(Ast.Leq) -> "Leq"
  | Bin(Ast.Greater) -> "Gt"
  | Bin(Ast.Geq) -> "Geq"
  | Bin(Ast.BoolAnd) -> "And"
  | Bin(Ast.BoolOr) -> "Or"
  | Lod -> "Lod"
  | Ldh -> "Ldh"
  | Str -> "Str"
  | Lfp -> "Lfp"
  | Sfp -> "Sfp"
  | Sth -> "Sth"
  | Cnd -> "Cnd"
  | Jsr(i) -> "Jsr " ^ string_of_int i
  | Ssp -> "Ssp"
  | Rsp -> "Rsp"
  | Ent(i) -> "Ent " ^ string_of_int i
  | Rts(i) -> "Rts " ^ string_of_int i
  | Bne(i) -> "Bne " ^ string_of_int i
  | Beq(i) -> "Beq " ^ string_of_int i
  | Bra(i) -> "Bra " ^ string_of_int i
  | Hlt    -> "Hlt"

let string_of_prog p =
  string_of_int p.num_globals ^ " global variables\n" ^
  let funca = Array.mapi
      (fun i s -> string_of_int i ^ " " ^ string_of_stmt s) p.text
  in String.concat "\n" (Array.to_list funca)
