(* 
scanner.mll
-----------
This file specifies the regular expressions and the semantic actions for the lexical analyzer
*)

(* header *)
{
    open Parser
    open Printf
    open Char
}

(* definitions section *)
let alphabet = ['a'-'z''A'-'Z']
let digit = ['0'-'9']
let start_namechar = alphabet | '_' | '-'
let name_char = alphabet | '-' | '_' | digit
let name = name_char+
let ident = start_namechar name_char*
let decimal_point = '.'
let num = digit+ | digit* decimal_point digit+
let at = '@'
let nl = '\n' | "\r\n" | '\r'
let squote = "'"
let dquote = '"'
let single_quote_str = squote ([^'\n' '\r' '\''] | '\\' nl )* squote
let double_quote_str = dquote ([^'\n' '\r' '"'] | '\\' nl )* dquote
let quoted_str = single_quote_str | double_quote_str
let noquote_str = [^'\n' '\r' '\'' '"']+
let space = [' ' '\t' '\r']
let url = "url(" (quoted_str | noquote_str) ")"
let hex = ['0'-'9''a'-'f''A'-'F']
let hex_color = "#" (hex hex hex | hex hex hex hex hex hex)

(* rules section *)
rule csslang_token = parse
| space { csslang_token lexbuf } (* Whitespace *)
| nl { Lexing.new_line lexbuf; csslang_token lexbuf } (* update position *)
| url as s { URI(s) }
| "em" as s { EM(s) }
| "ex" as s { EX(s) }
| "px" as s { PX(s) }
| "cm" as s { CM(s) }
| "mm" as s { MM(s) }
| "in" as s { IN(s) }
| "pt" as s { PT(s) }
| "pc" as s { PC(s) }
| hex_color as s { HEXCOLOR(s) }
| "$"      { DOLLAR }
| "/*"          { comment lexbuf }           (* Comments *)
| "(" as s { LPAREN(Char.escaped s) }
| ")" as s { RPAREN(Char.escaped s) }
| "{"       { LBRACE }
| "}"       { RBRACE }
| "[" as s { LBRACKET(Char.escaped s) }
| "]" as s { RBRACKET(Char.escaped s) }
| "~=" as s { INCLUDES(s) }
| "|=" as s { DASHMATCH(s) }
| ";"       { SEMI }
| "+" as s { PLUS(Char.escaped s) }
| "-" as s { MINUS(Char.escaped s) }
| "*" as s { TIMES(Char.escaped s) }
| "/" as s { DIVIDE(Char.escaped s) }
| "." as s { PERIOD(Char.escaped s); }
| ":" as s { COLON(Char.escaped s) }
| '=' as s { ASSIGN(Char.escaped s) }
| ','      { COMMA }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| ident as s { ID(s) }
| '@' { ATRULE }
| "#" name as s      { HASHID(s) }
| single_quote_str as s { SSTRING(s) }
| double_quote_str as s { DSTRING(s) }
| num as s  { NUMBER(s) }
| '%' as s { PERCENT(Char.escaped s) }

| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }
| eof { EOF }
and comment = parse
  "*/" { csslang_token lexbuf }
| _    { comment lexbuf }

