(* 
interpret.ml
-----------
This file implements the interpreter that walks the ast and evaluates statements and expressions
*)


open Ast
open Printf

module NameMap = Map.Make(struct
  type t = string
  let compare x y = Pervasives.compare x y
end)

let match_type a b = 
    if (a.bv_type != b.bv_type) then
        raise (Failure ("Operand type mismatch"));;

let check_num a b =
    if (a.bv_type != TYPE_NUM || b.bv_type != TYPE_NUM) then
        raise (Failure ("Operand is not a number"));;
    
let match_num a b = 
    (match_type a b);
    (check_num a b)

let convert_unit ufrom uto =
    match ufrom,uto with
        "in", "cm" -> 2.54
        | "in", "mm" -> 25.4
        | "in", "pt" -> 72.0
        | "in", "pc" -> 6.0
        | "cm", "in" -> 0.3937
        | "cm", "mm" -> 10.0
        | "cm", "pt" -> 28.3465
        | "cm", "pc" -> 2.3622
        | "mm", "cm" -> 0.1
        | "mm", "in" -> 0.03937
        | "mm", "pt" -> 2.8347
        | "mm", "pc" -> 0.23622
        | "pt", "in" -> 0.01388
        | "pt", "cm" -> 0.03527
        | "pt", "mm" -> 0.35277
        | "pt", "pc" -> 0.08333
        | "pc", "in" -> 0.16667
        | "pc", "cm" -> 0.42333
        | "pc", "mm" -> 4.23333
        | "pc", "pt" -> 12.0
        | "in", "in" -> 1.0
        | "cm", "cm" -> 1.0
        | "mm", "mm" -> 1.0
        | "pt", "pt" -> 1.0
        | "pc", "pc" -> 1.0
        | _, _ -> raise (Failure ("Can not convert units"))


let normalize_unit a b =
    (* check unit type *)
    match a.bv_unit, b.bv_unit with
        (* convert b to a's unit type *)
        Abslen(i), Abslen(j) ->
            let new_b = { b with bv_fval = (convert_unit j i) *. b.bv_fval} in
            a,new_b
        | Rellen(i), Rellen(j) -> 
            if i = j then a, b
            else raise (Failure ("Invalid operation on different relative unit types"))
        | Nounit, Nounit -> a,b
        | _, _ -> raise (Failure ("Incompatible operand types"))
    

let string_of_unit u = 
    match u with
        Abslen(i) -> i
        | Rellen(i) -> i
        | Nounit -> "";;
        
let trim c str =
    let n = String.length str in
    let str =
        if str.[n-1] = c then String.sub str 0 (n-1)
        else str in
    str
        
(* Evaluate an expression and return (value, updated environment) *)
let rec eval env = function
    Literal(i) -> i, env
    | Id(var) ->
        let locals, globals = env in
        if NameMap.mem var locals then
            (NameMap.find var locals), env
        else if NameMap.mem var globals then
            (NameMap.find var globals), env
        else raise (Failure ("undeclared identifier " ^ var))
    | Binop(e1, op, e2) ->
        let v1, env = eval env e1 in
        let v2, env = eval env e2 in
        (* let boolean i = if i then 1 else 0 in *)
        (match op with
            Add -> 
                let (v1, v2) = (normalize_unit v1 v2) in
                {
                    bv_fval = v1.bv_fval +. v2.bv_fval;
                    bv_type = TYPE_NUM;
                    bv_unit = v1.bv_unit;
                    bv_sval = "";
                }
            | Sub ->
                let (v1, v2) = (normalize_unit v1 v2) in
                {
                    bv_fval = v1.bv_fval -. v2.bv_fval;
                    bv_type = TYPE_NUM;
                    bv_unit = v1.bv_unit;
                    bv_sval = "";
                }
            | Mult ->
                let (v1, v2) = (normalize_unit v1 v2) in
                {
                    bv_fval = v1.bv_fval *. v2.bv_fval;
                    bv_type = TYPE_NUM;
                    bv_unit = v1.bv_unit;
                    bv_sval = "";
                }
            | Div ->
                let (v1, v2) = (normalize_unit v1 v2) in
                {
                    bv_fval = v1.bv_fval /. v2.bv_fval;
                    bv_type = TYPE_NUM;
                    bv_unit = v1.bv_unit;
                    bv_sval = "";
                }), env

            (*
            | Equal -> boolean (v1 = v2)
            | Neq -> boolean (v1 != v2)
            | Less -> boolean (v1 < v2)
            | Leq -> boolean (v1 <= v2)
            | Greater -> boolean (v1 > v2)
            | Geq -> boolean (v1 >= v2)), env
            | Assign(var, e) ->
                let v, (locals, globals) = eval env e in
                    if NameMap.mem var locals then
                    v, (NameMap.add var v locals, globals)
                    else if NameMap.mem var globals then
                    v, (locals, NameMap.add var v globals)
                    else raise (Failure ("undeclared identifier " ^ var))
            *)
    | Id(id) -> 
        {
            bv_fval = 0.0;
            bv_type = TYPE_NUM;
            bv_unit = Nounit;
            bv_sval = "";
        }, env
    | Assign(s, e) ->
        {
            bv_fval = 0.0;
            bv_type = TYPE_NUM;
            bv_unit = Nounit;
            bv_sval = "";
        }, env


let rec string_of_bval a env = 
    match a.bv_type with
        TYPE_NUM -> (trim '.' (string_of_float a.bv_fval)) ^ (string_of_unit a.bv_unit), env
        | TYPE_SQUOTE_STR -> a.bv_sval, env
        | TYPE_DQUOTE_STR -> a.bv_sval, env
        | TYPE_NOQUOTE_STR -> a.bv_sval, env
        | TYPE_VAR -> 
            let x = Id(a.bv_sval) in
            let (bval, env) = eval env x in
            string_of_bval bval env
            

let run (var_decls, ruleset_decls, def_decls) = 

    (* helper function to populate a map with a var_decl list *)
    let load_vars_to_map env var_decls =
        let (locals, globals) = env in
        let get_eval env bvalue =
            let (v, _) = eval env bvalue in v 
        in
        let varmap = 
        List.fold_left (fun varmap decl -> NameMap.add decl.vdecl_key (get_eval (locals,globals) decl.vdecl_val) varmap) NameMap.empty var_decls
        in varmap
    in

    (* Populate global symbol table *)
    let globals = load_vars_to_map (NameMap.empty, NameMap.empty) var_decls in 

    (* debug: dump symbol table *)
    (*
    NameMap.fold (fun k v a -> printf "Symbols: key=%s val=%s\n" k (string_of_bval v (NameMap.empty, globals))) globals ();
    *)

    let includes = List.fold_left
        (fun def_decls def -> NameMap.add def.fname def def_decls)
        NameMap.empty def_decls
    in

    (* debug: dump def ruleset table *)
    (*
    NameMap.fold (fun k v a -> printf "Includes: key=%s\n" k) includes ();
    *)


    (*
    @return string, (NameMap, NameMap)
    *)
    let rec string_of_stmt stmt env =
        (* debug 
        NameMap.fold (fun k v a -> printf "locals: key=%s val=%s\n" k (string_of_bval v (NameMap.empty, globals))) (fst env) ();
        *)
    
        match (stmt) with
        PropDecl(vname, expr) ->
            let (bval, env) = (eval env expr) in
            let (str, env) = (string_of_bval bval env) in
            "\t" ^ vname ^ ": " ^ str ^ ";\n", env
        | IncludeDef(dname, actuals) -> 
            let inc = 
                try NameMap.find dname includes
                with Not_found -> raise (Failure ("undefined template " ^ dname))
            in
            (* evaluate the parameters passed in *)
            let actuals, env = List.fold_left
                (fun (actuals, env) actual ->
                let v, env = eval env actual in v :: actuals, env)
                ([], env) actuals
            in
            let actuals = List.rev actuals in

            (* debug: print actuals *)
            (*
            List.iter (fun s -> printf "actual=%s\n" (string_of_bval s env)) actuals;
            *)

            (* bind actual values to formal arguments *)
            let tpl_locals =
                try List.fold_left2
                    (fun locals formal actual -> NameMap.add formal actual locals)
                    NameMap.empty inc.formals actuals
                with Invalid_argument(_) ->
                raise (Failure ("wrong number of arguments passed to " ^ dname))
            in

            (* debug 
            NameMap.fold (fun k v a -> printf "bounded local key=%s val=%s\n" k (string_of_bval v env)) locals ();
            *)

            (* evaluate statements in  template definition *)
            let tpl_env = (tpl_locals, snd env) in
            let str_list, _ = List.fold_left
                (fun (str_list, env) stmt ->
                    let s, env = string_of_stmt stmt env in s :: str_list, env)
                ([], tpl_env) inc.body
            in
            let str = String.concat "" (List.rev str_list) in
            str, env

        | VarDecl(vdecl) -> 
            let varname = vdecl.vdecl_key in
            let varval = vdecl.vdecl_val in
            let (bval , env) = eval env varval in

            (* add the var decl to the locals *) 
            let (locals, globals) = env in

            (*
            NameMap.fold (fun k v a -> printf "bounded local key=%s val=%s\n" k (string_of_bval v env)) locals ();
            *)
            let locals = NameMap.add varname bval locals in
            "", (locals, globals)
    in

    (* @param decl: stmt list *)
    let string_of_declaration stmt_list env =
        let str_list, env = List.fold_left 
            (fun (str_list, env) stmt -> 
                let s, env = string_of_stmt stmt env in s :: str_list, env)
            ([], env) stmt_list
        in
        String.concat "" (List.rev str_list)
        (*
        List.fold_left (fun s stmt -> s ^ (string_of_stmt stmt env))
        *)
    in

    let string_of_ruleset ruleset =
        let locals = load_vars_to_map (NameMap.empty, globals) ruleset.rlocals in
        let env = (locals, globals) in
        (* debug
        printf "selector=%s\n" ruleset.selector;
        *)
        ruleset.selector ^ " {\n" ^
        (string_of_declaration ruleset.declaration env) ^ "}\n"
    in

    let string_of_program globals rulesets =
        String.concat "" (List.map string_of_ruleset (List.rev rulesets))
    in

    printf "%s" (string_of_program globals ruleset_decls)


