/*
 * This Java program is modified from
 * the Apache Software Foundation (ASF)'s Commons Math project.
 * Java document of the PolynormialFunction class at
 * http://commons.apache.org/math/api-2.2/index.html
 * Original source code at
 * http://commons.apache.org/math/download_math.cgi
 */
package poly;

import java.util.Arrays;

/**
 * Immutable representation of a real polynomial function with real coefficients.
 * <p>
 * <a href="http://mathworld.wolfram.com/HornersMethod.html">Horner's Method</a>
 *  is used to evaluate the function.</p>
 *
 * @version $Revision: 1042376 $ $Date: 2010-12-05 16:54:55 +0100 (dim. 05 déc. 2010) $
 */
public class PolynomialFunction {

    /**
     * The coefficients of the polynomial, ordered by degree -- i.e.,
     * coefficients[0] is the constant term and coefficients[n] is the
     * coefficient of x^n where n is the degree of the polynomial.
     */
   // private final double coefficients[];
      public double coefficients[];

    /**
     * Construct a polynomial with the given coefficients.  The first element
     * of the coefficients array is the constant term.  Higher degree
     * coefficients follow in sequence.  The degree of the resulting polynomial
     * is the index of the last non-null element of the array, or 0 if all elements
     * are null.
     * <p>
     * The constructor makes a copy of the input array and assigns the copy to
     * the coefficients property.</p>
     *
     * @param c polynomial coefficients
     * @throws NullPointerException if c is null
     * @throws NoDataException if c is empty
     */
    public PolynomialFunction(double c[]) {
        int n = c.length;
        // assume n > 0
        while ((n > 1) && (c[n - 1] == 0)) {
            --n;
        }
        this.coefficients = new double[n];
        System.arraycopy(c, 0, this.coefficients, 0, n);
    }

    // may not need this one!
    public PolynomialFunction(int degree) {
        int n = degree + 1;
        // assume n > 0
        this.coefficients = new double[n];
        for (int i=0; i < n; i++)
        {
            this.coefficients[i] = 0;
        }
    }
    /* shiftLeft corespond to the polynormail operator << in pcpl
       p = {1.0, 2.0,3.0}
       q = p.shiftLeft(1)
       q is {2.0, 3.0}
     */
    public PolynomialFunction shiftLeft(int i) throws Exception
    {
        if (i >= this.coefficients.length || i < 0)
        {
            throw new Exception("illegal shift left");
        }

        double[] nc = new double[this.coefficients.length - i];
        System.arraycopy(this.coefficients, i, nc, 0, nc.length);
        return new PolynomialFunction(nc);
    }

    /* shiftRight corespond to the polynormail operator >> in pcpl
       p = {1.0, 2.0,3.0}
       q = p.shiftRight(1)
       q is {0.0, 1.0, 2.0, 3.0}
     */
    public PolynomialFunction shiftRight(int i) throws Exception
    {
        if (i < 0)
        {
            throw new Exception("illegal shift right");
        }

        double[] nc = new double[this.coefficients.length + i];
        System.arraycopy(this.coefficients, 0, nc, i, this.coefficients.length);
        return new PolynomialFunction(nc);
    }

     /**
     *  Returns the degree of the polynomial
     *
     * @return the degree of the polynomial
     */
    public int degree() {
        return coefficients.length - 1;
    }

    /**
     * Returns a copy of the coefficients array.
     * <p>
     * Changes made to the returned copy will not affect the coefficients of
     * the polynomial.</p>
     *
     * @return  a fresh copy of the coefficients array
     */
    public double[] getCoefficients() {
        return coefficients.clone();
    }

     /**
     * Add a polynomial to the instance.
     * @param p polynomial to add
     * @return a new polynomial which is the sum of the instance and p
     */
    public PolynomialFunction add(final PolynomialFunction p) {

        // identify the lowest degree polynomial
        final int lowLength  = Math.min(coefficients.length, p.coefficients.length);
        final int highLength = Math.max(coefficients.length, p.coefficients.length);

        // build the coefficients array
        double[] newCoefficients = new double[highLength];
        for (int i = 0; i < lowLength; ++i) {
            newCoefficients[i] = coefficients[i] + p.coefficients[i];
        }
        System.arraycopy((coefficients.length < p.coefficients.length) ?
                         p.coefficients : coefficients,
                         lowLength,
                         newCoefficients, lowLength,
                         highLength - lowLength);

        return new PolynomialFunction(newCoefficients);

    }

    /**
     * Subtract a polynomial from the instance.
     * @param p polynomial to subtract
     * @return a new polynomial which is the difference the instance minus p
     */
    public PolynomialFunction subtract(final PolynomialFunction p) {

        // identify the lowest degree polynomial
        int lowLength  = Math.min(coefficients.length, p.coefficients.length);
        int highLength = Math.max(coefficients.length, p.coefficients.length);

        // build the coefficients array
        double[] newCoefficients = new double[highLength];
        for (int i = 0; i < lowLength; ++i) {
            newCoefficients[i] = coefficients[i] - p.coefficients[i];
        }
        if (coefficients.length < p.coefficients.length) {
            for (int i = lowLength; i < highLength; ++i) {
                newCoefficients[i] = -p.coefficients[i];
            }
        } else {
            System.arraycopy(coefficients, lowLength, newCoefficients, lowLength,
                             highLength - lowLength);
        }

        return new PolynomialFunction(newCoefficients);

    }

    /**
     * Negate the instance.
     * @return a new polynomial
     */
    public PolynomialFunction negate() {
        double[] newCoefficients = new double[coefficients.length];
        for (int i = 0; i < coefficients.length; ++i) {
            newCoefficients[i] = -coefficients[i];
        }
        return new PolynomialFunction(newCoefficients);
    }

    /**
     * Multiply the instance by a polynomial.
     * @param p polynomial to multiply by
     * @return a new polynomial
     */
    public PolynomialFunction multiply(final PolynomialFunction p) {

        double[] newCoefficients = new double[coefficients.length + p.coefficients.length - 1];

        for (int i = 0; i < newCoefficients.length; ++i) {
            newCoefficients[i] = 0.0;
            for (int j = Math.max(0, i + 1 - p.coefficients.length);
                 j < Math.min(coefficients.length, i + 1);
                 ++j) {
                newCoefficients[i] += coefficients[j] * p.coefficients[i-j];
            }
        }

        return new PolynomialFunction(newCoefficients);

    }

    /*divide the instance by a double.*/
    public PolynomialFunction divide(double x) throws Exception {
        if (x == 0)
        {
            throw new Exception("division by zero");
        }

        double[] newCoefficients = new double[coefficients.length];

        for (int i = 0; i < newCoefficients.length; ++i) {
            newCoefficients[i] = this.coefficients[i]/x;
        }

        return new PolynomialFunction(newCoefficients);
    }

    /* p1 != p2 */
    public boolean notEqual(final PolynomialFunction other) {
        return !this.equals(other);
    }    

     /** Returns a string representation of the polynomial.
     * <p>The representation is user oriented. Terms are displayed lowest
     * degrees first. The multiplications signs, coefficients equals to
     * one and null terms are not displayed (except if the polynomial is 0,
     * in which case the 0 constant term is displayed). Addition of terms
     * with negative coefficients are replaced by subtraction of terms
     * with positive coefficients except for the first displayed term
     * (i.e. we display <code>-3</code> for a constant negative polynomial,
     * but <code>1 - 3 x + x^2</code> if the negative coefficient is not
     * the first one displayed).</p>

     * @return a string representation of the polynomial

     */
    @Override
     public String toString() {

       StringBuilder s = new StringBuilder();
       if (coefficients[0] == 0.0) {
         if (coefficients.length == 1) {
           return "0";
         }
       } else {
         s.append(Double.toString(coefficients[0]));
       }

       for (int i = 1; i < coefficients.length; ++i) {

         if (coefficients[i] != 0) {

           if (s.length() > 0) {
             if (coefficients[i] < 0) {
               s.append(" - ");
             } else {
               s.append(" + ");
             }
           } else {
             if (coefficients[i] < 0) {
               s.append("-");
             }
           }

           double absAi = Math.abs(coefficients[i]);
           if ((absAi - 1) != 0) {
             s.append(Double.toString(absAi));
             s.append(' ');
           }

           s.append("x");
           if (i > 1) {
             s.append('^');
             s.append(Integer.toString(i));
           }
         }

       }

       return s.toString();

     }

    /** {@inheritDoc} */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + Arrays.hashCode(coefficients);
        return result;
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof PolynomialFunction))
            return false;
        PolynomialFunction other = (PolynomialFunction) obj;
        if (!Arrays.equals(coefficients, other.coefficients))
            return false;
        return true;
    }
}
