%{ 
  open Ast
  let parse_error s = Printf.ksprintf failwith "ERROR: %s" s
%}

%token SEMI LPAREN RPAREN LBRACKET RBRACKET LBRACE RBRACE COMMA COLON
%token PLUS MINUS TIMES DIVIDE CARET LSHIFT RSHIFT ASSIGN
%token EQ NEQ LT LEQ GT GEQ
%token AND OR NOT
%token RETURN IF ELSE FOR WHILE
%token INT FLOAT BOOLEAN STRING POLY
%token TRUE FALSE
%token <int> INTLITERAL
%token <float> FLOATLITERAL
%token <string> ID
%token <string> STRINGLITERAL
%token EOF

%nonassoc NOCOMMA
%nonassoc COMMA

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE

%start program
%type <Ast.program> program

%%

program:
   /* nothing */ { [], [] }
 | program vdecl { ($2 :: fst $1), snd $1 }
 | program fdecl { fst $1, ($2 :: snd $1) }

fdecl:
   type_decl ID LPAREN formals_opt RPAREN LBRACE vdecl_list stmt_list RBRACE
     { { rtype   = $1;
         fname   = $2;
	 formals = $4;
	 locals  = List.rev $7;
	 body    = $8
       }
     }

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    formal                   { [$1] }
  | formal_list COMMA formal { $3 :: $1 }

formal:
    type_decl ID
    { { vtype = $1;
        vname = $2;
    } }

vdecl_list:
    /* nothing */    { [] }
  | vdecl_list vdecl { $2 :: $1 }

type_decl:
    INT     { Int }
  | FLOAT   { Float }
  | BOOLEAN { Boolean }
  | POLY    { Poly }
  | STRING  { String }
  
vdecl:
    type_decl ID SEMI
    { { vtype = $1;
        vname = $2;
      }
    }

stmt_list:
    /* nothing */  { [] }
  | stmt stmt_list { $1 :: $2 }

stmt:
    expr SEMI                                                   { Expr($1) }
  | RETURN expr SEMI                                            { Return($2) }
  | LBRACE stmt_list RBRACE                                     { Block($2) }
  | IF LPAREN expr RPAREN stmt %prec NOELSE                     { If($3, $5, Block([])) }
  | IF LPAREN expr RPAREN stmt ELSE stmt                        { If($3, $5, $7) }
  | FOR LPAREN expr_opt SEMI expr_opt SEMI expr_opt RPAREN stmt { For($3, $5, $7, $9) }
  | WHILE LPAREN expr RPAREN stmt                               { While($3, $5) }
        
expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

/*
float_list:
    FLOATLITERAL %prec NOCOMMA    { [$1] }
  | FLOATLITERAL COMMA float_list { $1 :: $3 }
*/

lvalue:
    ID               		{ Id($1) }
  | ID LBRACKET expr RBRACKET   { PolyElmt($1,$3)}
  
expr:
    INTLITERAL       		    { IntLiteral($1) }
  | FLOATLITERAL     		    { FloatLiteral($1) }
  | STRINGLITERAL                   { StringLiteral($1) }
  | LBRACE actuals_list RBRACE      { PolyLiteral(List.rev $2) }
  | POLY LBRACE expr RBRACE         { PolyInit($3) }
  | lvalue                          { Lvalue($1) }
  | TRUE             		    { BooleanLiteral(true) }
  | FALSE            		    { BooleanLiteral(false) }
  | expr PLUS   expr 		    { Binop($1, Add,   $3) }
  | expr MINUS  expr 		    { Binop($1, Sub,   $3) }
  | expr TIMES  expr 		    { Binop($1, Mult,  $3) }
  | expr DIVIDE expr 		    { Binop($1, Div,   $3) }
  | expr LSHIFT expr                { Binop($1, Lshift, $3) }
  | expr RSHIFT expr                { Binop($1, Rshift, $3) }
  | expr EQ     expr 		    { Binop($1, Equal, $3) }
  | expr NEQ    expr 		    { Binop($1, Neq,   $3) }
  | expr LT     expr 		    { Binop($1, Less,  $3) }
  | expr LEQ    expr 		    { Binop($1, Leq,   $3) }
  | expr GT     expr 		    { Binop($1, Greater,  $3) }
  | expr GEQ    expr 		    { Binop($1, Geq,   $3) }
  | MINUS expr                      { Negate($2) }  
  | lvalue ASSIGN expr   	    { Assign($1, $3) }
  | ID LPAREN actuals_opt RPAREN    { Call($1, $3) }
  | LPAREN expr RPAREN 		    { $2 }

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
    expr                    { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }
