type op = Add | Sub | Mult | Div | Lshift | Rshift| Equal | Neq | Less | Leq | Greater | Geq
type datatype = Int | Float | Boolean | Poly | String | UnknownType

type lvalue =
    Id of string
  | PolyElmt of string * expr

and expr =
    IntLiteral of int
  | FloatLiteral of float
  | BooleanLiteral of bool
  | StringLiteral of string
  | PolyLiteral of expr list
  | PolyInit of expr
  | Lvalue of lvalue
  | Binop of expr * op * expr
  | Negate of expr
  | Assign of lvalue * expr
  | Call of string * expr list
  | Noexpr
  


type stmt =
    Block of stmt list
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt
  
type var_decl = {
    vtype: datatype;
    vname: string;    
}

type func_decl = {
    rtype : datatype;
    fname : string;
    formals : var_decl list;
    locals : var_decl list;
    body : stmt list;
  }

type program = var_decl list * func_decl list

let string_of_datatype dtype =
  match dtype with
    Int -> "int"
  | Float -> "float"
  | Boolean -> "boolean"
  | Poly -> "poly"
  | String -> "string"
  | UnknownType -> "unknownype"

let string_of_binop o =
  match o with
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Lshift -> "<<"
  | Rshift -> ">>"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="

let rec string_of_expr = function
    IntLiteral(l) -> string_of_int l
  | FloatLiteral(l) -> string_of_float l
  | BooleanLiteral(l) -> string_of_bool l
  | StringLiteral(l) -> l
  | PolyLiteral(l) -> "{ "^ String.concat ", " (List.map string_of_expr l)^" }"
  | PolyInit(l) -> "poly{ "^ string_of_expr l^" }"
  | Lvalue(lv) -> string_of_lvalue lv
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^
      string_of_binop o ^ " " ^
      string_of_expr e2
  | Negate(e) -> "-" ^ string_of_expr e
  | Assign(v, e) -> string_of_lvalue v ^ " = " ^ string_of_expr e
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr -> ""

and string_of_lvalue = function
    Id(s) -> s
  | PolyElmt(s, e) -> s ^ "[ " ^ string_of_expr e  ^ " ]"

let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
      string_of_expr e3  ^ ") " ^ string_of_stmt s
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s

let string_of_vdecl vdecl = (string_of_datatype vdecl.vtype) ^ " " ^ vdecl.vname ^ ";\n"

let string_of_formal formal =
  string_of_datatype formal.vtype ^ " " ^ formal.vname

let string_of_fdecl fdecl =
  string_of_datatype fdecl.rtype ^ " " ^ fdecl.fname ^ "(" ^ String.concat ", " (List.map string_of_formal fdecl.formals) ^ ")\n{\n" ^
  String.concat "" (List.map string_of_vdecl fdecl.locals) ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_program (vars, funcs) =
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl funcs)
  