#include <stdio.h>
#include <math.h>
#include "filter.h"

/* This program demonstrates the complex arithmetic and array
features of COAL.  Parameters for a simple RLC filter are passed
to a function which returns its complex frequency response for a
given interval.  Magnitude and phase data is generated and dumped.*/
int main()
{
    int i;
    coal_arr freqs;
    coal_arr resp;
    coal_arr m;
    coal_arr p;
    coal_num R;
    coal_num L;
    coal_num C;
    coal_num bf;
    coal_num num;
    coal_num step;

    /* these component values yield a band pass filter centered
              at approx. 100kHz */
    R = cl_num(1.0,0.0); /* 1 Ohm */
    L = cl_num(250.0e-6,0.0); /* 250 uH */
    C = cl_num(.01e-6,0.0); /* .01uF */
    bf = cl_num(90e3, 0.0); /* 90 kHz */
    step = cl_num(1e3, 0.0); /* 1kHz */
    num = cl_num(20.0, 0.0);
    
    freqs = get_freqs(bf, num, step);
    if (!cl_valid_arr(freqs))
    {
        printf("Error %d at get_freqs.\n", cl_last_err());
        return -1;
    }
    
    /* generate complex response */
    resp = bp_filter(R,L,C,freqs);
    if (!cl_valid_arr(resp))
    {
        printf("Error %d at bp_filter.\n", cl_last_err());
        return -1;
    }
    
    /* get and dump magnitude response */
    m = mag_resp(resp);
    if (!cl_valid_arr(m))
    {
        printf("Error %d at mag_resp.\n", cl_last_err());
        return -1;
    }
    
    printf("magnitude response\n");
    for (i=0; i<cl_len(freqs); ++i)
    {
        printf("%f %f\n"
          , cl_dbl_re(cl_get_elem(freqs,i))
          , cl_dbl_re(cl_get_elem(m,i)) );
    }
    printf("\n");
    
    /* get and dump phase response */
    p = phase_resp(resp);
    if (!cl_valid_arr(p))
    {
        printf("Error %d at phase_resp.\n", cl_last_err());
        return -1;
    }
    
    printf("phase response\n");
    for (i=0; i<cl_len(freqs); ++i)
    {
        printf("%f %f\n"
         , cl_dbl_re(cl_get_elem(freqs,i))
         , cl_dbl_re(cl_get_elem(p,i)));
    }
    printf("\n");
    
    cl_free();
    
    return 0;
}
