#include <stdio.h>
#include <math.h>
#include <assert.h>
#include "cbindings.h"

/* This program demonstrates the C API to COAL */
int main()
{
    double ext_re[5];
    double ext_cp[10];
    int i;

    coal_num x;
    coal_arr rnga;
    coal_arr cia;
    coal_arr cia_out_of_mem;
    coal_arr cea;
    coal_arr rea;
    
    /* initialize external arrays */
    for (i=0; i<5; ++i)
    {
        ext_re[i] = (double)i;
        ext_cp[2*i] = (double)i;    /* real */
        ext_cp[2*i+1] = -(double)i; /* imag */
    }
    
    /* cl_num */
    x = cl_num(17.0, -93.0);
    
    /* cl_dbl_re, cl_dbl_im */
    assert(cl_dbl_re(x)==17.0);
    assert(cl_dbl_im(x)==-93.0);
    
    /* cl_real_arr */
    rea = cl_real_arr(ext_re, 5);  /* external real array */
    
    /* cl_cplx_arr */
    cea = cl_cplx_arr(ext_cp, 10); /* external complex array - interleaved */
    
    rnga = rng_arr(); /* range array */
    cia = cplx_int_arr(); /* internally generated array */
    
    /* cl_len */
    assert(5==cl_len(rea));
    assert(5==cl_len(cea));
    assert(5==cl_len(rnga));
    assert(5==cl_len(cia));
    
    /* cl_last */
    assert(4==cl_last(rea));
    assert(4==cl_last(cea));
    assert(4==cl_last(rnga));
    assert(4==cl_last(cia));
    
    /* cl_put_elem & cl_get_elem */
    cl_put_elem(rea, 3, cl_num(89.0, -89.0));
    x = cl_get_elem(rea, 3);
    assert(89.0==cl_dbl_re(x));
    assert(0.0==cl_dbl_im(x));
    
    cl_put_elem(cea, 3, cl_num(27.0, -27.0));
    x = cl_get_elem(cea, 3);
    assert(27.0==cl_dbl_re(x));
    assert(-27.0==cl_dbl_im(x));
    
    x = cl_get_elem(rnga, 3);
    assert(3.0==cl_dbl_re(x));
    assert(0.0==cl_dbl_im(x));
    
    cl_put_elem(cia, 3, cl_num(21.0, -21.0));
    x = cl_get_elem(cia, 3);
    assert(21.0==cl_dbl_re(x));
    assert(-21.0==cl_dbl_im(x));
    
    /* CL_ERR_ILLEGAL_INDEX - range array */
    assert(!cl_valid_num(illegal_index1(rnga)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index2(rnga)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    
    /* CL_ERR_ILLEGAL_INDEX - complex internal array */
    assert(!cl_valid_num(illegal_index1(cia)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index2(cia)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index3(cia)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index4(cia)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    
    /* CL_ERR_ILLEGAL_INDEX - complex external array */
    assert(!cl_valid_num(illegal_index1(cea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index2(cea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index3(cea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index4(cea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
        
    /* CL_ERR_ILLEGAL_INDEX - real external array */
    assert(!cl_valid_num(illegal_index1(rea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index2(rea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index3(rea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    assert(!cl_valid_num(illegal_index4(rea)));
    assert(cl_last_err()==CL_ERR_ILLEGAL_INDEX);
    
    /* CL_ERR_ARRAY_IMMUTABLE */
    assert(!cl_valid_num(immutable(rnga)));
    assert(cl_last_err()==CL_ERR_ARRAY_IMMUTABLE);
    
    /* CL_ERR_OUT_OF_MEMORY - storesize set to 9 */
    cia_out_of_mem = cplx_int_arr();
    assert(!cl_valid_arr(cia_out_of_mem));
    assert(cl_last_err()==CL_ERR_OUT_OF_MEMORY);
    
    /* CL_ERR_STEP_IS_ZERO */
    assert(!cl_valid_arr(
        get_rng(cl_num(1.0,0.0)
              , cl_num(5.0,0.0)
              , cl_num(0.0,0.0))
        ));
    assert(cl_last_err()==CL_ERR_STEP_IS_ZERO);
    
    /* CL_ERR_STEP_WRONG_SIGN */
    assert(!cl_valid_arr(
        get_rng(cl_num(5.0,0.0)
              , cl_num(1.0,0.0)
              , cl_num(1.0,0.0))
        ));
    assert(cl_last_err()==CL_ERR_STEP_WRONG_SIGN);
    
    assert(!cl_valid_arr(
        get_rng(cl_num(1.0,0.0)
              , cl_num(5.0,0.0)
              , cl_num(-1.0,0.0))
        ));
    assert(cl_last_err()==CL_ERR_STEP_WRONG_SIGN);

    
    /* CL_ERR_BAD_ARRAY */
    cl_free();

    /* since we've free'd, cia should be invalid now as it was created during a
              different era */
    assert(!cl_valid_arr(cia));
    
    /* trying a reduce operation should cause an error */
    assert(!cl_valid_num(sum(cia)));
    assert(cl_last_err()==CL_ERR_BAD_ARRAY);
    
    /* reallocate */
    cia = cplx_int_arr();
    assert(cl_last_err()==0);
    assert(cl_valid_arr(cia));
    
    /* free memory again and repeat test... */
    cl_free();
    assert(!cl_valid_arr(cia));
    
    /* again, trying a reduce operation should cause an error */
    assert(!cl_valid_num(sum(cia)));
    assert(cl_last_err()==CL_ERR_BAD_ARRAY);
 
    /* reallocate */
    cia = cplx_int_arr();
    assert(cl_valid_arr(cia));
    
    cl_free();
    return 0;
}
