(* COMS W4115, COAL, Eliot Scull, CUID: C000056091 *)

open Coalopts
open Arg
open Printf
open Types

let debug = false
 
(* entry point *)
let _ =
    if debug then printf "start\n";

    (* declare built in functions so that user code may refer to them *)
    let builtins = 
    [
        ("re"       ,(Func(Num, [Num])));
        ("im"       ,(Func(Num, [Num])));
        ("conj"     ,(Func(Num, [Num])));
        ("mag"      ,(Func(Num, [Num])));
        ("phase"    ,(Func(Num, [Num])));
        ("sin"      ,(Func(Num, [Num])));
        ("cos"      ,(Func(Num, [Num])));
        ("tan"      ,(Func(Num, [Num])));
        ("sqrt"     ,(Func(Num, [Num])));
        ("exp"      ,(Func(Num, [Num])));
        ("distance" ,(Func(Num, [Num; Num])));
        ("len"      ,(Func(Num, [NumArr])));
        ("last"     ,(Func(Num, [NumArr])));
        ("not"      ,(Func(Num, [Num]))) 
    ]
    in 
    if debug then printf "add builtins\n";
    (* add builtins to global namespace *)
    List.iter (fun (n,t) -> Semantics.addbuiltin n t) builtins;
    
    (* process command line arguments *)
    let copts =
    
      if debug then printf "cmd line opts\n";
      let emoderef = ref Normal in
      let outfilerootref = ref "coalout" in
      let storesizeref = ref 128 in
      let sourcesref = ref [] in
        Arg.parse
         (* options defined *)
         [ 
           ("-sast"
           , Arg.Unit(fun () -> emoderef:= Sast)
           , "Dump semantically checked abstract syntax tree for each specified .coal file to standard out.  Inhibits generation of .c output.");
    	   
    	   ("-o"
    	   , Arg.String(fun name -> outfilerootref:=name )
    	   , "Specify root name of c output files.  If not specified, default is coalout, resulting in coalout.h/c being generated.");
           
    	   ("-storesize"
    	   , Arg.Int(fun size -> if ((size < 1)||(size > 65535)) 
                         then raise (Bad("Illegal size " ^ (string_of_int size)))
                         else storesizeref:=size)
    	   ,"Specify the total number of elements in the array store used for dynamic array creation.  Default is 128, minimum is 1, and maximum is 65535.");
           
           ("-trace"
           , Arg.Unit(fun () -> emoderef:= Trace)
           , "Generate C output with debug tracing.  Traces dumped to standard out via printf.  This option ignored if -sast specified after.")
         ]
         (* callback to build list of files to compile *)
         (fun f -> if (Filename.check_suffix f ".coal")
                   then sourcesref := f::!sourcesref
                   else raise (Arg.Bad(f)))
         (* message to user output when there's a usage problem *)
         "usage: coal [options] <file1>.coal <file2>.coal ...";

        {emode = !emoderef
        ; sources = List.rev !sourcesref
        ; storesize = !storesizeref
        ; outfileroot = !outfilerootref}

    in 

    let total_sast = 

        (* build a source file *)
        let build total_sast file = 
            let fin = open_in file in
            try
                (* analysis *)
                if debug then printf "build %s\n" file;
                let lexbuf = Lexing.from_channel fin in
                let ast = Parser.program Scanner.token lexbuf in
                let sast = Semantics.check ast in
                close_in fin;
                total_sast @ sast
            with e -> (close_in fin; raise e)
        in
        
        (* build all specified sources *)
        List.fold_left build [] copts.sources
        
    in
    
        if debug then printf "emit code\n";
        (* output generation *)
        match copts.emode with 
          Sast -> print_string (Printer.string_of_program total_sast)
        | Normal 
        | Trace ->  Cbackend.emit copts (List.map (fun pair->fst pair) builtins) total_sast
