(* COMS W4115, COAL, Eliot Scull, CUID: C000056091 *)

open Coalopts
open Types
open Sym
open Sast
open Printf

let debug = false

let rec to_c_type = function
    Num -> "coal_num"
  | NumArr -> "coal_arr"
  | Tbd -> "coal_num"
  | Func(_,_) -> "/*Func not supported*/"
  | Var({contents = inner}) -> to_c_type inner;;

let to_c_var_decl env id = 
    let t = findtyp env id in
    (to_c_type t) ^ " " ^ id;;

let to_c_proto fdef = 
    let beg = sprintf "%s %s(" (to_c_type (snd fdef.fbody)) fdef.fname in
    let args = List.map (to_c_var_decl fdef.flocals) fdef.fargs in
    beg ^ (String.concat ", " args) ^ ")";;
  
let string_of_op = function
    Ast.Add -> "_cl_add"
  | Ast.Sub -> "_cl_sub"
  | Ast.Mul -> "_cl_mul"
  | Ast.Div -> "_cl_div"
  | Ast.Pow -> "_cl_expon"
  | Ast.Eq  -> "_cl_eq"
  | Ast.Neq -> "_cl_neq"
  | Ast.Lt  -> "_cl_lt"
  | Ast.Lte -> "_cl_lte"
  | Ast.Gt  -> "_cl_gt"
  | Ast.Gte -> "_cl_gte";;
  
let emit_header_beg copts doth = 
    (* idempotency *)
    let ppdef = sprintf "_%s_h" copts.outfileroot in
    fprintf doth "#ifndef %s\n" ppdef;
    fprintf doth "#define %s\n\n" ppdef;;
    
let emit_header_end doth =
    fprintf doth "#endif\n";;

let emit_internals copts doth dotc = 
   
    (* .h *)
    fprintf doth "#define CL_ERR_ILLEGAL_INDEX 0x1\n";
    fprintf doth "#define CL_ERR_ARRAY_IMMUTABLE 0x2\n";
    fprintf doth "#define CL_ERR_OUT_OF_MEMORY 0x3\n";
    fprintf doth "#define CL_ERR_STEP_IS_ZERO 0x4\n";
    fprintf doth "#define CL_ERR_STEP_WRONG_SIGN 0x5\n";
    fprintf doth "#define CL_ERR_BAD_ARRAY 0x6\n\n";
    fprintf doth "typedef struct\n\
                  {\n\
                  \   double re;\n\
                  \   double im;\n\
                  } coal_num;\n\n" ;
    fprintf doth "typedef union \n\
                  {\n\
                  \   struct\n\
                  \   {\n\
                  \       unsigned int kind: 2;\n\
                  \       unsigned int era: 14;\n\
                  \       unsigned int num: 16;\n\
                  \       coal_num* parr;\n\
                  \   } cp;\n\
                  \   struct\n\
                  \   {\n\
                  \       unsigned int kind: 2;\n\
                  \       unsigned int reserved: 14;\n\
                  \       unsigned int num: 16;\n\
                  \       double* parr;\n\
                  \   } re;\n\
                  \   struct\n\
                  \   {\n\
                  \       unsigned int kind: 2;\n\
                  \       unsigned int reserved: 14;\n\
                  \       unsigned int num: 16;\n\
                  \       short start;\n\
                  \       short step;\n\
                  \   } rng;\n\
                  } coal_arr;\n\n" ;
    fprintf doth "extern void cl_free();\n";
    fprintf doth "extern int cl_last_err();\n";
    fprintf doth "#define cl_valid_num(x) (!isnan(x.re))\n";
    fprintf doth "#define cl_dbl_re(x) (x.re)\n";
    fprintf doth "#define cl_dbl_im(x) (x.im)\n";
    fprintf doth "extern int cl_valid_arr(coal_arr);\n";
    fprintf doth "extern coal_num cl_get_elem(coal_arr, int);\n";
    fprintf doth "extern coal_num cl_put_elem(coal_arr, int, coal_num);\n";
    fprintf doth "#define cl_len(x) (x.cp.num)\n";
    fprintf doth "#define cl_last(x) (x.cp.num-1)\n";
    fprintf doth "extern coal_num cl_num(double re, double im);\n";
    fprintf doth "extern coal_arr cl_real_arr(double* re, int size);\n";
    fprintf doth "extern coal_arr cl_cplx_arr(double* re, int size);\n";

    (* .c *)
    fprintf dotc "#include <math.h>\n";
    fprintf dotc "#include <setjmp.h>\n";
    if (copts.emode=Trace) then
        fprintf dotc "#include <stdio.h>\n";
    fprintf dotc "#include \"%s.h\"\n\n" copts.outfileroot;
    fprintf dotc "#define CL_STORE_SIZE %d\n" copts.storesize;
    fprintf dotc "#define CL_MAX_ARR_SIZE 65535\n";
    fprintf dotc "#define CL_ARR_CPX_INT 0\n";
    fprintf dotc "#define CL_ARR_CPX_EXT 1\n";
    fprintf dotc "#define CL_ARR_RE_EXT 2\n";
    fprintf dotc "#define CL_ARR_RNG 3\n\n";
    
    fprintf dotc "static coal_num _cl_zero = {0.0, 0.0};\n";
    fprintf dotc "static coal_num _cl_one = {1.0, 0.0};\n\n";
    
    (* _cl_handle_err needs forward declaration *)
    if (copts.emode=Trace) then fprintf dotc "char* _cl_err_to_str(int err);\n";
    
    (* error handler macro for client visible functions *)
    fprintf dotc 
"#define _cl_handle_err(ret)\\\n\
if (_cl_call_lev==0) \\\n\
{\\\n\
\ if ((_cl_last_err=setjmp(_cl_top))!=0)\\\n\
\ {\\\n";

    (* output any encountered run time error in trace mode *)
    if (copts.emode=Trace) then
        fprintf dotc 
"   printf(\"%s\", _cl_err_to_str(_cl_last_err));\\\n" "%s.\\n";

    fprintf dotc 
"\   _cl_call_lev = -1;\\\n\
\   return ret;\\\n\
\ }\\\n\
}\n\n";

    fprintf dotc "static int _cl_last_err = 0;\n";
    fprintf dotc "static int _cl_call_lev = -1;\n";
    fprintf dotc "static jmp_buf _cl_top;\n\n";
    fprintf dotc "static coal_num _cl_arr_store[CL_STORE_SIZE];\n";
    fprintf dotc "static int _cl_used = 0;\n";
    fprintf dotc "static int _cl_era = 0;\n\n";
    
    (* C optimizer should be able to remove this *)
    fprintf dotc "void _cl_compile_asserts()\n\
                 {\n\
                 #define COAL_CT_ASSERT(e) ((void)sizeof(char[1 - 2*!(e)]))\n\
                 COAL_CT_ASSERT(sizeof(double)==8u);\n\
                 COAL_CT_ASSERT(sizeof(unsigned int)==4u);\n\
                 }\n\n";
                 
    fprintf dotc "coal_num _cl_invalid_num()\n\
                  {\n\
                  \   unsigned int nan[4];\n\
                  \   nan[0]=0xFFFFFFFF;\n\
                  \   nan[1]=0xFFFFFFFF;\n\
                  \   nan[2]=0xFFFFFFFF;\n\
                  \   nan[3]=0xFFFFFFFF;\n\
                  \   return (*((coal_num*)(&nan)));\n\
                  }\n\n";
    fprintf dotc "coal_arr _cl_invalid_arr()\n\
                  {\n\
                  \   coal_arr bad;\n\
                  \   bad.cp.kind = 0;\n\
                  \   bad.cp.era = 0;\n\
                  \   bad.cp.num = 0;\n\
                  \   bad.cp.parr = 0;\n\
                  \   return bad;\n\
                  }\n\n";
    fprintf dotc "#define _cl_check_arr(ca) { if ((ca.cp.parr==0)||((ca.cp.kind==CL_ARR_CPX_INT) && (ca.cp.era!=_cl_era))) longjmp(_cl_top, CL_ERR_BAD_ARRAY); } \n\n";
    fprintf dotc "#define _cl_check_idx(idx,ca) { if ((idx<0)||(idx>=ca.cp.num)) longjmp(_cl_top, CL_ERR_ILLEGAL_INDEX); }\n\n";
    fprintf dotc "int _cl_is_zero(coal_num x)\n\
                  {\n\
                  \   return ((x.re==0.0) && (x.im==0.0));\n\
                  }\n\n";
    fprintf dotc "int _cl_is_true(coal_num x)\n\
                  {\n\
                  \   return ((x.re>=0.5)||(x.re<=-0.5));\n\
                  }\n\n";
    fprintf dotc "coal_num _cl_rect(double re, double im)\n\
                  {\n\
                  \   coal_num r;\n\
                  \   r.re = re; r.im = im;\n\
                  \   return r;\n\
                  }\n\n";
    fprintf dotc "coal_num _cl_polar(double m, double theta)\n\
                  {\n\
                  \   coal_num r;\n\
                  \   r.re = m * cos(theta);\n\
                  \   r.im = m * sin(theta);\n\
                  \   return r;\n\
                  }\n\n";
    fprintf dotc "double _cl_mag_dbl(coal_num x)\n\
                  {\n\
                  \   return sqrt(x.re*x.re + x.im*x.im);\n\
                  }\n\n";
    fprintf dotc "double _cl_phase_dbl(coal_num x)\n\
                  {\n\
                  \   return atan2(x.im, x.re);\n\
                  }\n\n";
    fprintf dotc "coal_num _cl_negate(coal_num x)\n\
                 {\n\
                 \   x.re = -x.re; x.im = -x.im;\n\
                 \   return x;\n\
                 }\n";
    fprintf dotc "coal_num _cl_add(coal_num a, coal_num b)\n\
                  {\n\
                  \   coal_num r;\n\
                  \   r.re = a.re + b.re;\n\
                  \   r.im = a.im + b.im;\n\
                  \   return r;\n\
                  }\n\n";
    fprintf dotc "coal_num _cl_sub(coal_num a, coal_num b)\n\
                  {\n\
                  \   coal_num r;\n\
                  \   r.re = a.re - b.re;\n\
                  \   r.im = a.im - b.im;\n\
                  \   return r;\n\
                  }\n\n";
    fprintf dotc "coal_num _cl_mul(coal_num a, coal_num b)\n\
                  {\n\
                  \   coal_num r;\n\
                  \   r.re = (a.re * b.re - a.im * b.im);\n\
                  \   r.im = (a.im * b.re + a.re * b.im);\n\
                  \   return r;\n\
                  }\n\n";
    fprintf dotc "coal_num _cl_div(coal_num a, coal_num b)\n\
                  {\n\
                  \   coal_num r;\n\
                  \   double denom;\n\n\
                  \   if (_cl_is_zero(a))\n\
                  \   {\n\
                  \       r.re = 0.0;\n\
                  \       r.im = 0.0;\n\
                  \       return r;\n\
                  \   }\n\n\
                  \   if (_cl_is_zero(b))\n\
                  \   {\n\
                  \       r.re = HUGE_VAL;\n\
                  \       r.im = HUGE_VAL;\n\
                  \       return r;\n\
                  \   }\n\n\
                  \   denom = (b.re * b.re + b.im * b.im);\n\n\
                  \   r.re = (a.re * b.re + a.im * b.im) / denom;\n\
                  \   r.im = (b.re * a.im - a.re * b.im) / denom;\n\
                  \   return r;\n\
                  }\n\n";
                  
    (* 
            x ^ p
            
            x = m * e^(i*theta)
            x = e ^ (ln m + i*theta)
            x ^ p = e ^ ((ln m + i*theta) * p)
            
            q = (ln m + i*theta) * p
            
            x ^ p = (e ^ re{q}) *(e ^ i*im{q})
            
            m ' = (e ^ re{q})
            theta' = im{q}
            
            x ^ p = m' * e ^ i*theta'
         *)
    fprintf dotc "coal_num _cl_expon(coal_num x, coal_num p)\n\
                 {\n\
                 \    coal_num r;\n\
                 \    double ln_m;\n\
                 \    double theta;\n\
                 \    coal_num q;\n\
                 \    double m_prime;\n\
                 \    double theta_prime;\n\n\
                 \    if (_cl_is_zero(x)) return x;\n\n\
                 \    if (_cl_is_zero(p)) return _cl_rect(1.0, 0.0);\n\n\
                 \    ln_m = log(_cl_mag_dbl(x));\n\
                 \    theta = _cl_phase_dbl(x);\n\
                 \    q = _cl_mul(_cl_rect(ln_m, theta), p);\n\n\
                 \    m_prime = exp(q.re);\n\
                 \    theta_prime = q.im;\n\n\
                 \    r = _cl_polar(m_prime, theta_prime);\n\n\
                 \    return r;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_eq(coal_num a, coal_num b)\n\
                 {\n\
                 \   return (a.re==b.re && a.im==b.im)?_cl_one:_cl_zero;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_neq(coal_num a, coal_num b)\n\
                 {\n\
                 \   return (a.re!=b.re || a.im!=b.im)?_cl_one:_cl_zero;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_lt(coal_num a, coal_num b)\n\
                 {\n\
                 \   return (a.re<b.re)?_cl_one:_cl_zero;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_lte(coal_num a, coal_num b)\n\
                 {\n\
                 \   return (a.re<=b.re)?_cl_one:_cl_zero;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_gt(coal_num a, coal_num b)\n\
                 {\n\
                 \   return (a.re>b.re)?_cl_one:_cl_zero;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_gte(coal_num a, coal_num b)\n\
                 {\n\
                 \   return (a.re>=b.re)?_cl_one:_cl_zero;\n\
                 }\n\n";

    fprintf dotc "#define _cl_ge2(ca, cnidx) cl_get_elem(ca, (int)cnidx.re)\n\n";
    fprintf dotc "#define _cl_pe2(ca, cnidx, val) cl_put_elem(ca,(int)cnidx.re,val)\n\n";
    fprintf dotc "coal_arr _cl_new_arr(int N)\n\
                 {\n\
                 \   coal_arr nca;\n\
                 \   if (N > (CL_STORE_SIZE - _cl_used)) longjmp(_cl_top, CL_ERR_OUT_OF_MEMORY);\n\
                 \   nca.cp.kind = CL_ARR_CPX_INT;\n\
                 \   nca.cp.era = _cl_era;\n\
                 \   nca.cp.num = N;\n\
                 \   nca.cp.parr = _cl_arr_store + _cl_used;\n\
                 \   _cl_used += N;\n\
                 \   return nca;\n\
                 }\n\n";
    fprintf dotc "coal_arr _cl_map(coal_num (*f)(coal_num), coal_arr ca)\n\
                 {\n\
                 \   int idx;\n\
                 \   coal_arr nca;\n\
                 \   _cl_check_arr(ca);\n\
                 \   nca = _cl_new_arr(cl_len(ca));\n\
                 \   for (idx=0; idx<cl_len(ca); ++idx)\n\
                 \   {\n\
                 \       nca.cp.parr[idx] = f(cl_get_elem(ca, idx));\n\
                 \   }\n\
                 \   return nca;\n\
                 }\n\n";
    fprintf dotc "coal_arr _cl_rng1(int start, int stop, int step)\n\
                 {\n\
                 \   coal_arr nca;\n\
                 \   int diff;\n\
                 \   if (step == 0) longjmp(_cl_top, CL_ERR_STEP_IS_ZERO);\n\
                 \   if (  ((start > stop) && (step > 0))\n\
                 \       ||((start < stop) && (step < 0)) ) longjmp(_cl_top, CL_ERR_STEP_WRONG_SIGN);\n\
                 \   nca.rng.kind = CL_ARR_RNG;\n\
                 \   nca.rng.num = ((stop - start)/step + 1);\n\
                 \   nca.rng.start = (short) start;\n\
                 \   nca.rng.step = (short) step;\n\
                 \   return nca;\n\
                 }\n\n";
    fprintf dotc "#define _cl_rng2(start, stop, step) _cl_rng1((int)start.re, (int)stop.re, (int)step.re)\n\n";
    fprintf dotc "coal_num _cl_reduce_num(coal_num (*f)(coal_num, coal_num), coal_num acc, coal_arr ca)\n\
                 {\n\
                 \    int idx;\n\
                 \    _cl_check_arr(ca);\n\
                 \    for (idx=0; idx<cl_len(ca); ++idx)\n\
                 \    {\n\
                 \        acc = f(acc, cl_get_elem(ca, idx));\n\
                 \    }\n\
                 \    return acc;\n\
                 }\n\n";
    fprintf dotc "coal_arr _cl_reduce_arr(coal_arr (*f)(coal_arr, coal_num), coal_arr acc, coal_arr ca)\n\
                 {\n\
                 \    int idx;\n\
                 \    _cl_check_arr(ca);\n\
                 \    _cl_check_arr(acc);\n\
                 \    for (idx=0; idx<cl_len(ca); ++idx)\n\
                 \    {\n\
                 \        acc = f(acc, cl_get_elem(ca, idx));\n\
                 \    }\n\
                 \    return acc;\n\
                 }\n\n";
                 
    (* builtins *)
    fprintf dotc "coal_num _cl_re(coal_num x)\n\
                 {\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_im(coal_num x)\n\
                 {\n\
                 \   x.re = x.im;\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_conj(coal_num x)\n\
                 {\n\
                 \   x.im = -x.im;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_mag(coal_num x)\n\
                 {\n\
                 \   x.re = _cl_mag_dbl(x);\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_phase(coal_num x)\n\
                 {\n\
                 \   x.re = _cl_phase_dbl(x);\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_sin(coal_num x)\n\
                 {\n\
                 \   x.re = sin(x.re);\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_cos(coal_num x)\n\
                 {\n\
                 \   x.re = cos(x.re);\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_tan(coal_num x)\n\
                 {\n\
                 \   x.re = tan(x.re);\n\
                 \   x.im = 0.0;\n\
                 \   return x;\n\
                 }\n\n";
    fprintf dotc "coal_num _cl_sqrt(coal_num x)\n\
                 {\n\
                 \   double m;\n\
                 \   double theta;\n\
                 \   coal_num r;\n\
                 \   m = sqrt(_cl_mag_dbl(x));\n\
                 \   theta = _cl_phase_dbl(x) / 2.0;\n\
                 \   r = _cl_polar(m, theta);\n\
                 \   return r;\n\
                 }\n\n";
    (*
            e ^ x 
            
            x = (a + i*b)

            e ^ (a + i*b)
            e^a * e^(i*b)
            e^a * (cos b + i*sin b)
          *)
    fprintf dotc "coal_num _cl_exp(coal_num x)\n\
                 {\n\
                 \   double exp_a;\n\
                 \   double b;\n\
                 \   coal_num r;\n\
                 \   if (_cl_is_zero(x)) return _cl_one;\n\
                 \   exp_a = exp(x.re);\n\
                 \   b = x.im;\n\
                 \   r = _cl_rect(exp_a*cos(b), exp_a*sin(b));\n\
                 \   return r;\n\
                 }\n\n";
                 
    fprintf dotc "coal_num _cl_distance(coal_num a, coal_num b)\n\
                 {\n\
                 \   return _cl_mag(_cl_sub(a,b));\n\
                 }\n\n";                 
    fprintf dotc "#define _cl_len(x) _cl_rect(cl_len(x),0.0)\n\n";
    fprintf dotc "#define _cl_last(x) _cl_rect(cl_last(x),0.0)\n\n";
    fprintf dotc "coal_num _cl_not(coal_num x)\n\
                 {\n\
                 \   return (fabs(x.re)<.5?_cl_one:_cl_zero);\n\
                 }\n\n";
                 
    (* externally visible helpers *)
    fprintf dotc "void cl_free()\n\
                  {\n\
                  \   _cl_used = 0;\n\
                  \   _cl_era = 0x3FFF&(_cl_era+1);\n\
                  }\n\n";
    fprintf dotc "int cl_last_err()\n\
                  {\n\
                  \   return _cl_last_err;\n\
                  }\n\n";
    fprintf dotc "int cl_valid_arr(coal_arr ca)\n\
                  {\n\
                  \   if (ca.cp.parr == 0)\n\
                  \   {\n\
                  \       return 0;\n\
                  \   }\n\
                  \   else if ((ca.cp.kind==CL_ARR_CPX_INT) && (ca.cp.era!=_cl_era))\n\
                  \   {\n\
                  \       return 0;\n\
                  \   }\n\
                  \   else\n\
                  \   {\n\
                  \       return 1;\n\
                  \   }\n\
                  }\n\n";
    fprintf dotc "coal_num cl_get_elem(coal_arr ca, int idx)\n\
                 {\n\
                 \   coal_num _ret;\n\
                 \   ++_cl_call_lev;\n\
                 \   _cl_handle_err(_cl_invalid_num());\n\
                 \   _cl_check_idx(idx,ca);\n\
                 \   switch (ca.cp.kind)\n\
                 \   {\n\
                 \       case(CL_ARR_CPX_INT):\n\
                 \       case(CL_ARR_CPX_EXT):\n\
                 \         _ret = ca.cp.parr[idx];\n\
                 \         break;\n\
                 \       case(CL_ARR_RE_EXT):\n\
                 \         _ret = _cl_rect(ca.re.parr[idx], 0.0);\n\
                 \         break;\n\
                 \       case(CL_ARR_RNG):\n\
                 \         _ret = _cl_rect((ca.rng.start + idx * ca.rng.step), 0.0);\n\
                 \         break;\n\
                 \       default:\n\
                 \         longjmp(_cl_top, CL_ERR_BAD_ARRAY);\n\
                 \   }\n\
                 \   --_cl_call_lev;\n\
                 \   return _ret;\n\
                 }\n\n";
    fprintf dotc "coal_num cl_put_elem(coal_arr ca, int idx, coal_num val)\n\
                 {\n\
                 \   ++_cl_call_lev;\n\
                 \   _cl_handle_err(_cl_invalid_num());\n\
                 \   _cl_check_idx(idx,ca);\n\
                 \   switch (ca.cp.kind)\n\
                 \   {\n\
                 \       case(CL_ARR_CPX_INT):\n\
                 \       case(CL_ARR_CPX_EXT):\n\
                 \         ca.cp.parr[idx] = val;\n\
                 \         break;\n\
                 \       case(CL_ARR_RE_EXT):\n\
                 \         ca.re.parr[idx] = val.re;\n\
                 \         break;\n\
                 \       case(CL_ARR_RNG):\n\
                 \         longjmp(_cl_top, CL_ERR_ARRAY_IMMUTABLE);\n\
                 \         break;\n\
                 \       default:\n\
                 \         longjmp(_cl_top, CL_ERR_BAD_ARRAY);\n\
                 \   }\n\
                 \   --_cl_call_lev;\n\
                 \   return val;\n\
                 }\n\n";
    fprintf dotc "coal_num cl_num(double re, double im)\n\
                 {\n\
                 \    return _cl_rect(re, im);\n\
                 }\n\n";
    fprintf dotc "coal_arr cl_real_arr(double* re, int size)\n\
                 {\n\
                 \    coal_arr r;\n\
                 \    if ((re == 0)||(size<1)||(size>CL_MAX_ARR_SIZE))\n\
                 \    {\n\
                 \        _cl_last_err = CL_ERR_BAD_ARRAY;\n\
                 \        r.re.parr = 0;\n\
                 \    }\n\
                 \    else\n\
                 \    {\n\
                 \        r.re.kind = CL_ARR_RE_EXT;\n\
                 \        r.re.num = size;\n\
                 \        r.re.parr = re;\n\
                 \    }\n\
                 \    return r;\n\
                 }\n\n";
    fprintf dotc "coal_arr cl_cplx_arr(double* re, int size)\n\
                 {\n\
                 \    coal_arr r;\n\
                 \    if ((re == 0)||(size<2)||(size>(2*CL_MAX_ARR_SIZE)))\n\
                 \    {\n\
                 \        _cl_last_err = CL_ERR_BAD_ARRAY;\n\
                 \        r.cp.parr = 0;\n\
                 \    }\n\
                 \    else\n\
                 \    {\n\
                 \        r.cp.kind = CL_ARR_CPX_EXT;\n\
                 \        r.cp.era = 0;\n\
                 \        r.cp.num = size/2;\n\
                 \        r.cp.parr = (coal_num*)re;\n\
                 \    }\n\
                 \    return r;\n\
                 }\n\n";
                 
    (* trace support *)
    if (copts.emode=Trace) then
        (fprintf dotc "void _cl_dump(int indent, char *msg)\n\
                      {\n\
                      \   printf(\"%s\", indent, \"\", msg);\n\
                      }\n\n" "%*.s%s\\n";
         fprintf dotc "void _cl_dump_num(int indent, coal_num x)\n\
                      {\n\
                      \   printf(\"%s\", indent, \"\", x.re, x.im);\n\
                      }\n\n" "%*.s(%f,%f)\\n";
         fprintf dotc "char* _cl_arr_kind_to_str(coal_arr ca)\n\
                      {\n\
                      \   switch(ca.cp.kind)\n\
                      \   {\n\
                      \       case(CL_ARR_CPX_INT): return \"%s\";\n\
                      \       case(CL_ARR_CPX_EXT): return \"%s\";\n\
                      \       case(CL_ARR_RE_EXT): return \"%s\";\n\
                      \       case(CL_ARR_RNG): return \"%s\";\n\
                      \       default: return \"UNKNOWN ARRAY KIND\";\n\
                      \   }\n\
                      }\n\n" "CL_ARR_CPX_INT" "CL_ARR_CPX_EXT" "CL_ARR_RE_EXT" "CL_ARR_RNG";
         fprintf dotc "char* _cl_err_to_str(int err)\n\
                      {\n\
                      \   switch(err)\n\
                      \   {\n\
                      \       case(CL_ERR_ILLEGAL_INDEX): return \"%s\";\n\
                      \       case(CL_ERR_ARRAY_IMMUTABLE): return \"%s\";\n\
                      \       case(CL_ERR_OUT_OF_MEMORY): return \"%s\";\n\
                      \       case(CL_ERR_STEP_IS_ZERO): return \"%s\";\n\
                      \       case(CL_ERR_STEP_WRONG_SIGN): return \"%s\";\n\
                      \       case(CL_ERR_BAD_ARRAY): return \"%s\";\n\
                      \       default: return \"UNKNOWN ERROR\";\n\
                      \   }\n\
                      }\n\n" "CL_ERR_ILLEGAL_INDEX" "CL_ERR_ARRAY_IMMUTABLE" "CL_ERR_OUT_OF_MEMORY" "CL_ERR_STEP_IS_ZERO" "CL_ERR_STEP_WRONG_SIGN" "CL_ERR_BAD_ARRAY";
         fprintf dotc "void _cl_dump_arr(int indent, coal_arr ca)\n\
                      {\n\
                      \   int idx;\n\
                      \   printf(\"%s\", indent, \"\", _cl_arr_kind_to_str(ca));\n\
                      \   if (ca.cp.kind!=CL_ARR_RNG)\n\
                      \   {\n\
                      \       for(idx=0; idx<cl_len(ca); ++idx)\n\
                      \       {\n\
                      \           _cl_dump_num(indent, cl_get_elem(ca, idx));\n\
                      \       }\n\
                      \   }\n\
                      \   else\n\
                      \   {\n\
                      \       printf(\"%s\", indent, \"\", ca.rng.num, ca.rng.start, ca.rng.step);\n\
                      \   }\n\
                      }\n\n" "%*.s%s\\n" "%*.snum=%d start=%d step=%d\\n")
              
let emit_func_def copts dotc builtins link_from link_to fdef = 

    let extract_int_lits selist =
        List.rev (
        List.fold_left (fun lits se ->
              match se with 
                Real(fps, true), _ -> fps::lits
              | _, _ -> lits
              ) [] selist
        )
    in 
    let rec emit_exp = function

      (* leaf expression nodes *)
        Real(fps, _), _ -> fprintf dotc "_cl_rect(%s, 0.0)" fps
        
      | Imag(fps), _ -> fprintf dotc "_cl_rect(0.0, %s)" fps
      
      | Id(name), _ ->
        let parent_env = findenv fdef.flocals name in
        if (parent_env.context = fdef.fname) (* local variable? *)
            then fprintf dotc "%s" name (* locally defined *)
            else fprintf dotc "*p%s" name (* imported via static link *)
            
      | Negate(se), _ ->
        fprintf dotc "_cl_negate(\n";
        emit_exp se;
        fprintf dotc ")\n"
        
      | Binop(se1, op, se2), _ -> 
        fprintf dotc "%s(" (string_of_op op);
        emit_exp se1;
        fprintf dotc ",\n";
        emit_exp se2;
        fprintf dotc ")\n"
      
      | Assign(id, se), _ ->
        fprintf dotc "%s=" id;
        emit_exp se
        
      | GetElem(se1, se2), _ ->
        (match (extract_int_lits [se2]) with
          [l2] ->
            fprintf dotc "cl_get_elem(";
            emit_exp se1;
            fprintf dotc ",\n%s)\n" l2
        | _ ->
            fprintf dotc "_cl_ge2("; 
            emit_exp se1;
            fprintf dotc ",\n";
            emit_exp se2;
            fprintf dotc ")\n")
     
      | PutElem(se1, se2, se3), _ ->
        (match (extract_int_lits [se2]) with
            [l2] ->
            fprintf dotc "cl_put_elem(";
            emit_exp se1;
            fprintf dotc ",\n%s,\n" l2
          | _ ->
            fprintf dotc "_cl_pe2("; 
            emit_exp se1;
            fprintf dotc ",\n";
            emit_exp se2;
            fprintf dotc ",\n");
        emit_exp se3;
        fprintf dotc ")\n"
        
      | Map(callee, se), _ ->
        fprintf dotc "_cl_map(";
        (match callee with
            Named(name) ->
            fprintf dotc "%s" (if (List.mem name builtins) then "_cl_"^name else name)
          | Lambda(fdef) -> 
            fprintf dotc "%s" fdef.fname);
        fprintf dotc ",\n";
        emit_exp se;
        fprintf dotc ")\n"

      | Range(se1, se2, se3), _ ->
        (match (extract_int_lits [se1;se2;se3]) with
            (* all int literals *)
            [l1; l2; l3] ->
              
            fprintf dotc "_cl_rng1(%s,%s,%s)\n" l1 l2 l3
              
              (* 2 or fewer int literals *)
         |  _ ->
            
            fprintf dotc "_cl_rng2(";
            emit_exp se1;
            fprintf dotc ",\n";
            emit_exp se2;
            fprintf dotc ",\n";
            emit_exp se3;
            fprintf dotc ")\n"
        )
        
      | Invoke(callee, actuals), _ ->
        (match callee with
            Named(name) ->
            fprintf dotc "%s" (if (List.mem name builtins) then "_cl_"^name else name)
          | Lambda(fdef) -> 
            fprintf dotc "%s" fdef.fname);
        fprintf dotc "(";
        (match actuals with
            [] -> ()
          | _  -> 
            (emit_exp (List.hd actuals);
            (List.iter (fun a -> fprintf dotc ", "; emit_exp a)(List.tl actuals))
                        ) );
        fprintf dotc ")\n"
        
      | Reduce(callee, se1, se2), _ ->
        let rec which_reduce = function
            Num -> fprintf dotc "_cl_reduce_num("
          | Tbd -> fprintf dotc "_cl_reduce_num("
          | NumArr -> fprintf dotc "_cl_reduce_arr("
          | Var({contents=inner})-> which_reduce inner
          | Func(_) -> fprintf dotc "/* Func not supported */"
        in 
        which_reduce (snd se1);
        (match callee with
            Named(name) ->
            fprintf dotc "%s,\n" (if (List.mem name builtins) then "_cl_"^name else name)
          | Lambda(fdef) -> 
            fprintf dotc "%s,\n" fdef.fname);
      
        emit_exp se1;
        fprintf dotc ",\n";
        emit_exp se2;
        fprintf dotc ")\n"
        
      | IfThenElse(se1, se2, se3), _ ->
        fprintf dotc "(_cl_is_true(";
        emit_exp se1;
        fprintf dotc ")?";
        emit_exp se2;
        fprintf dotc ":";
        emit_exp se3;
        fprintf dotc ")\n"
        
      | Sequence(se1, se2), _ ->
      
        let rec flatten_sequence selist se =
        match se with
           Sequence(se1, se2), _ -> flatten_sequence (se2::selist) se1
         | non_seq_se -> non_seq_se::selist in 

        let selist = flatten_sequence [se2] se1 in
        fprintf dotc "(";
        emit_exp (List.hd selist);
        (List.iter (fun a -> fprintf dotc ",\n "; emit_exp a) (List.tl selist));
        fprintf dotc ")"
        
    in
    
    if debug then printf "emitting %s\n" fdef.fname;
    
    (* definition prototype *)
    fprintf dotc "%s\n{\n" (to_c_proto fdef);
    
    (* emit locals *)
    List.iter
        (fun vn -> if (   (* not a function argument... *)
                          (not (List.mem vn fdef.fargs))
                          (* ... and a variable type *)
                       && (match baretyp (findtyp fdef.flocals vn) with
                           Num | NumArr -> true | _ -> false) )
                      then fprintf dotc "%s;\n" (to_c_var_decl fdef.flocals vn)
                    ) (symbol_names fdef.flocals);
                    
    (* temporary for return expression - see below *)
    fprintf dotc "%s _ret;\n" (to_c_type (snd fdef.fbody));
    
    (* emit link from variable *)
    List.iter
        (fun (ctxt, vn, idx) ->
            if (ctxt = fdef.fname)
                then fprintf dotc "_links[%d]=(void*)&%s;\n" idx vn
                ) link_from;
               
    (* emit link to variable *)
    List.iter
      (fun (ctxt, vn, idx) ->
         if (ctxt = fdef.fname)
             then let ctypename = to_c_type (findtyp fdef.flocals vn) 
             in
             fprintf dotc "%s* p%s=(%s*)_links[%d];\n" ctypename vn ctypename idx
             ) link_to;
    
    let rt = baretyp (snd fdef.fbody) in
    
    let is_named = (((String.length fdef.fname)<7)||(String.sub fdef.fname 0 7) <> "_lambda")
    in 
    
    (* error handling - named (top level) functions only *)
    if is_named then (
            (* increment current call level so that setjmp only called once at top *)
            fprintf dotc "++_cl_call_lev;\n";
            fprintf dotc "_cl_handle_err(%s);\n" 
                           (if (rt = NumArr)
                                then "_cl_invalid_arr()"
                                else "_cl_invalid_num()") );
    (* trace for debug *)
    if (copts.emode=Trace) then
        fprintf dotc "_cl_dump(_cl_call_lev, \"%s\");\n" (fdef.fname^" begin");
       
    (* emit body *)
    fprintf dotc "_ret = \n";
    emit_exp fdef.fbody;
    fprintf dotc ";\n";
    
    (* trace for debug *)
    if (copts.emode=Trace) then
       (fprintf dotc "_cl_dump(_cl_call_lev, \"%s\");\n" (fdef.fname^" end with");
        fprintf dotc "%s(_cl_call_lev, _ret);\n" (if (rt=NumArr) then "_cl_dump_arr" else "_cl_dump_num"));
    
    (* decrement call level indicating that we're going back up the stack *)
    if is_named then fprintf dotc "--_cl_call_lev;\n";
    fprintf dotc "return _ret;\n}\n\n";;

let emit_lambda_support o fdefs = 

    let link_exists env id links = 
        List.exists (fun (ctxt, vn, _) -> ctxt=env.context && vn=id) links
    in
    
    (* Extract static link info as well as lambda definitions. *)
    (* Returns: maximum static link index, map of variables to static link indexes, list of lamda definitions. *)
    let rec extract_lambda_info res env = function

        Id(name), _ -> 
        
        let (curr_idx, link_from, link_to, lfdefs) = res in
        
        let parent_env, t = find_env_typ env name in
        
        if debug then 
             printf "found %s in %s. def in %s\n w/ type %s.\n"
                name env.context parent_env.context (to_c_type t);
                
        (match (baretyp t) with
            (* we've come across a variable while descending SAST  *)
            (* figure out if we need to link from or to a variable across parent/child scopes  *)
            Num 
          | NumArr when (env.context <> parent_env.context) ->
            if debug then printf "link %s from %s to %s\n" name parent_env.context env.context;
            let curr_idx, link_from = 
            
              if (link_exists parent_env name link_from)
                  (* scope and var combo exists.  pass on link index and link_from list *)
                  then curr_idx, link_from
                  (* first time for this scope and var combo.  create new link index & pass on with updated link_from list *)
                  else curr_idx+1, (parent_env.context,name,curr_idx+1)::link_from
                            
            in 
            let link_to = 
            
              if (link_exists env name link_to)
                   (* have already come across this imported var from above scope *)
                   then link_to
                   (* first time to have come across imported var.  record and pass on *)
                   else (env.context,name,curr_idx)::link_to

            in
            (curr_idx, link_from, link_to, lfdefs)
                  
          | _ -> res (* nothing to do *)
        )
            
      | Negate(se), _ ->
        extract_lambda_info res env se
            
      | Binop(se1, _, se2), _ -> 
        let res = extract_lambda_info res env se1 in
        extract_lambda_info res env se2
                
      | Assign(_, se), _ ->
        extract_lambda_info res env se
        
      | GetElem(se1, se2), _ ->
        let res = extract_lambda_info res env se1 in
        extract_lambda_info res env se2
                
      | PutElem(se1, se2, se3), _ ->
        let res = extract_lambda_info res env se1 in
        let res = extract_lambda_info res env se2 in
        extract_lambda_info res env se3
        
      | Map(callee, se), _ ->
        let res = descend_lambda res callee in
        extract_lambda_info res env se
        
      | Range(se1, se2, se3), _ ->
        let res = extract_lambda_info res env se1 in
        let res = extract_lambda_info res env se2 in
        extract_lambda_info res env se3
    
      | Reduce(callee, se1, se2), _ -> 
        let res = descend_lambda res callee in
        let res = extract_lambda_info res env se1 in
        extract_lambda_info res env se2
                
      | IfThenElse(se1, se2, se3), _ -> 
        let res = extract_lambda_info res env se1 in
        let res = extract_lambda_info res env se2 in
        extract_lambda_info res env se3
        
      | Sequence(se1, se2), _ ->
        let res = extract_lambda_info res env se1 in
        extract_lambda_info res env se2
                
      | Invoke(callee, actuals), _ ->
        let res = descend_lambda res callee in
        List.fold_left (fun res se -> extract_lambda_info res env se) res actuals
        
      | _, _ -> res (* nothing to do *)
    
   (* Used for when we hit a lambda definition - recurse into definition. *)
   and descend_lambda res = function
        Named(_) -> res (* nothing to do *)
      | Lambda(fdef) ->
        let curr_idx, link_from, link_to, lfdefs = res in
        (* descend into lambda with new scope *)
        extract_lambda_info (curr_idx,link_from,link_to,(fdef::lfdefs)) fdef.flocals fdef.fbody        
        
   in
   (* collect static link info for lambdas by descending into all named functions *)
   (* max_idx - maximum index ever needed at one time by static link table *)
   (* link_from - which variables are exported to child lambda scopes *)
   (* link_to - which variables are imported from parent scopes *)
   (* lfdefs - all lambda definitions encountered *)
   let max_idx, link_from, link_to, lfdefs
        = List.fold_left
            (fun (curr_idx, link_from, link_to, lfdefs) fdef ->
                let curr_idx', link_from', link_to', lfdefs'
                  = extract_lambda_info (-1,[],[],[]) fdef.flocals fdef.fbody
                in
                (max curr_idx' curr_idx,
                 (List.rev link_from') @ link_from,
                 (List.rev link_to') @ link_to,
                 lfdefs' @ lfdefs)
                )
            (-1,[],[],[]) fdefs
   in
   
   (* If required, declare static array used for establishing static links between parent and  *)
   (* subordinate scopes. *)
   if (max_idx>=0) then fprintf o "static void* _links[%d];\n\n" (max_idx + 1);
   
   (* output forward declarations for lambda functions so that named functions can reference them: *)
   (* lambda definitions will be placed after named function definitions *)
   List.iter (fun fdef -> fprintf o "static %s;\n" (to_c_proto fdef)) lfdefs;
   
   fprintf o "\n";
   
   (* return the static link index map and the Lambda definitions *)
   (link_from, link_to, lfdefs)

let emit copts builtins fdefs = 

    let hdr, impl = (copts.outfileroot ^ ".h"), (copts.outfileroot ^ ".c") in
    let doth, dotc = (open_out hdr), (open_out impl) in
       
    (* header declarations and internals *)
    emit_header_beg copts doth;
    emit_internals copts doth dotc;
        
    (* export user functions *)
    List.iter (fun fdef ->
               fprintf doth "extern %s;\n" (to_c_proto fdef)
               ) fdefs;
    (* end of header *)
    emit_header_end doth;
        
    (* setup static linkage between lambdas and parent functions *)
    let link_from, link_to, lfdefs = emit_lambda_support dotc fdefs in 
        
    (* emit function definitions *)
    List.iter (emit_func_def copts dotc builtins link_from link_to) fdefs;
    List.iter (emit_func_def copts dotc builtins link_from link_to) lfdefs;
        
    close_out doth;
    close_out dotc
