{ open Parser }

rule token = parse
  [ ' ' '\t' '\n' ] { token lexbuf }
| '#'       { comment lexbuf }
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| ';'      { SEMI }
| ','      { COMMA }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '"'      { string_lit "" lexbuf }
| '='      { ASSIGN }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&&"     { AND }
| "||"     { OR }
| "!"      { NOT }
| "bool"   { BOOL }
| "draw"   { DRAW }
| "else"   { ELSE }
| "end"    { END }
| "exit"   { EXIT }
| "false"  { FALSE }
| "float"  { FLOAT }
| "has"    { HAS }
| "if"     { IF }
| "int"    { INT }
| "local"  { LOCAL }
| "none"   { NONE }
| "print"  { PRINT }
| "read"   { READ }
| "rule"   { RULE }
| "string" { STRING }
| "true"   { TRUE }
| "while"  { WHILE }
| "to_string" { TO_STRING }
| "to_int" { TO_INT }
| "to_float" { TO_FLOAT }
| [ '0'-'9' ]+ as value { INTLIT(int_of_string(value)) }
| [ '0'-'9' ]+'.'[ '0'-'9' ]* as value { FLOATLIT(float_of_string(value)) }
| [ '0'-'9' ]*'.'[ '0'-'9' ]+ as value { FLOATLIT(float_of_string(value)) }
| [ 'a'-'z' 'A'-'Z'][ 'a'-'z' 'A'-'Z' '_' '0'-'9' ]* as value { ID(value) } 
| eof      { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and string_lit str = parse
  "\\"                       { escaped_char str lexbuf }
| "\""                       { STRINGLIT(str) }
| "\n"                      {raise (Failure("multi-line string literals not
allowed, input character #" ^ (string_of_int (Lexing.lexeme_start lexbuf)))) }
|  [^ '\\' '\n' '"' ]+ as part    { string_lit  (str ^ part) lexbuf } 

and escaped_char str = parse
  [ '"' ]     { string_lit (str ^ "\"") lexbuf }
| [ 'n' ]     { string_lit (str ^ "\n") lexbuf }
| [ 'r' ]     { string_lit (str ^ "\r") lexbuf }
| [ 't' ]     { string_lit (str ^ "\t") lexbuf }
| eof          { raise (Failure("Unterminated string literal")) }
| _ as char    { raise (Failure("Invalid escape character \"\\" ^ Char.escaped char)) }

and comment = parse
  '\n'  { token lexbuf }
| _     { comment lexbuf }


