open Ast

type exvals = (* expression value *)
    VInt of int
  | VFloat of float
  | VString of string
  | VBool of bool
  | VNothing

let exval_to_string ex = match (ex) with
  VInt ex -> string_of_int ex
| VFloat ex -> string_of_float ex
| VString ex -> ex
| VBool ex -> if ex = true then "true" else "false"
| VNothing -> "nothing";;

module NameMap = Map.Make(struct
  type t = string
  let compare x y = Pervasives.compare x y
end)



type exval = { v: exvals ; t : t }


(* exception ReturnException of int * int NameMap.t *)
exception ReturnException of exvals * exvals NameMap.t

(* Main entry point: run a program *)

let run (program) =
  let (comps,globals,rules) = program
  in
 
  (* Put rule declarations in a symbol table *)
  let rule_decls = List.fold_left
      (fun rules rdecl ->
          if NameMap.mem rdecl.rname rules then
              raise(Failure("Error: multiple declarations for rule " ^
              rdecl.rname))
          else
              NameMap.add rdecl.rname rdecl rules
      ) NameMap.empty rules
              
  in

  (* Invoke a function and return an updated global symbol table *)
  let rec call rdecl globals =

    (* Evaluate an expression and return (value, updated environment) *)
    let rec eval env = function
       IntLit(l) -> VInt l , env
      | BoolLit(l) -> VBool l, env
      | StringLit(l) -> VString l, env
      | FloatLit(l) -> VFloat l, env
      | Noexpr -> VNothing, env (* must be non-zero for the for loop predicate *)
      | Id(var) ->
	  let locals, globals = env in
	  if NameMap.mem var locals then
	    (NameMap.find var locals), env
	  else if NameMap.mem var globals then
	    (NameMap.find var globals), env
	  else raise (Failure ("undeclared identifier " ^ var))
      | LazyBinop(e1, op, e2) ->
	  let v1, env = eval env e1 in
          (match op with
            And -> (match v1 with
                VBool(true) ->
                    let v2, env = eval env e2 in
                    (match v2 with
                        VBool(true)  -> VBool(true), env
                       | VBool(false) -> VBool(false), env
                       | _ -> raise (Failure("Only bool expressions can be used with &&
                        operator")) )
              | VBool(false) -> VBool(false), env
              | _ -> raise (Failure("Only bool expressions can be used with &&
              operator")) )

          |  Or -> (match v1 with
            VBool(false) ->
                let v2, env = eval env e2 in
                (match v2 with
                    VBool(true)  -> VBool(true), env
                   | VBool(false) -> VBool(false), env
                   | _ -> raise (Failure("Only bool expressions can be used with &&
                    operator")) )
              | VBool(true) -> VBool(true), env
              | _ -> raise (Failure("Only bool expressions can be used with &&
              operator")) )
          )
      | Binop(e1, op, e2) ->
	  let v1, env = eval env e1 in
          let v2, env = eval env e2 in
	  (match op with
	    Add -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VInt (v1 + v2)
              | (VFloat v1, VFloat v2) -> VFloat (v1 +. v2)
              | (VString v1, VString v2) -> VString (v1 ^ v2)
              | (VBool _ , _) -> raise (Failure ("Invalid argument to + operator"))
              | (_,_) -> raise (Failure("Operands to + operator don't have matching
    types")))

	  | Sub -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VInt (v1 - v2)
              | (VFloat v1, VFloat v2) -> VFloat (v1 -. v2)
              | ( (VInt _ | VFloat _), _) -> raise (Failure("Operands to - operator don't have matching
    types"))
              | (_,_) -> raise (Failure ("Invalid argument to - operator")))

	  | Mult -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VInt (v1 * v2)
              | (VFloat v1, VFloat v2) -> VFloat (v1 *. v2)
              | ( (VInt _ | VFloat _), _) -> raise (Failure("Operands to * operator don't have matching
    types"))
              | (_,_) -> raise (Failure ("Invalid argument to * operator")))

	  | Div -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VInt (v1 / v2)
              | (VFloat v1, VFloat v2) -> VFloat (v1 /. v2)
              | ( (VInt _ | VFloat _), _) -> raise (Failure("Operands to * operator don't have matching
    types"))
              | (_,_) -> raise (Failure ("Invalid argument to * operator")))

	  | Equal -> (match(v1,v2) with
                (VInt v1, VInt v2) -> VBool (v1 = v2)
              | (VFloat v1, VFloat v2) -> VBool (v1 = v2)
              | (VBool v1, VBool v2) -> VBool (v1 = v2)
              | (VString v1, VString v2) -> VBool (v1 = v2)
              | (_,_) -> raise (Failure ("Trying to compare values of different
              types")))

	  | Neq -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VBool (v1 <> v2)
              | (VFloat v1, VFloat v2) -> VBool (v1 <> v2)
              | (VBool v1, VBool v2) -> VBool (v1 <> v2)
              | (VString v1, VString v2) -> VBool (v1 <> v2)
              | (_,_) -> raise (Failure ("Trying to compare values of different
              types")))

	  | Less -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VBool (v1 < v2)
              | (VFloat v1, VFloat v2) -> VBool (v1 < v2)
              | ( (VString _ | VBool _), _) -> raise (Failure("Invalid type for
              comparison"))
              | (_,_) -> raise (Failure ("Trying to compare values of different
              types")))

	  | Leq -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VBool (v1 <= v2)
              | (VFloat v1, VFloat v2) -> VBool (v1 <= v2)
              | ( (VString _ | VBool _), _) -> raise (Failure("Invalid type for
              comparison"))
              | (_,_) -> raise (Failure ("Trying to compare values of different
              types")))

	  | Greater -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VBool (v1 > v2)
              | (VFloat v1, VFloat v2) -> VBool (v1 > v2)
              | ( (VString _ | VBool _), _) -> raise (Failure("Invalid type for
              comparison"))
              | (_,_) -> raise (Failure ("Trying to compare values of different
              types")))

	  | Geq -> (match (v1,v2) with
                (VInt v1, VInt v2) -> VBool (v1 >= v2)
              | (VFloat v1, VFloat v2) -> VBool (v1 >= v2)
              | ( (VString _ | VBool _), _) -> raise (Failure("Invalid type for
              comparison"))
              | (_,_) -> raise (Failure ("Trying to compare values of different
              types")))
        ), env
      | Negate(e) ->
            let v, (locals, globals) = eval env e in
            (match v with
            (VInt v1)    -> VInt(- v1)
          | (VFloat v1)  -> VFloat(-. v1)
          | _           -> raise(Failure("Attempting to negate non-numeric
          value"))
            ), env
      | Not(e) ->
            let v, (locals, globals) = eval env e in
            (match v with
            (VBool v1)    -> VBool(not v1)
          | _           -> raise(Failure("Trying to apply boolean ! to
          non-boolean value"))
            ), env
      | Assign(var, e, local) ->
          let do_assign (vname, exp, map) = (
              if (not (NameMap.mem vname map)) then  (* New declaration *)
                NameMap.add var exp map
              else
                (match (NameMap.find vname map, exp) with
            (VInt v1, VInt v2)          ->  NameMap.add var exp map
          | (VFloat v1, VFloat v2)      ->  NameMap.add var exp map
          | (VString v1, VString v2)    ->  NameMap.add var exp map
          | (VBool v1, VBool v2)        ->  NameMap.add var exp map
          | (VNothing, _)               ->  NameMap.add var exp map
          |  (_,_) -> raise (Failure ("Type error: attempting to assign
            incompatible types"))) (* end match *)
          ) (* end do_assign *)
          in
              
	  let v, (locals, globals) = eval env e in
          if local then
                if NameMap.mem var locals then
                    raise (Failure ("Tried to redeclare local variable"))
                else (* need to check type before assigning *)
                    (* v, (NameMap.add var v locals, globals) *)
                v, (do_assign (var, v, locals), globals)
	  else if NameMap.mem var locals then  (* Assign to local *)
                v, (do_assign (var, v, locals), globals)
          else if NameMap.mem var globals then   (* Assign to global *)
                v, (locals, do_assign(var, v, globals))
          else                             (* Declare new local *)
                v, (do_assign (var, v, locals), globals)
    | ToString(e) ->
            let v, env = eval env e in
            (match v with
            (VInt e) -> VString(string_of_int e), env
          | (VFloat e) -> VString(string_of_float e), env
          | (VBool e) -> VString(if e = true then "true" else "false"), env
          | (VString e) -> VString(e), env
          | _ -> raise (Failure("Type error: invalid type passed to to_string"))
            )
    | ToInt(e) ->
            let v, env = eval env e in
            (match v with
            (VInt e) -> VInt(e), env
          | (VFloat e) -> VInt(truncate e), env
          | (VString e) -> VInt(int_of_string e), env
          | _ -> raise (Failure("Type error: invalid type passed to to_int"))
            )
    | ToFloat(e) ->
            let v, env = eval env e in
            (match v with
            (VInt e) -> VFloat(float e), env
          | (VFloat e) -> VFloat(e), env
          | (VString e) -> VFloat(float_of_string e), env
          | _ -> raise (Failure("Type error: invalid type passed to to_float"))
            )
    | Read -> VString(input_line stdin), env
    | Call(_,_) -> raise(Failure("Call() not implemented"))
    | Draw      -> raise(Failure("Draw not implemented"))

    in

    (* Execute a statement and return an updated environment *)
    let rec exec env = function
	Block(stmts) -> List.fold_left exec env stmts
      | Expr(e) -> let _, env = eval env e in env
      | If(e, s1, s2) ->
	  let v, env = eval env e in
          (match v with
          VBool b -> exec env (if b then s1 else s2)
        | _       -> raise(Failure("Invalid condition: all conditionals must
          have boolean values")))

      | While(e, s) ->
	  let rec loop env =
	    let v, env = eval env e in
          (match v with
            VBool b -> if b then loop (exec env s) else env
          | _       -> raise(Failure("Invalid condition: all conditionals must
                      have boolean values")))
	  in loop env
      | Print(e) ->
              let str,env = eval env e
              in
              (match (str) with
              (VString str) -> print_string str ; flush stdout; env
            | _ -> print_string "Not Printing String" ; raise (Failure("Trying to a print a value that isn't a
            string")) )
      | End  ->
              let v, (locals, globals) = eval env Noexpr in (* Do nothing, but get globals *)
              raise(ReturnException(v,globals)) 
      | Exit ->
              print_string "Exit called\n" ; exit 0; NameMap.empty, NameMap.empty 
    in

    (* Enter the function: bind actual values to formal arguments *)
    (* Execute each statement in sequence, return updated global symbol table *)
    snd (List.fold_left exec (NameMap.empty, globals) rdecl.body)

  (* Run a program: initialize global variables to VNothing, find and run "main" *)
  in let globs = 
      List.fold_left (fun gmap x ->
          (match x with
            Expr(Assign(var,_,_)) -> NameMap.add var VNothing gmap
          | _               -> raise(Failure("Fatal error: global declarations
            contain a statement that isn't an assignment")) )
      ) NameMap.empty globals

  in let globs = 
      call {rname = ""; body = globals} globs
  (* in
     let _ = NameMap.iter (fun k v -> print_endline k ) globs *)
  in
    if NameMap.mem "start_game" rule_decls then
    try
      call (NameMap.find "start_game" rule_decls) globs
    with
        ReturnException(v,globals) -> (globals)
    else
      raise(Failure("start_game rule not defined"))
      (*
  in try
    call (NameMap.find "main" func_decls) [] globals
  with Not_found -> raise (Failure ("did not find the main() function"))
  *)
