(*
 * This module defines various generic utilities.
 *)

(**************)
(* List Utils *)

(* Global random initialization *)
Random.self_init ()

(* Randomly shuffle a list using Fisher-Yates. *)
let shuffle : 'a list -> 'a list =
  fun l ->
    let s = List.length l in
    let a = Array.of_list l in
    for n = s downto 2 do
      let r = Random.int n in
      let t = a.(n - 1) in
      a.(n - 1) <- a.(r);
      a.(r) <- t
    done;
    Array.to_list a

(*
 * Split a list at the first occurrence of the given element.
 *
 * The first item in the resulting pair will contain all elements
 * before the given one is found, and the second item in the pair
 * will include the item and everything after it.
 *
 * Raises 'Not_found' if the item is not in the list.
 *
 * Equality is performed by the given function.
 *)
let split : ('a -> 'a -> bool) -> 'a -> 'a list -> ('a list * 'a list) =
  fun f e l ->
    let rec helper l accum =
      match l with
          [] -> raise(Not_found)
        | h :: t -> if f h e then (List.rev accum, l)
                             else helper t (h :: accum)
    in helper l []

(*
 * Remove the first occurrence of the given element from the list.
 * Use the given function to test equality.
 *)
let rec lremove : ('a -> 'a -> bool) -> 'a -> 'a list -> 'a list =
  fun f e -> function
    [] -> []
  | h :: t -> if f h e then t else h :: (lremove f e t)

(*
 * Return the 0-based index of the element in the list, using
 * the given function to test equality. If the element is not
 * found, return the default value.
 *)
let lindex : ('a -> 'a -> bool) -> 'a list -> 'a -> int -> int =
  fun fn l e def ->
    let rec helper l ind =
      match l with
        []   -> def
      | h::t ->
          if fn h e then ind else helper t (ind + 1)
    in helper l 0

(*
 * Order the elements of the first list so that they are
 * in the same relative ordering as the same elements in
 * the second list.
 *
 * Use the given function to test equality.
 *)
let order : ('a -> 'a -> bool) -> 'a list -> 'a list -> 'a list =
  fun f l1 l2 ->
    let fn e1 e2 =
      (* Return positive if e1 should go after e2 in the result,
         negative if the reverse, and 0 if it doesn't matter *)
      let max = List.length l2 in
      let i1 = lindex f l2 e1 max in
      let i2 = lindex f l2 e2 max in
      i1 - i2
    in
    List.sort fn l1

(*
 * Call the function once for every combination of
 * elements from 'a and 'b. This is the Cartesian Product.
 * Returns a list of the results.
 *)
let xmap : ('a -> 'b -> 'c) -> 'a list -> 'b list -> 'c list =
  fun f al bl ->
    let rec helper al' bl' cl' =
      match (al', bl') with
        ([], _) -> cl'
      | (h::t, []) -> helper t bl cl'
      | (h::_, h'::t') -> let c = f h h' in
                          helper al' t' (c::cl')
    in List.rev (helper al bl [])


(****************)
(* String Utils *)

(* char list to string *)
let str_implode : char list -> string =
  fun clist ->
    let slist = List.map (fun c -> String.make 1 c) clist in
    String.concat "" slist

(* string to char list *)
let str_explode : string -> char list =
  fun s ->
    let rec helper s n l =
      if n == l then []
                else s.[n] :: helper s (n + 1) l
    in helper s 0 (String.length s)

(* Map a function across the chars in a string and concatenate the results *)
let str_map : (char -> string) -> string -> string =
  fun f s ->
    let clist = str_explode s in
    let slist = List.map f clist in
    String.concat "" slist

(* Map a function over a list and concatenate the resulting strings *)
let catmap : string -> ('a -> string) -> 'a list -> string =
  fun s f alist -> String.concat s (List.map f alist)


(*************)
(* Int utils *)

(* Return a list range including the end points *)
let rec int_range : int -> int -> int list =
  fun low high ->
    if low = high then [low]
    else if low > high then List.rev (int_range high low)
    else low :: (int_range (low + 1) high)


(***************)
(* Error utils *)

(* Raise a Failure exception with line/column information *)
let pos_error : int * int -> string -> 'a =
  fun pos str ->
    let prefix = "Error: Line " ^ (string_of_int (fst pos)) ^
                       ", Col " ^ (string_of_int (snd pos)) ^
                       ": "
    in raise(Failure(prefix ^ str))

(* Raise an internal error *)
let ie : string -> 'a =
  fun s -> raise(Failure("Internal Error: " ^ s))


(****************)
(* Line Numbers *)

(* Track line numbers when scanning and parsing. *)
let line_num = ref 1

(* Return the (line, col) of the Lexing position *)
let lexpos : Lexing.position -> int * int =
  fun pos ->
    let line = pos.Lexing.pos_lnum
    and col  = pos.Lexing.pos_cnum - pos.Lexing.pos_bol
    in (line, col + 1)
