(*
 * This module performs semantic analysis on the Ast resolving
 * symbols and type checking the program. The result is a
 * Cast ("Checked Ast") that can be interpreted directly with
 * no potential unresolved symbol errors or type errors.
 *)

module StringMap = Map.Make(String)

(*
 * Utils
 *)

(*
 * Merge two StringMaps. If a key exists in both, the value
 * in the second map will win.
 *)
let map_merge : 'a StringMap.t -> 'a StringMap.t -> 'a StringMap.t =
  fun m1 m2 -> StringMap.fold StringMap.add m2 m1

(*
 * When given a StringMap and a string, this raises an error if
 * the given key is already in the map. Otherwise, it does nothing.
 *)
let assert_no_dup : 'a StringMap.t -> string -> Cast.pos
                                    -> string -> unit =
  fun map id p s ->
    if StringMap.mem id map
      then Utils.pos_error p ("Duplicate " ^ s ^ ": " ^ id)
      else ()


(*
 * Scopes
 *)

(*
 * There is one scope for global variables and
 * one scope for each new Cast.block that is entered.
 *
 * Each scope contains a lookup table for resolving
 * variables referenced in that scope.
 *)
type scope = Cast.variable StringMap.t

(*
 * Add a variable to the given scope, returning a new scope.
 * Ensure the scope doesn't contain a variable of the given name.
 *)
let scope_add_var : scope -> string -> Cast.variable -> Cast.pos -> scope =
  fun sc id var p ->
    assert_no_dup sc id p "Variable";
    StringMap.add id var sc


(*
 * Environment
 *)

(*
 * The environment contains a list of scopes for variables,
 * which are resolved differently depending on the current
 * scope, and a single lookup table for actions, rules, and
 * orderings, which are always global.
 *
 * The scope list contains the current scope at the head.
 * Successive scopes in the list represent successively
 * nested scopes in the program with the last scope in the
 * list being the global scope.
 *)
type env = { e_vars      : scope list;
             e_actions   : Cast.action StringMap.t;
             e_rules     : Cast.rule StringMap.t;
             e_orderings : Cast.ordering StringMap.t; }

(* Push a new, empty scope on to the env, returning a new env. *)
let env_push_scope : env -> env =
  fun ev -> { ev with e_vars = StringMap.empty :: ev.e_vars }

(* Add a variable to env's current scope, returning a new env. *)
let env_add_var : env -> string -> Cast.variable -> Cast.pos -> env =
  fun ev id var p ->
    match ev.e_vars with
      []       -> Utils.ie "No current scope"
    | (h :: t) -> { ev with e_vars = scope_add_var h id var p :: t }

(*
 * Add an action to the env, returning a new env.
 * Ensure the env doesn't contain an action of the given name.
 *)
let env_add_action : env -> string -> Cast.action -> Cast.pos -> env =
  fun ev id act p ->
    let acts = ev.e_actions in
    assert_no_dup acts id p "Action";
    { ev with e_actions = StringMap.add id act acts }

(*
 * Add a rule to the env, returning a new env.
 *
 * It is a semantic error if the env already contains a rule
 * of the given name.
 *)
let env_add_rule : env -> string -> Cast.rule -> Cast.pos -> env =
  fun ev id rule p ->
    let rules = ev.e_rules in
    assert_no_dup rules id p "Rule";
    { ev with e_rules = StringMap.add id rule rules }

(*
 * Add an ordering to the env, returning a new env.
 *
 * It is a semantic error if the env already contains an ordering
 * of the given name.
 *)
let env_add_ordering : env -> string -> Cast.ordering -> Cast.pos -> env =
  fun ev id ord p ->
    let ords = ev.e_orderings in
    assert_no_dup ords id p "Ordering";
    { ev with e_orderings = StringMap.add id ord ords }


(*
 * Semantic analysis of types
 *)

(*
 * Given an Ast.dtype, this results in the corresponding Cast.t
 *)
let sem_type : Ast.dtype -> Cast.t =
  function
    Ast.Boolean    -> Cast.Boolean
  | Ast.Card       -> Cast.Card
  | Ast.Number     -> Cast.Number
  | Ast.Player     -> Cast.Player
  | Ast.Rank       -> Cast.Rank
  | Ast.String     -> Cast.String
  | Ast.Suit       -> Cast.Suit
  | Ast.Team       -> Cast.Team
  
  | Ast.CardList   -> Cast.L(Cast.Card)
  | Ast.Deck       -> Cast.L(Cast.Card)
  | Ast.PlayerList -> Cast.L(Cast.Player)
  | Ast.RankList   -> Cast.L(Cast.Rank)
  | Ast.SuitList   -> Cast.L(Cast.Suit)
  | Ast.TeamList   -> Cast.L(Cast.Team)

(* Return the element type if t is a list type; otherwise return t *)
let element_type : Cast.t -> Cast.t =
  function Cast.L(t) -> t
         | t -> t

(* Decide if the second type can be used where the first is expected. *)
let rec ok_type : Cast.t -> Cast.t -> bool =
  fun exp act ->
    match (exp, act) with
      (_, Cast.EmptyType)      -> true
    | (Cast.EmptyType, _)      -> true
    | (Cast.Rank, Cast.Number) -> true
    | (Cast.Number, Cast.Rank) -> true
    | (Cast.L(x), Cast.L(y))   -> ok_type x y
    | (x, y)                   -> x = y

(*
 * Ensure the second type can be used where the first type
 * was expected.
 *)
let assert_ok_type : Cast.t -> Cast.t -> Cast.pos -> unit =
  fun t1 t2 p ->
    if not (ok_type t1 t2)
      then Utils.pos_error p
             ("Expected type " ^ (Cast.str_type t1) ^
              " but found type " ^ (Cast.str_type t2))

(*
 * Ensure the type is a list type. Return the
 * type of its elements.
 *)
let assert_list_type : Cast.t -> Cast.pos -> Cast.t =
  fun t p ->
    let elt = element_type t in
    if elt = t
      then Utils.pos_error p
             ("Expected a list type but found type " ^ (Cast.str_type t));
    elt

(*
 * Ensure the second type is ok in a list literal that
 * already contains elements of the first type.
 *
 * This returns the new element type of the list, which
 * may be different if the current element type is
 * unknown (Cast.EmptyType).
 *)
let assert_listlit_type : Cast.t -> Cast.t -> Cast.pos -> Cast.t =
  fun old t p ->
    let oldelt = element_type old in
    let elt    = element_type t in
    assert_ok_type oldelt elt p;
    elt

(* Ensure the type matches one in the list using assert_ok_type. *)
let assert_type_opts : Cast.t list -> Cast.t -> Cast.pos -> unit =
  fun ts t p ->
    let to_str ts = Utils.catmap " or " Cast.str_type ts in
    if not (List.exists (fun t' -> ok_type t' t) ts)
      then Utils.pos_error p
             ("Expected type to be one of " ^ (to_str ts) ^
              " but found type " ^ (Cast.str_type t))

(* Ensure the type is valid to deal from. *)
let assert_deal_from_type : Cast.t -> Cast.pos -> unit =
  fun t p ->
    let ts = [Cast.L(Cast.Card); Cast.Player; Cast.Area] in
    assert_type_opts ts t p

(* Ensure the type is valid to deal to. *)
let assert_deal_to_type : Cast.t -> Cast.pos -> unit =
  fun t p ->
    let ts = [Cast.L(Cast.Card); Cast.Player;
              Cast.L(Cast.Player); Cast.Area] in
    assert_type_opts ts t p

(* Ensure the type is valid for a "play" and "canplay" destination. *)
let assert_play_type : Cast.t -> Cast.pos -> unit =
  fun t p ->
    let ts = [Cast.L(Cast.Card); Cast.Player; Cast.Area] in
    assert_type_opts ts t p

(* Ensure the type is valid for a "message" statement. *)
let assert_msg_type : Cast.t -> Cast.pos -> unit =
  fun t p ->
    let ts = [Cast.Player; Cast.L(Cast.Player); Cast.Team] in
    assert_type_opts ts t p

(* Ensure the type is valid for a "winner" statement. *)
let assert_winner_type : Cast.t -> Cast.pos -> unit =
  fun t p ->
    let ts = [Cast.Player; Cast.Team] in
    assert_type_opts ts t p

(*
 * Ensure the types are valid for the binary operator.
 * Return the result type.
 *)
let assert_op_types : Cast.op -> Cast.t -> Cast.t
                   -> Cast.pos -> Cast.pos -> Cast.t =
  fun op t1 t2 p1 p2 ->
    match op with
      Ast.Plus ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Number
    | Ast.Minus ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Number
    | Ast.Times ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Number
    | Ast.Divide ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Number
    | Ast.Eq ->
        assert_ok_type t1 t2 p2;
        if t1 = Cast.Area
          then Utils.pos_error p1 "Invalid type Area for comparison";
        Cast.Boolean
    | Ast.NotEq ->
        assert_ok_type t1 t2 p2;
        if t1 = Cast.Area
          then Utils.pos_error p1 "Invalid type Area for comparison";
        Cast.Boolean
    | Ast.Lt ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Boolean
    | Ast.LtEq ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Boolean
    | Ast.Gt ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Boolean
    | Ast.GtEq ->
        assert_ok_type Cast.Number t1 p1;
        assert_ok_type Cast.Number t2 p2;
        Cast.Boolean
    | Ast.And ->
        assert_ok_type Cast.Boolean t1 p1;
        assert_ok_type Cast.Boolean t2 p2;
        Cast.Boolean
    | Ast.Or ->
        assert_ok_type Cast.Boolean t1 p1;
        assert_ok_type Cast.Boolean t2 p2;
        Cast.Boolean
    | Ast.In ->
        let lt = assert_list_type t2 p2 in
        if lt = Cast.Card
          then assert_type_opts [Cast.Card;
                                 Cast.Rank;
                                 Cast.Suit] t1 p1
          else assert_ok_type lt t1 p1;
        Cast.Boolean

(*
 * Ensure the types are valid for the left and right operand
 * of a Card expression. Return the result type.
 *)
let assert_cardexpr_types : Cast.t -> Cast.t
                         -> Cast.pos -> Cast.pos -> Cast.t =
  fun t1 t2 p1 p2 ->
    assert_type_opts [Cast.Rank; Cast.L(Cast.Rank)] t1 p1;
    assert_type_opts [Cast.Suit; Cast.L(Cast.Suit)] t2 p2;
    match (t1, t2) with
        (Cast.Rank, Cast.Suit)   -> Cast.Card
      | (Cast.Number, Cast.Suit) -> Cast.Card
      | _                        -> Cast.L(Cast.Card)

(*
 * Ensure the types are valid for the left and right operand
 * of a sequence expression. Return the result type.
 *)
let assert_seq_types : Cast.t -> Cast.t
                    -> Cast.pos -> Cast.pos -> Cast.t =
  fun t1 t2 p1 p2 ->
    let elt1 = element_type t1 in
    let elt2 = element_type t2 in
    let valid = [Cast.Rank; Cast.Suit] in
    assert_type_opts valid elt1 p1;
    assert_type_opts valid elt2 p2;
    assert_ok_type elt1 elt2 p2;
    Cast.L(elt1)

(* Given a variable, this results in its type. *)
let var_type : Cast.variable -> Cast.t =
  function
      Cast.Global(_, t) -> t
    | Cast.Local(_, t)  -> t
    | Cast.PlayersVar   -> Cast.L(Cast.Player)
    | Cast.TeamsVar     -> Cast.L(Cast.Team)
    | Cast.StandardVar  -> Cast.L(Cast.Card)

(* Given a property, this results in its type. *)
let prop_type : Cast.prop -> Cast.t =
  function
      Cast.ObjProp(_, t) -> t
    | Cast.ListProp(_, t) -> t

(* Ensure the type can be converted to a Cast.String. *)
let assert_str_conv_ok : Cast.t -> Cast.pos -> unit =
  fun t p ->
    let str_conv_ok = function
        Cast.Area       -> false
      | Cast.Boolean    -> true
      | Cast.Card       -> true
      | Cast.Number     -> true
      | Cast.Player     -> true
      | Cast.Rank       -> true
      | Cast.String     -> true
      | Cast.Suit       -> true
      | Cast.Team       -> true
      | Cast.L(_)       -> false
      | Cast.EmptyType  -> false
    in
    if str_conv_ok t
      then ()
      else Utils.pos_error p
             ("Can't convert " ^ (Cast.str_type t) ^ " to a String")


(*
 * Name resolution
 *)
 
(* This resolves the variable's name to a variable. *)
let rec res_var : scope list -> Ast.id -> Ast.pos -> Cast.variable =
  fun sl id p ->
    match sl with
      []       -> Utils.pos_error p ("Unknown identifier '" ^ id ^ "'")
    | (h :: t) -> try StringMap.find id h
                  with Not_found -> res_var t id p

(*
 * This resolves the type and property name into a property.
 *
 * A property consists of its offset into the array of
 * properties that object values are represented by,
 * along with its type.
 *
 * List properties are represented symbolically and are
 * calculated at run-time. They are read-only and have
 * no storage.
 *)
let res_prop : Cast.t -> string -> Cast.pos -> Cast.prop =
  fun t id p ->
    match (t, id) with
      (Cast.Card, "rank")           -> Cast.ObjProp(0, Cast.Rank)
    | (Cast.Card, "suit")           -> Cast.ObjProp(1, Cast.Suit)
    | (Cast.Card, "last_played_by") -> Cast.ObjProp(2, Cast.Player)
    
    | (Cast.Player, "name")  -> Cast.ObjProp(0, Cast.String)
    | (Cast.Player, "hand")  -> Cast.ObjProp(1, Cast.L(Cast.Card))
    | (Cast.Player, "stash") -> Cast.ObjProp(2, Cast.L(Cast.Card))
    | (Cast.Player, "score") -> Cast.ObjProp(3, Cast.Number)
    | (Cast.Player, "team")  -> Cast.ObjProp(4, Cast.Team)
    
    | (Cast.Team, "members") -> Cast.ObjProp(0, Cast.L(Cast.Player))
    | (Cast.Team, "stash")   -> Cast.ObjProp(1, Cast.L(Cast.Card))
    | (Cast.Team, "score")   -> Cast.ObjProp(2, Cast.Number)
    
    | (Cast.Area, "name")         -> Cast.ObjProp(0, Cast.String)
    | (Cast.Area, "cards")        -> Cast.ObjProp(1, Cast.L(Cast.Card))
    | (Cast.Area, "is_facedown")  -> Cast.ObjProp(2, Cast.Boolean)
    | (Cast.Area, "is_squaredup") -> Cast.ObjProp(3, Cast.Boolean)
    
    | (Cast.L(_), "size")   -> Cast.ListProp(Cast.Size, Cast.Number)
    | (Cast.L(t), "first")  -> Cast.ListProp(Cast.First, t)
    | (Cast.L(t), "last")   -> Cast.ListProp(Cast.Last, t)
    | (Cast.L(t), "top")    -> Cast.ListProp(Cast.Top, t)
    | (Cast.L(t), "bottom") -> Cast.ListProp(Cast.Bottom, t)

    | _ -> Utils.pos_error p ("Type " ^ (Cast.str_type t) ^
                                " has no property named " ^
                                "'" ^ id ^ "'")

(* This resolves the action's name to an action. *)
let res_action : env -> Ast.id -> Ast.pos -> Cast.action =
  fun ev id p -> try StringMap.find id ev.e_actions
                 with Not_found ->
                   Utils.pos_error p ("Unknown action '" ^ id ^ "'")

(* This resolves an ordering's name to an ordering. *)
let res_ordering : env -> Ast.id -> Ast.pos -> Cast.ordering =
  fun ev id p -> try StringMap.find id ev.e_orderings
                 with Not_found ->
                   Utils.pos_error p ("Unknown ordering '" ^ id ^ "'")

(* This resolves a rule's name to a rule. *)
let res_rule : env -> Ast.id -> Ast.pos -> Cast.rule =
  fun ev id p -> try StringMap.find id ev.e_rules
                 with Not_found ->
                   Utils.pos_error p ("Unknown rule '" ^ id ^ "'")


(*
 * Semantic analysis of expressions and references
 *)

(*
 * Given a reference, this results in its position.
 * For a property, this is the position of the property's name.
 *)
let ref_pos : Cast.reference -> Cast.pos =
  function
      Cast.Var(_, p)     -> p
    | Cast.Prop(_, _, p) -> p
    | Cast.EmptyRef      -> (-1, -1)

(* Given an expr, this results in its position. *)
let rec expr_pos : Cast.expr -> Cast.pos =
  function
      Cast.Not(_, p)            -> p
    | Cast.Binary(e, _, _, _)   -> expr_pos e
    | Cast.Defined(_, p)        -> p
    | Cast.CanPlay(_, _, _, p)  -> p
    | Cast.CardExpr(e, _, _)    -> expr_pos e
    | Cast.RangeExpr(e, _, _)   -> expr_pos e
    | Cast.RankSeqExpr(e, _, _) -> expr_pos e
    | Cast.SuitSeqExpr(e, _, _) -> expr_pos e
    | Cast.WildRank(p, _, _)    -> p
    | Cast.WildSuit(e, _, _)    -> expr_pos e
    | Cast.WildCard(p, _, _)    -> p
    | Cast.List(_, _, p)        -> p
    | Cast.Ref(_, p)            -> p
    | Cast.Literal(_, p)        -> p
    | Cast.StrLiteral(_, p)     -> p
    | Cast.EmptyExpr            -> (-1, -1)

(*
 * Analyze the Ast.reference, creating a Cast.reference
 * and its type.
 *)
let rec sem_ref : env -> Ast.reference -> Cast.reference * Cast.t =
  fun ev -> function
    Ast.Id(id, p)      -> let v = res_var ev.e_vars id p
                          in Cast.Var(v, p), var_type v
  | Ast.Prop(e, id, p) -> let (e, t) = sem_expr ev e in
                          let f = res_prop t id p in
                          Cast.Prop(e, f, p), prop_type f
  | Ast.Players(p)     -> let v = Cast.PlayersVar
                          in Cast.Var(v, p), var_type v
  | Ast.Teams(p)       -> let v = Cast.TeamsVar
                          in Cast.Var(v, p), var_type v
  | Ast.Standard(p)    -> let v = Cast.StandardVar
                          in Cast.Var(v, p), var_type v
  | Ast.EmptyRef       -> Cast.EmptyRef, Cast.EmptyType

(*
 * Analyze the Ast.expr, creating the Cast.expr and
 * its type.
 *)
and sem_expr : env -> Ast.expr -> Cast.expr * Cast.t =
  fun ev -> function
      Ast.Not(e, p) ->
        let (e, t) = sem_expr ev e in
        assert_ok_type Cast.Boolean t (expr_pos e);
        Cast.Not(e, p), Cast.Boolean
    | Ast.Binary(e1, op, e2, p) ->
        let (e1, t1) = sem_expr ev e1 in
        let (e2, t2) = sem_expr ev e2 in
        let p1 = expr_pos e1 in
        let p2 = expr_pos e2 in
        let t = assert_op_types op t1 t2 p1 p2 in
        Cast.Binary(e1, op, e2, p), t
    | Ast.Defined(r, p) ->
        let (r, _) = sem_ref ev r in
        Cast.Defined(r, p), Cast.Boolean
    | Ast.CanPlay(id, r1, r2, p) ->
        let rule = res_rule ev id p in
        let (r1, t1) = sem_ref ev r1 in
        assert_ok_type Cast.Player t1 (ref_pos r1);
        let (r2, t2) = sem_ref ev r2 in
        assert_play_type t2 (ref_pos r2);
        Cast.CanPlay(rule, r1, r2, p), Cast.Boolean
    | Ast.CardExpr(e1, p, e2) ->
        let (e1, t1) = sem_expr ev e1 in
        let (e2, t2) = sem_expr ev e2 in
        let p1 = expr_pos e1 in
        let p2 = expr_pos e2 in
        let t = assert_cardexpr_types t1 t2 p1 p2 in
        Cast.CardExpr(e1, p, e2), t
    | Ast.RangeExpr(e1, e2, p) ->
        let (e1, t1) = sem_expr ev e1 in
        let (e2, t2) = sem_expr ev e2 in
        let p1 = expr_pos e1 in
        let p2 = expr_pos e2 in
        assert_ok_type Cast.Rank t1 p1;
        assert_ok_type Cast.Rank t2 p2;
        Cast.RangeExpr(e1, e2, p), Cast.L(Cast.Rank)
    | Ast.SeqExpr(e1, e2, p) ->
        let (e1, t1) = sem_expr ev e1 in
        let (e2, t2) = sem_expr ev e2 in
        let p1 = expr_pos e1 in
        let p2 = expr_pos e2 in
        let t = assert_seq_types t1 t2 p1 p2 in
        if t = Cast.L(Cast.Rank) then
          Cast.RankSeqExpr(e1, e2, p), t
        else
          Cast.SuitSeqExpr(e1, e2, p), t          
    | Ast.WildRank(p1, p2, e) ->
        let (e, t) = sem_expr ev e in
        let p = expr_pos e in
        assert_type_opts [Cast.Suit; Cast.L(Cast.Suit)] t p;
        Cast.WildRank(p1, p2, e), Cast.L(Cast.Card)
    | Ast.WildSuit(e, p1, p2) ->
        let (e, t) = sem_expr ev e in
        let p = expr_pos e in
        assert_type_opts [Cast.Rank; Cast.L(Cast.Rank)] t p;
        Cast.WildSuit(e, p1, p2), Cast.L(Cast.Card)
    | Ast.WildCard(p1, p2, p3) ->
        Cast.WildCard(p1, p2, p3), Cast.L(Cast.Card)
    | Ast.List(es, p) ->
        let f (es, elt) e =
          let (e', elt') = sem_expr ev e in
          let elt' = assert_listlit_type elt elt' (expr_pos e') in
          (e' :: es, elt')
        in
        let (es, elt) = List.fold_left f ([], Cast.EmptyType) es in
        Cast.List(List.rev es, elt, p), Cast.L(elt)
    | Ast.Ref(r, p) ->
        let (r, t) = sem_ref ev r in
        Cast.Ref(r, p), t
    | Ast.BoolLit(b, p) ->
        Cast.Literal(Cast.BoolVal(b), p), Cast.Boolean
    | Ast.NumLit(i, p) ->
        Cast.Literal(Cast.NumVal(i), p), Cast.Number
    | Ast.StringLit(sl, p) ->
        let sl = sem_str_lit ev sl in
        Cast.StrLiteral(sl, p), Cast.String
    | Ast.RankLit(rk, p) ->
        Cast.Literal(Cast.RankVal(rk), p), Cast.Rank
    | Ast.SuitLit(st, p) ->
        Cast.Literal(Cast.SuitVal(st), p), Cast.Suit
    | Ast.EmptyExpr ->
        Cast.EmptyExpr, Cast.EmptyType

(*
 * Analyze the Ast.str_lit, resolving any nested
 * references against the current env and ensuring
 * they can be converted to strings.
 *
 * This results in a Cast.str_lit.
 *)
and sem_str_lit : env -> Ast.str_lit -> Cast.str_lit =
  fun ev sl ->
    let sem_part ev = function
        Ast.StrRef(r) -> let (r, t) = sem_ref ev r in
                         assert_str_conv_ok t (ref_pos r);
                         Cast.StrRef(r)
      | Ast.StrLit(s) -> Cast.StrLit(s)
    in List.map (sem_part ev) sl


(*
 * Semantic analysis of statements
 *)

(*
 * This statement information is threaded through each statement
 * as it is analyzed.
 *
 * The si_locals member is a count of how many locals have
 * been declared so far in the entire action. It is set to 0
 * when an action is entered and it is incremented every
 * time a "let" or "for" introduces a new variable that
 * needs space.
 *
 * The si_skips member maps a skip name to the position at
 * which the skip was encountered. If more than one skip for
 * the same name is active, any one of those positions is
 * used, since only one is needed when reporting errors.
 * It is set to empty every time a new scope is entered,
 * and each time a scope is exited, the inner skips are
 * merged with the outer skips. Skips are removed from
 * this map when their corresponding label is found.
 * Any skips remaining at the end of an action were
 * unresolved.
 *
 * The si_labels member maps a label name to the position
 * at which it was encountered. This starts empty at
 * the beginning of an action and accumulates labels
 * for every statement in the action. It is used to detect
 * duplicate labels.
 *)
type sinfo = { si_locals : int;
               si_skips  : Cast.pos StringMap.t;
               si_labels : Cast.pos StringMap.t; }

(*
 * Increment the si_locals count and return a new sinfo.
 *)
let si_inc_locals : sinfo -> sinfo =
  fun si -> { si with si_locals = si.si_locals + 1 }

(*
 * Helper. This takes a triplet: ('a list, 'b, 'c) and
 * a function: (('a, 'b, 'c) -> ('d, 'b, 'c)) and acts
 * a bit like fold_left.
 *
 * The function will be applied to each item in the 'a list.
 * The 'b and 'c arguments for each application of the
 * function come from the return value of the previous
 * invocation of the function (they start with the given
 * 'b and 'c for thread).
 *
 * Each 'd value the function returns is accumulated
 * in to a list, and the final 'd list is returned
 * with the final 'b and 'c from the final invocation
 * of the function.
 *)
let thread2 : ('a * 'b * 'c -> 'd * 'b * 'c)
           -> 'a list * 'b * 'c
           -> 'd list * 'b * 'c =
  fun f (alist, b, c) ->
    let walk (dlist, b, c) a =
      (let (d, b, c) = f (a, b, c) in (d :: dlist, b, c)) in
    let (dlist, b, c) = List.fold_left walk ([], b, c) alist in
    (List.rev dlist, b, c)

(* Like thread2 but only threads one item instead of two *)
let thread1 : ('a * 'b -> 'c * 'b)
           -> 'a list * 'b
           -> 'c list * 'b =
  fun f (alist, b) ->
    let walk (clist, b) a =
      (let (c, b) = f (a, b) in (c :: clist, b)) in
    let (clist, b) = List.fold_left walk ([], b) alist in
    (List.rev clist, b)

(* Ensure there are no pending skips that could pass the Let *)
let assert_skip_let : 'a StringMap.t -> Cast.pos -> unit =
  fun map p ->
    if StringMap.is_empty map
      then ()
      else Utils.pos_error p "Can't skip past Let"

(*
 * Analyze the Ast.stmt, creating a Cast.stmt.
 * Thread the sinfo and env through it.
 *)
let rec sem_stmt : Ast.stmt * sinfo * env
                -> Cast.stmt * sinfo * env =
  fun (stmt, si, ev) ->
    match stmt with
      Ast.Ask(r, qs, p) ->
        let (r, t) = sem_ref ev r in
        assert_ok_type Cast.Player t (ref_pos r);
        let (qs, si) = sem_questions ev (qs, si) in
        Cast.Ask(r, qs, p), si, ev
    | Ast.Assign(r, e, p) ->
        let (r, t1) = sem_ref ev r in
        let (e, t2) = sem_expr ev e in
        assert_ok_type t1 t2 (ref_pos r);
        Cast.Assign(r, e, p), si, ev
    | Ast.Compound(r, op, e, p) ->
        let (r, t1) = sem_ref ev r in
        assert_ok_type Cast.Number t1 (ref_pos r);
        let (e, t2) = sem_expr ev e in
        assert_ok_type Cast.Number t2 (expr_pos e);
        Cast.Compound(r, op, e, p), si, ev
    | Ast.Deal(e, r1, r2, p) ->
        let (e, t) = sem_expr ev e in
        assert_type_opts [Cast.Number; Cast.Card] t (expr_pos e);
        let (r1, t1) = sem_ref ev r1 in
        assert_deal_from_type t1 (ref_pos r1);
        let (r2, t2) = sem_ref ev r2 in
        assert_deal_to_type t2 (ref_pos r2);
        (* Decide on which Deal it is *)
        if t = Cast.Card then
          (if t2 = Cast.L(Cast.Player) then
             Utils.pos_error p "Can't deal a single Card to a PlayerList";
           Cast.DealCard(e, r1, r2, p), si, ev)
        else
          Cast.DealCards(e, r1, r2, p), si, ev
    | Ast.Forever(b, p) ->
        let (b, si) = sem_block ev (b, si) in
        Cast.Forever(b, p), si, ev
    | Ast.For(id, e1, e2, b, p) ->
        let (e1, t1) = sem_expr ev e1 in
        let elt = assert_list_type t1 (expr_pos e1) in
        let (e2, t2) = sem_expr ev e2 in
        assert_ok_type elt t2 (expr_pos e2);
        (* Create a temporary local var *)
        let var = Cast.Local(si.si_locals, elt) in
        let si = si_inc_locals si in
        (* Create a new scope for the duration of the For *)
        let evtmp = env_push_scope ev in
        let evtmp = env_add_var evtmp id var p in
        let (b, si) = sem_block evtmp (b, si) in
        Cast.For(var, e1, e2, b, p), si, ev
    | Ast.If(ess, p) ->
        let sem_ess ev ((e, b), si) =
          let (e, t) = sem_expr ev e in
          assert_ok_type Cast.Boolean t (expr_pos e);
          let (b, si) = sem_block ev (b, si) in
          (e, b), si
        in
        let (ess, si) = thread1 (sem_ess ev) (ess, si) in
        Cast.If(ess, p), si, ev
    | Ast.Invoke(id, p) ->
        let act = res_action ev id p in
        Cast.Invoke(act, p), si, ev
    | Ast.Label(id, p) ->
        (* Ensure no duplicate labels *)
        assert_no_dup si.si_labels id p "Label";
        (* Remove any skips from the sinfo.*)
        let skips = if StringMap.mem id si.si_skips
                      then StringMap.remove id si.si_skips
                      else Utils.pos_error p "Unused label" in
        let labs = StringMap.add id p si.si_labels in
        let si = { si with si_skips = skips;
                           si_labels = labs; } in
        Cast.Label(id, p), si, ev
    | Ast.Let(id, e, p) ->
        (* Ensure there are no active skips *)
        assert_skip_let si.si_skips p;
        let (e, t) = sem_expr ev e in
        let var = Cast.Local(si.si_locals, t) in
        let decl = Cast.VarDecl(var, e, p) in
        let si = si_inc_locals si in
        let ev = env_add_var ev id var p in
        Cast.Let(decl, p), si, ev
    | Ast.Message(r, e, p) ->
        let (r, t1) = sem_ref ev r in
        assert_msg_type t1 p;
        let (e, t2) = sem_expr ev e in
        assert_ok_type Cast.String t2 (expr_pos e);
        Cast.Message(r, e, p), si, ev
    | Ast.Order(r, id, p) ->
        let (r, t) = sem_ref ev r in
        assert_ok_type (Cast.L(Cast.Card)) t (ref_pos r);
        let ord = res_ordering ev id p in
        Cast.Order(r, ord, p), si, ev
    | Ast.Play(id, r1, r2, p) ->
        let rule = res_rule ev id p in
        let (r1, t1) = sem_ref ev r1 in
        assert_ok_type Cast.Player t1 (ref_pos r1);
        let (r2, t2) = sem_ref ev r2 in
        assert_play_type t2 (ref_pos r2);
        Cast.Play(rule, r1, r2, p), si, ev
    | Ast.Rotate(r, p) ->
        let (r, t) = sem_ref ev r in
        ignore (assert_list_type t (ref_pos r));
        Cast.Rotate(r, p), si, ev
    | Ast.Shuffle(r, p) ->
        let (r, t) = sem_ref ev r in
        ignore (assert_list_type t (ref_pos r));
        Cast.Shuffle(r, p), si, ev
    | Ast.Skip(id, p) ->
        (* Add this skip to the sinfo *)
        let skips = StringMap.add id p si.si_skips in
        let si = { si with si_skips = skips } in
        Cast.Skip(id, p), si, ev
    | Ast.Winner(r, p) ->
        let (r, t) = sem_ref ev r in
        assert_winner_type t (ref_pos r);
        Cast.Winner(r, p), si, ev

(*
 * Analyze the Ast.question list, creating a Cast.question list.
 * Thread the sinfo through it.
 *)
and sem_questions : env -> (Ast.question list * sinfo)
                        -> (Cast.question list * sinfo) =
  fun ev qsi -> thread1 (sem_question ev) qsi

(*
 * Analyze a single Ast.question, creating a Cast.question.
 * Thread the sinfo through it.
 *)
and sem_question : env -> (Ast.question * sinfo)
                       -> (Cast.question * sinfo) =
  fun ev (q, si) ->
    match q with
      Ast.Uncond(sl, b, p) ->
        let sl = sem_str_lit ev sl in
        let (b, si) = sem_block ev (b, si) in
        Cast.Uncond(sl, b, p), si
    | Ast.Cond(sl, e, b, p) ->
        let sl = sem_str_lit ev sl in
        let (e, t) = sem_expr ev e in
        assert_ok_type Cast.Boolean t (expr_pos e);
        let (b, si) = sem_block ev (b, si) in
        Cast.Cond(sl, e, b, p), si

(*
 * Analyze the Ast.block, creating a Cast.block.
 * Thread the sinfo through it.
 *
 * A new local scope is introduced for the duration of the block.
 *
 * The pending skips on the sinfo is cleared before entering the
 * block, and any pending skips that leave the block are merged
 * into the currently pending skips as the sinfo is returned.
 *)
and sem_block : env -> (Ast.block * sinfo) -> (Cast.block * sinfo) =
  fun ev (b, si) ->
    let ev = env_push_scope ev in
    let si' = { si with si_skips = StringMap.empty } in
    let (b, si', _) = thread2 sem_stmt (b, si', ev) in
    let skips = map_merge si.si_skips si'.si_skips in
    let si' = { si' with si_skips = skips } in
    (b, si')


(*
 * Semantic analysis of declarations
 *)

(*
 * Ensure there are no unresolved skips left in the map.
 *)
let assert_no_skips : Cast.pos StringMap.t -> unit =
  fun map ->
    let err lbl pos =
      Utils.pos_error pos ("Can't find label '" ^ lbl ^ "'")
    in StringMap.iter err map

(*
 * Convert a list of Area options into a bool for facedown
 * and a bool for squaredup.
 *
 * Ensure there are no contradictions.
 *)
let rec sem_areaopts : Ast.areaopts -> bool * bool =
  fun opts ->
    let rec ensure_no item list p =
      match list with
        []     -> ()
      | h :: t ->
        if item = fst h
          then Utils.pos_error p
                 "This option contradicts earlier options"
          else ensure_no item t p
    in
    match opts with
      [] -> (true, true)
    | (Ast.Faceup, p) :: rest ->
        let (_, b) = sem_areaopts rest in
        ensure_no Ast.Facedown rest p;
        (true, b)
    | (Ast.Facedown, p) :: rest ->
        let (_, b) = sem_areaopts rest in
        ensure_no Ast.Faceup rest p;
        (false, b)
    | (Ast.Squaredup, p) :: rest ->
        let (b, _) = sem_areaopts rest in
        ensure_no Ast.Spreadout rest p;
        (b, true)
    | (Ast.Spreadout, p) :: rest ->
        let (b, _) = sem_areaopts rest in
        ensure_no Ast.Squaredup rest p;
        (b, false)

(*
 * Analyze a single global Ast.decl when given the
 * current env and current list of global variables
 * in reverse declaration order.
 *
 * This results in the new env and new list of global
 * variables in reverse declaration order.
 *)
let sem_decl : (env * Cast.var_decl list) -> Ast.decl
            -> (env * Cast.var_decl list) =
  fun (ev, vs) -> function
    Ast.Area(id, sl, opts, p) ->
      let sl = sem_str_lit ev sl in
      let (facedown, squaredup) = sem_areaopts opts in
      let var = Cast.Global(ref Cast.UndefVal, Cast.Area) in
      let decl = Cast.AreaDecl(var, sl, facedown, squaredup, p) in
      env_add_var ev id var p, decl :: vs
  | Ast.Action(id, b, p) ->
      let si = { si_locals = 0;
                 si_skips  = StringMap.empty;
                 si_labels = StringMap.empty; } in
      let (b, si) = sem_block ev (b, si) in
      assert_no_skips si.si_skips;
      let act = (si.si_locals, b, p) in
      env_add_action ev id act p, vs
  | Ast.Rule(id, args, e, p) ->
      (* Create a temporary local environment with the args in it. *)
      let name0 = List.nth args 0 in
      let name1 = List.nth args 1 in
      let name2 = List.nth args 2 in
      let arg0  = Cast.Local(0, Cast.Player) in
      let arg1  = Cast.Local(1, Cast.Card) in
      let arg2  = Cast.Local(2, Cast.L(Cast.Card)) in
      let map   = StringMap.empty in
      let map   = StringMap.add name0 arg0 map in
      let map   = StringMap.add name1 arg1 map in
      let map   = StringMap.add name2 arg2 map in
      let evtmp = { ev with e_vars = map :: ev.e_vars } in
      (* Use that temporary env for the expression only. *)
      let (e, t) = sem_expr evtmp e in
      let rule = (e, p) in
      assert_ok_type Cast.Boolean t p;
      env_add_rule ev id rule p, vs
  | Ast.Ordering(id, arg, e, p) ->
      (* Create a temporary local environment with the arg in it. *)
      let arg0  = Cast.Local(0, Cast.L(Cast.Card)) in
      let map   = StringMap.empty in
      let map   = StringMap.add arg arg0 map in
      let evtmp = { ev with e_vars = map :: ev.e_vars } in
      (* Use that temporary env for the expression only. *)
      let (e, t) = sem_expr evtmp e in
      let ord = (e, p) in
      assert_ok_type (Cast.L(Cast.Card)) t p;
      env_add_ordering ev id ord p, vs
  | Ast.Var(t, id, e, p) ->
      let t = sem_type t in
      let (e, t') = sem_expr ev e in
      let var = Cast.Global(ref Cast.UndefVal, t) in
      let decl = Cast.VarDecl(var, e, p) in
      assert_ok_type t t' p;
      env_add_var ev id var p, decl :: vs

(*
 * Analyze the list of global Ast.decls in the given env.
 *
 * The result is a list of variable declarations
 * and the main action.
 *)
let sem_decls : env -> Ast.decl list
                    -> (Cast.action * Cast.var_decl list) =
  fun ev ds ->
    let (ev, vs) = List.fold_left sem_decl (ev, []) ds in
    let main = try StringMap.find "main" ev.e_actions
               with Not_found ->
                 raise(Failure("Error: No 'main' action was declared."))
    in (main, List.rev vs)


(*
 * Semantic analysis of the game
 *)

(* Analyze the Ast.game and convert to Cast.game *)
let sem_game : Ast.game -> Cast.game =
  fun (sl, c, ds, p) ->
    (* The initial env *)
    let ev = { e_vars      = [StringMap.empty];
               e_actions   = StringMap.empty;
               e_rules     = StringMap.empty;
               e_orderings = StringMap.empty } in
    let sl = sem_str_lit ev sl in
    let (main, vs) = sem_decls ev ds in
    (sl, c, vs, main, p)
