{
  open Parser

  (* Increment position based on seeing a newline *)
  let newline lexbuf =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <- {
      pos with Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
               Lexing.pos_bol = pos.Lexing.pos_cnum;
    };
    Utils.line_num := lexbuf.Lexing.lex_curr_p.Lexing.pos_lnum

  (* Get the position from the lexbuf. Back up n chars since they
     were already scanned. *)
  let pos lexbuf n = let p = lexbuf.Lexing.lex_curr_p in
                     let (l, c) = Utils.lexpos p in
                     (l, c - n)
}

let identifier = ['A'-'Z' 'a'-'z'] ['A'-'Z' 'a'-'z' '0'-'9' '_']*

rule token = parse
  eof { EOF }

(* Grouping *)

| "{" { LCURLY }
| "}" { RCURLY }
| "(" { LPAREN }
| ")" { RPAREN }
| "[" { LSQUARE }
| "]" { RSQUARE }

(* Operators and punctuation *)
(* Note: and, or, in, not in, defined, not defined, canplay are below in keywords *)

| "->" { ARROW }
| ","  { COMMA }
| "."  { DOT }
| ".." { DOTDOT }
| ";"  { SEMI }
| "~"  { TILDE }
| "%"  { PERCENT }

| "="  { ASSIGN }

| "=="  { EQ }
| "!="  { NOTEQ }
| "<"   { LT }
| "<="  { LTEQ }
| ">"   { GT }
| ">="  { GTEQ }

| "+" { PLUS }
| "-" { MINUS }
| "*" { TIMES }
| "/" { DIVIDE }

| "+=" { PLUSEQ }
| "-=" { MINUSEQ }
| "*=" { TIMESEQ }
| "/=" { DIVIDEEQ }

(* Types *)

| "Action"     { ACTION }
| "Area"       { AREA }
| "Boolean"    { BOOLEAN }
| "Card"       { CARD }
| "CardList"   { CARDLIST }
| "Deck"       { DECK }
| "Game"       { GAME }
| "Number"     { NUMBER }
| "Ordering"   { ORDERING }
| "Player"     { PLAYER }
| "PlayerList" { PLAYERLIST }
| "Rank"       { RANK }
| "RankList"   { RANKLIST }
| "Rule"       { RULE }
| "String"     { STRING }
| "Suit"       { SUIT }
| "SuitList"   { SUITLIST }
| "Team"       { TEAM }
| "TeamList"   { TEAMLIST }

(* Keywords *)

| "all"       { ALL }
| "and"       { AND }
| "ask"       { ASK }
| "at"        { AT }
| "be"        { BE }
| "by"        { BY }
| "canplay"   { CANPLAY }
| "deal"      { DEAL }
| "defined"   { DEFINED }
| "else"      { ELSE }
| "elseif"    { ELSEIF }
| "facedown"  { FACEDOWN }
| "faceup"    { FACEUP }
| "for"       { FOR }
| "forever"   { FOREVER }
| "from"      { FROM }
| "if"        { IF }
| "in"        { IN }
| "is"        { IS }
| "label"     { LABEL }
| "labeled"   { LABELED }
| "leaving"   { LEAVING }
| "let"       { LET }
| "message"   { MESSAGE }
| "not"       { NOT }
| "of"        { OF }
| "or"        { OR }
| "order"     { ORDER }
| "play"      { PLAY }
| "players"   { PLAYERS }
| "requires"  { REQUIRES }
| "rotate"    { ROTATE }
| "shuffle"   { SHUFFLE }
| "skip"      { SKIP }
| "spreadout" { SPREADOUT }
| "squaredup" { SQUAREDUP }
| "standard"  { STANDARD }
| "starting"  { STARTING }
| "teams"     { TEAMS }
| "to"        { TO }
| "winner"    { WINNER }

(* Literals *)

| 'A' { ACE }
| 'K' { KING }
| 'Q' { QUEEN }
| 'J' { JACK }

| 'C' { CLUBS }
| 'H' { HEARTS }
| 'S' { SPADES }
| 'D' { DIAMONDS }

| "True"             { TRUE }
| "False"            { FALSE }
| ['0'-'9']+ as num  { NUMLIT(int_of_string num) }
| '"'                { STRINGLIT(str_lit [] [] lexbuf) }

(* Identifiers *)

| identifier as lit { ID(lit) }

(* Whitespace *)

| '\n'          { newline lexbuf ; token lexbuf } (* Newline *)
| [' ' '\t']    { token lexbuf }                  (* Whitespace *)
| "#" [^ '\n']* { token lexbuf }                  (* Line Comments *)

(* Anything else is an error *)

| _ as c { let ce = Char.escaped c in
           Utils.pos_error (pos lexbuf 1) ("Illegal character: " ^ ce) }


(*
 * String literals can contain nested references to identifiers, wrapped
 * in { and }. This rule returns an Ast.str_lit.
 *
 * For this sequence of characters:
 *
 *   "ab{c}{d}"
 *
 * this returns
 *
 *   [StrLit("ab"); StrRef(Id("c")); StrLit(""); StrRef(Id("d")); StrLit("")]
 *)

and str_lit cs ls = parse
  (* End of string *)
  '"'              { let s = Utils.str_implode (List.rev cs)
                     in List.rev (Ast.StrLit(s) :: ls) }
  
  (* Escapes we convert *)
| '\\' 'n'         { str_lit ('\n' :: cs) ls lexbuf }
| '\\' 't'         { str_lit ('\t' :: cs) ls lexbuf }

  (* Escapes that pass through *)
| '\\' ('\n' as c) { newline lexbuf ; str_lit (c :: cs) ls lexbuf }
| '\\' (_ as c)    { str_lit (c :: cs) ls lexbuf }

  (* Identifier start *)
| '{'              { let s = Utils.str_implode (List.rev cs)
                     in str_id [] (Ast.StrLit(s) :: ls) lexbuf }

  (* Regular characters *)
| '\n' as c        { newline lexbuf ; str_lit (c :: cs) ls lexbuf }
| _ as c           { str_lit (c :: cs) ls lexbuf }

and str_id cs ls = parse
  (* End of identifier *)
  '}'                { str_lit cs ls lexbuf }

| (* Identifier characters *)
  identifier as lit { let p = pos lexbuf (String.length lit) in
                      str_id cs (Ast.StrRef(Ast.Id(lit, p)) :: ls) lexbuf }

  (* Error *)
| _ as c { let ce = Char.escaped c in
           Utils.pos_error (pos lexbuf 1)
             ("Nested string identifier expected; found: " ^ ce) }
