%{
  (* Called in rules - return the start of the given item *)
  let pos n = let pos = rhs_start_pos n in
              Utils.lexpos pos

  (* Called during parse errors - show the line number *)
  let parse_error m =
    print_endline ("Error: " ^ m ^ " on line "
                     ^ (string_of_int !Utils.line_num))

%}

%token EOF

/* Operators and punctuation */

%token RCURLY LCURLY RPAREN LPAREN RSQUARE LSQUARE
%token ARROW DOT DOTDOT SEMI TILDE PERCENT
%token ASSIGN COMMA DIVIDE DIVIDEEQ EQ
%token GT GTEQ LT LTEQ MINUS MINUSEQ NOTEQ
%token PLUS PLUSEQ TIMES TIMESEQ

/* Types */

%token BOOLEAN CARD CARDLIST DECK NUMBER
%token PLAYER PLAYERLIST RANK RANKLIST STRING
%token SUIT SUITLIST TEAM TEAMLIST

%token ACTION AREA GAME ORDERING RULE

/* Keywords */

%token ALL ASK AT BE BY CANPLAY DEAL
%token ELSE ELSEIF FACEDOWN FACEUP FOR FOREVER FROM
%token IF IS LABEL LABELED LEAVING LET MESSAGE
%token OF ORDER PLAY REQUIRES ROTATE
%token SHUFFLE SKIP SPREADOUT SQUAREDUP
%token STARTING SUIT TO WINNER

%token AND OR NOT IN DEFINED
%token PLAYERS STANDARD TEAMS

/* Literals */

%token ACE KING QUEEN JACK
%token CLUBS HEARTS SPADES DIAMONDS
%token TRUE FALSE
%token <int> NUMLIT
%token <Ast.str_lit> STRINGLIT
%token <Ast.id> ID

/* Precedence and associativity */

%left OR
%left AND
%nonassoc NOT
%nonassoc EQ NOTEQ LT LTEQ GT GTEQ IN
%left PLUS MINUS
%left TIMES DIVIDE
%left COMMA
%nonassoc DOTDOT
%left ARROW

/* Start symbol and type */

%start game
%type <Ast.game> game

%%

/* game */
game: gameheading decls EOF
        { (fst $1, snd $1, List.rev $2, pos 1) }

/* (string, ptcount) */
gameheading: GAME STRINGLIT REQUIRES count DOT  { ($2, $4) }

/* pcount */
count: playernum PLAYERS          { Ast.PlayerCount($1, pos 1) }
     | playernum TEAMS OF NUMLIT  { Ast.TeamCount($1, $4, pos 1) }

/* int list */
playernum: NUMLIT TO NUMLIT  { Utils.int_range $1 $3 }
         | playernumlist     { List.rev $1 }

/* int list reversed */
playernumlist: NUMLIT                   { [$1] }
             | playernumlist OR NUMLIT  { $3 :: $1 }


/* Declarations */

/* decl list reversed */
decls: /* Empty */  { [] }
     | decls decl   { $2 :: $1 }

/* decl */
decl: vardecl     { $1 }
    | areadecl    { $1 }
    | actiondecl  { $1 }
    | ruledecl    { $1 }
    | orderdecl   { $1 }

/* decl */
vardecl: dtype ID DOT              { Ast.Var($1, $2, Ast.EmptyExpr, pos 1) }
       | dtype ID ASSIGN expr DOT  { Ast.Var($1, $2, $4, pos 1) } 

/* decl */
areadecl: AREA ID LABELED STRINGLIT DOT
            { Ast.Area($2, $4, [], pos 1) }

        | AREA ID LABELED STRINGLIT IS areaopts DOT
            { Ast.Area($2, $4, List.rev $6, pos 1) }

/* areaopts reversed */
areaopts: areaopt                 { [$1] }
        | areaopts COMMA areaopt  { $3 :: $1 }

/* areaopt */
areaopt: FACEDOWN   { (Ast.Facedown,  pos 1) }
       | FACEUP     { (Ast.Faceup,    pos 1) }
       | SQUAREDUP  { (Ast.Squaredup, pos 1) }
       | SPREADOUT  { (Ast.Spreadout, pos 1) }

/* decl */
actiondecl: ACTION ID block  { Ast.Action($2, $3, pos 1) }

/* decl */
ruledecl: RULE ID LPAREN ID COMMA ID COMMA ID RPAREN ASSIGN expr DOT
            { Ast.Rule($2, [$4; $6; $8], $11, pos 1) }

/* decl */
orderdecl: ORDERING ID LPAREN ID RPAREN ASSIGN expr DOT
            { Ast.Ordering($2, $4, $7, pos 1) }


/* Statements */

/* block */
block: LCURLY stmts RCURLY  { List.rev $2 }

/* stmt list reversed */
stmts: /* Empty */ { [] }
     | stmts stmt  { $2 :: $1 }

/* stmt */
stmt: askstmt       { $1 }
    | assignstmt    { $1 }
    | compoundstmt  { $1 }
    | dealstmt      { $1 }
    | foreverstmt   { $1 }
    | forstmt       { $1 }
    | ifstmt        { $1 }
    | invokestmt    { $1 }
    | labelstmt     { $1 }
    | letstmt       { $1 }
    | messagestmt   { $1 }
    | orderstmt     { $1 }
    | playstmt      { $1 }
    | rotatestmt    { $1 }
    | shufflestmt   { $1 }
    | skipstmt      { $1 }
    | winnerstmt    { $1 }

/* stmt */
askstmt: ASK ref questionblock  { Ast.Ask($2, $3, pos 1) }

/* question list */
questionblock: LCURLY questions RCURLY  { List.rev $2 }

/* question list reversed */
questions: question            { [$1] }
         | questions question  { $2 :: $1 }

/* question */
question: STRINGLIT block          { Ast.Uncond($1, $2, pos 1) }
        | STRINGLIT IF expr block  { Ast.Cond($1, $3, $4, pos 1) }

/* stmt */
assignstmt: ref ASSIGN expr DOT  { Ast.Assign($1, $3, pos 1) }

/* stmt */
compoundstmt: ref PLUSEQ   expr DOT
                { Ast.Compound($1, Ast.Plus, $3, pos 1) }

            | ref MINUSEQ  expr DOT
                { Ast.Compound($1, Ast.Minus, $3, pos 1) }

            | ref TIMESEQ  expr DOT
                { Ast.Compound($1, Ast.Times, $3, pos 1) }

            | ref DIVIDEEQ expr DOT
                { Ast.Compound($1, Ast.Divide, $3, pos 1) }

/* stmt */
dealstmt: DEAL ALL FROM ref TO ref DOT
            { Ast.Deal(Ast.EmptyExpr, $4, $6, pos 1) }

        | DEAL expr FROM ref TO ref DOT
            { Ast.Deal($2, $4, $6, pos 1) }

/* stmt */
foreverstmt: FOREVER block  { Ast.Forever($2, pos 1) }

/* stmt */
forstmt: FOR ID IN expr block
           { Ast.For($2, $4, Ast.EmptyExpr, $5, pos 1) }

       | FOR ID IN expr STARTING AT expr block
           { Ast.For($2, $4, $7, $8, pos 1) }


/* stmt */
ifstmt: IF expr block elseifs elseopt
          { Ast.If([($2, $3)] @ (List.rev $4) @ $5, pos 1) }

/* (expr, block) list reversed */
elseifs: /* Empty */                { [] }
       | elseifs ELSEIF expr block  { ($3, $4) :: $1 }

/* (expr, block) list */
elseopt: /* Empty */ { [] }
       | ELSE block  { [(Ast.EmptyExpr, $2)] }

/* stmt */
invokestmt: ID LPAREN RPAREN DOT  { Ast.Invoke($1, pos 1) }

/* stmt */
labelstmt: LABEL ID DOT  { Ast.Label($2, pos 1) }

/* stmt */
letstmt: LET ID BE expr DOT  { Ast.Let($2, $4, pos 1) }

/* stmt */
messagestmt: MESSAGE expr DOT
               { Ast.Message(Ast.EmptyRef, $2, pos 1) }

           | MESSAGE ref expr DOT
               { Ast.Message($2, $3, pos 1) }

/* stmt */
orderstmt: ORDER ref BY ID DOT  { Ast.Order($2, $4, pos 1) }

/* stmt */
playstmt: PLAY ID FROM ref TO ref DOT  { Ast.Play($2, $4, $6, pos 1) }

/* stmt */
rotatestmt: ROTATE ref DOT  { Ast.Rotate($2, pos 1) }

/* stmt */
shufflestmt: SHUFFLE ref DOT  { Ast.Shuffle($2, pos 1) }

/* stmt */
skipstmt: SKIP TO ID DOT  { Ast.Skip($3, pos 1) }

/* stmt */
winnerstmt: WINNER ref DOT  { Ast.Winner($2, pos 1) }


/* Declarable Types */

/* dtype */
dtype: BOOLEAN     { Ast.Boolean }
     | CARD        { Ast.Card }
     | CARDLIST    { Ast.CardList }
     | DECK        { Ast.Deck }
     | NUMBER      { Ast.Number }
     | PLAYER      { Ast.Player }
     | PLAYERLIST  { Ast.PlayerList }
     | RANK        { Ast.Rank }
     | RANKLIST    { Ast.RankList }
     | STRING      { Ast.String }
     | SUIT        { Ast.Suit }
     | SUITLIST    { Ast.SuitList }
     | TEAM        { Ast.Team }
     | TEAMLIST    { Ast.TeamList }


/* Expressions */

/* expr */
expr: LPAREN expr RPAREN  { $2 }
    | expr PLUS expr      { Ast.Binary($1, Ast.Plus, $3, pos 2) }
    | expr MINUS expr     { Ast.Binary($1, Ast.Minus, $3, pos 2) }
    | expr TIMES expr     { Ast.Binary($1, Ast.Times, $3, pos 2) }
    | expr DIVIDE expr    { Ast.Binary($1, Ast.Divide, $3, pos 2) }
    | expr EQ expr        { Ast.Binary($1, Ast.Eq, $3, pos 2) }
    | expr NOTEQ expr     { Ast.Binary($1, Ast.NotEq, $3, pos 2) }
    | expr LT expr        { Ast.Binary($1, Ast.Lt, $3, pos 2) }
    | expr LTEQ expr      { Ast.Binary($1, Ast.LtEq, $3, pos 2) }
    | expr GT expr        { Ast.Binary($1, Ast.Gt, $3, pos 2) }
    | expr GTEQ expr      { Ast.Binary($1, Ast.GtEq, $3, pos 2) }
    | expr AND expr       { Ast.Binary($1, Ast.And, $3, pos 2) }
    | expr OR expr        { Ast.Binary($1, Ast.Or, $3, pos 2) }
    | NOT expr            { Ast.Not($2, pos 1) }
    | expr IN expr        { Ast.Binary($1, Ast.In, $3, pos 2) }
    | DEFINED ref         { Ast.Defined($2, pos 1) }
    
    | CANPLAY ID FROM ref TO ref { Ast.CanPlay($2, $4, $6, pos 1) }

    | rankexpr TILDE suitexpr { Ast.CardExpr($1, pos 2, $3) }
    | PERCENT  TILDE suitexpr { Ast.WildRank(pos 1, pos 2, $3) }
    | rankexpr TILDE PERCENT  { Ast.WildSuit($1, pos 2, pos 3) }
    | PERCENT  TILDE PERCENT  { Ast.WildCard(pos 1, pos 2, pos 3) }

    | list  { Ast.List($1, pos 1) }
    | ref   { Ast.Ref($1, pos 1) }
    | lit   { $1 }

/* expr */
rankexpr:
      rankexpr DOTDOT rankexpr  { Ast.RangeExpr($1, $3, pos 2) }
    | rankexpr COMMA rankexpr   { Ast.SeqExpr($1, $3, pos 2) }
    | ref                       { Ast.Ref($1, pos 1) }
    | ranklit                   { $1 }
    | NUMLIT                    { Ast.NumLit($1, pos 1) }

/* expr */
suitexpr:
      suitexpr COMMA suitexpr  { Ast.SeqExpr($1, $3, pos 2) }
    | ref                      { Ast.Ref($1, pos 1) }
    | suitlit                  { $1 }

/* expr list */
list: LSQUARE listitemsopt RSQUARE  { $2 }

/* expr list */
listitemsopt: /* Empty */ { [] }
            | listitems   { List.rev $1 }

/* expr list reversed */
listitems: expr                 { [$1] }
         | listitems SEMI expr  { $3 :: $1 }

/* ref */
ref: ID                          { Ast.Id($1, pos 1) }
   | ref ARROW ID                { Ast.Prop(Ast.Ref($1, pos 1), $3, pos 3) }
   | LPAREN expr RPAREN ARROW ID { Ast.Prop($2, $5, pos 5) }
   | builtin                     { $1 }

/* ref */
builtin: PLAYERS   { Ast.Players(pos 1) }
       | TEAMS     { Ast.Teams(pos 1) }
       | STANDARD  { Ast.Standard(pos 1) }


/* Literals */

/* expr */
lit: ranklit    { $1 }
   | suitlit    { $1 }
   | STRINGLIT  { Ast.StringLit($1, pos 1) }
   | NUMLIT     { Ast.NumLit($1, pos 1) }
   | TRUE       { Ast.BoolLit(true, pos 1) }
   | FALSE      { Ast.BoolLit(false, pos 1) }

/* expr */
ranklit: ACE     { Ast.RankLit(Ast.Ace, pos 1) }
       | KING    { Ast.RankLit(Ast.King, pos 1) }
       | QUEEN   { Ast.RankLit(Ast.Queen, pos 1) }
       | JACK    { Ast.RankLit(Ast.Jack, pos 1) }

/* expr */
suitlit: CLUBS     { Ast.SuitLit(Ast.Clubs, pos 1) }
       | HEARTS    { Ast.SuitLit(Ast.Hearts, pos 1) }
       | SPADES    { Ast.SuitLit(Ast.Spades, pos 1) }
       | DIAMONDS  { Ast.SuitLit(Ast.Diamonds, pos 1) }
