(*
 * This module defines the interpreter.
 *
 * This interprets the checked AST structure directly
 * by walking it. The full game is run.
 *
 *)

(* A way to communicate the winner and short circut evaluation *)
exception Winner of string

(* A skip label *)
type skipstr = string

(*
 * This represents the state of the game. It contains
 * the value of special global objects like "players"
 * and "teams" and the deck. It also contains the
 * array of local values, which is changed each
 * time an Action is entered. This is similar to an
 * activation record.
 *)
type env = { e_players : Cast.value;
             e_teams   : Cast.value;
             e_deck    : Cast.value list;
             e_locals  : Cast.value array; }


(*
 * Property Accessors
 *)

(* Get a generic property. *)
let prop_get : Cast.t -> int -> Cast.value -> Cast.value =
  fun t n v ->
    match v with
      Cast.ObjVal(o, t') ->
        if t = t' then o.(n)
                  else Utils.ie "Wrong object for property"
    | _ -> Utils.ie "Wrong object for property"

(* Get the name from a player. *)
let player_name : Cast.value -> Cast.value =
  prop_get Cast.Player 0

(* Get the hand from a player. *)
let player_hand : Cast.value -> Cast.value =
  prop_get Cast.Player 1

(* Set a new team for a player. *)
let player_set_team : Cast.value -> Cast.value -> unit =
  fun player team ->
    match player with
      Cast.ObjVal(a, Cast.Player) -> a.(4) <- team
    | _ -> Utils.ie "Setting team on non-player"

(* Get the rank of a card. *)
let card_rank : Cast.value -> Cast.value =
  prop_get Cast.Card 0

(* Get the suit of a card. *)
let card_suit : Cast.value -> Cast.value =
  prop_get Cast.Card 1

(* Get the original rank of a card. *)
let card_orig_rank : Cast.value -> Cast.value =
  prop_get Cast.Card 3

(* Get the original suit of a card. *)
let card_orig_suit : Cast.value -> Cast.value =
  prop_get Cast.Card 4

(* Set the last_played_by field of a card. *)
let card_set_played_by : Cast.value -> Cast.value -> unit =
  fun card player ->
    match card with
      Cast.ObjVal(a, Cast.Card) -> a.(2) <- player
    | _ -> Utils.ie "Setting player on non-card"

(* Get the player list from a team. *)
let team_members : Cast.value -> Cast.value =
  prop_get Cast.Team 0

(* Get the cards from an area. *)
let area_cards : Cast.value -> Cast.value =
  prop_get Cast.Area 1


(*
 * Values
 *)

(* Create a new object from the list of properties. *)
let new_obj : Cast.t -> Cast.value list -> Cast.value =
  fun t props ->
    Cast.ObjVal(Array.of_list props, t)

(* Create a new Player with the given name and defaults for the rest. *)
let new_player : string -> Cast.value =
  fun name ->
    new_obj Cast.Player [Cast.StrVal(name);
                         Cast.ListVal(ref [], Cast.Card);
                         Cast.ListVal(ref [], Cast.Card);
                         Cast.NumVal(0);
                         Cast.UndefVal]

(* Create a new Team with the given Players and defaults for the rest. *)
let new_team : Cast.value list -> Cast.value =
  fun members ->
    new_obj Cast.Team [Cast.ListVal(ref members, Cast.Player);
                       Cast.ListVal(ref [], Cast.Card);
                       Cast.NumVal(0)]

(* Create a new Card with the given Rank and Suit. *)
let new_card : Cast.rank -> Cast.suit -> Cast.value =
  fun rank suit ->
    new_obj Cast.Card [Cast.RankVal(rank);
                       Cast.SuitVal(suit);
                       Cast.UndefVal;
      (* Orig rank *)  Cast.RankVal(rank);
      (* Orig suit *)  Cast.SuitVal(suit)]

(* Return the numeric equivalent of a rank. *)
let rank_num : Cast.rank -> int =
  function
    Ast.Ace      -> 1
  | Ast.King     -> 13
  | Ast.Queen    -> 12
  | Ast.Jack     -> 11
  | Ast.Minor(n) -> n

(* Return the rank equivalent of a number. *)
let num_rank : Cast.pos -> int -> Cast.rank =
  fun p -> function
    1  -> Ast.Ace
  | 11 -> Ast.Jack
  | 12 -> Ast.Queen
  | 13 -> Ast.King
  | n  -> if n >= 2 && n <= 10 then Ast.Minor(n)
            else Utils.pos_error p
                   ("The value " ^ (string_of_int n)
                                 ^ " can't be a Rank")

(* Return a numeric value for a suit. *)
let suit_num : Cast.suit -> int =
  function
    Ast.Clubs -> 1
  | Ast.Hearts -> 2
  | Ast.Spades -> 3
  | Ast.Diamonds -> 4

(* Get a native string from a Cast.StrVal. *)
let as_str : Cast.value -> string =
  function
    Cast.StrVal(s) -> s
  | _ -> Utils.ie "Not a string value"

(* Get a native bool from a Cast.BoolVal. *)
let as_bool : Cast.value -> bool =
  function
    Cast.BoolVal(b) -> b
  | _ -> Utils.ie "Not a boolean value"

(* Get a native int from a Cast.NumVal. *)
let as_num : Cast.value -> int =
  function
    Cast.NumVal(n) -> n
  | Cast.RankVal(r) -> rank_num r
  | _ -> Utils.ie "Not a numeric value"

(* Get a native Ast.rank from a Cast.SuitVal. *)
let as_rank : Cast.pos -> Cast.value -> Ast.rank =
  fun p -> function
    Cast.NumVal(n) -> num_rank p n
  | Cast.RankVal(r) -> r
  | _ -> Utils.ie "Not a rank value"

(* Get a native Ast.suit from a Cast.SuitVal. *)
let as_suit : Cast.value -> Ast.suit =
  function
    Cast.SuitVal(s) -> s
  | _ -> Utils.ie "Not a suit value"

(* Get a list of values from a Cast.ListVal. *)
let as_list : Cast.value -> Cast.value list =
  function
    Cast.ListVal(vs, _) -> !vs
  | _ -> Utils.ie "Not a list value"

(* Convert the rank to a string *)
let string_of_rank : Cast.rank -> string =
  function
    Ast.Ace       -> "Ace"
  | Ast.King      -> "King"
  | Ast.Queen     -> "Queen"
  | Ast.Jack      -> "Jack"
  | Ast.Minor(10) -> "Ten"
  | Ast.Minor(9)  -> "Nine"
  | Ast.Minor(8)  -> "Eight"
  | Ast.Minor(7)  -> "Seven"
  | Ast.Minor(6)  -> "Six"
  | Ast.Minor(5)  -> "Five"
  | Ast.Minor(4)  -> "Four"
  | Ast.Minor(3)  -> "Three"
  | Ast.Minor(2)  -> "Two"
  | _ -> Utils.ie "Invalid rank for string conversion"

(* Convert the suit to a string *)
let string_of_suit : Cast.suit -> string =
  function
    Ast.Clubs    -> "Clubs"
  | Ast.Hearts   -> "Hearts"
  | Ast.Spades   -> "Spades"
  | Ast.Diamonds -> "Diamonds"

(* Convert the value to a string. *)
let rec to_str : Cast.value -> string =
  fun value ->
    match value with
      Cast.BoolVal(b) -> if b then "True" else "False"
    | Cast.NumVal(n)  -> string_of_int n
    | Cast.RankVal(r) -> string_of_rank r
    | Cast.StrVal(s)  -> s
    | Cast.SuitVal(s) -> string_of_suit s
    | Cast.ObjVal(_, Cast.Card) ->
        let r = to_str (card_orig_rank value) in
        let s = to_str (card_orig_suit value) in
        r ^ " of " ^ s
    | Cast.ObjVal(_, Cast.Player) ->
        as_str (player_name value)
    | Cast.ObjVal(_, Cast.Team) ->
        let ps = team_members value in
        let ps = as_list ps in
        let ns = List.map player_name ps in
        "Team (" ^ (Utils.catmap ", " to_str ns) ^ ")"
    | _ -> Utils.ie "String conversion of invalid type"

(* Get a CardList from a Player, Area, or CardList *)
let to_cardlist : Cast.value -> Cast.value =
  fun v ->
    match v with
      Cast.ObjVal(_, Cast.Player) -> player_hand v
    | Cast.ObjVal(_, Cast.Area) -> area_cards v
    | Cast.ListVal(_, Cast.Card) -> v
    | Cast.ListVal(_, Cast.EmptyType) -> v
    | _ -> Utils.ie "Not a cardlist value"

(* Get a CardList List from a Player, Area, CardList, or PlayerList *)
let to_cardlistlist : Cast.value -> Cast.value list =
  fun v ->
    match v with
      Cast.ObjVal(_, Cast.Player)  -> [player_hand v]
    | Cast.ObjVal(_, Cast.Area)    -> [area_cards v]
    | Cast.ListVal(_, Cast.Card)   -> [v]
    | Cast.ListVal(_, Cast.Player) ->
        let players = as_list v in
        List.map player_hand players
    | Cast.ListVal(_, Cast.EmptyType) -> [v]
    | _ -> Utils.ie "Not a cardlistlist value"

(*
 * Get a Cast.value list from either a scalar Cast.value or
 * a list Cast.value.
 *)
let elt_or_list : Cast.value -> Cast.value list =
  function
    Cast.ListVal(l, _) -> !l
  | v -> [v]

(* Get the length of the list. *)
let list_length : 'a list -> Cast.value =
  fun l -> Cast.NumVal(List.length l)

(* Get the first item from the non-empty list. *)
let list_first : 'a list -> Cast.pos -> 'a =
  fun l p ->
    match l with
      [] -> Utils.pos_error p "Can't get item from empty list"
    | h :: _ -> h

(* Get the last item from the non-empty list. *)
let rec list_last : 'a list -> Cast.pos -> 'a =
  fun l p ->
    match l with
      [] -> Utils.pos_error p "Can't get item from empty list"
    | h :: [] -> h
    | _ :: t -> list_last t p

(* Compare two values for structural equality. *)
let rec val_eq : Cast.value -> Cast.value -> bool =
  fun v1 v2 ->
    match (v1, v2) with
      Cast.BoolVal(b1),    Cast.BoolVal(b2)   -> b1 = b2
    | Cast.NumVal(n1),     Cast.NumVal(n2)    -> n1 = n2
    | Cast.NumVal(n),      Cast.RankVal(r)    -> n = (rank_num r)
    | Cast.RankVal(r),     Cast.NumVal(n)     -> (rank_num r) = n
    | Cast.RankVal(r1),    Cast.RankVal(r2)   -> r1 = r2
    | Cast.StrVal(s1),     Cast.StrVal(s2)    -> s1 = s2
    | Cast.SuitVal(s1),    Cast.SuitVal(s2)   -> s1 = s2
    | Cast.ObjVal(o1, _),  Cast.ObjVal(o2, _) -> o1 == o2
    | Cast.ListVal(v1, _), Cast.ListVal(v2, _) ->
        let l1 = List.length !v1 in
        let l2 = List.length !v2 in
        (l1 = l2) && (List.for_all2 val_eq !v1 !v2)
    | Cast.UndefVal, Cast.UndefVal -> true
    | _ -> false

(*
 * Compare two values for 'a in b' equality. This comparison is done
 * between 'a' and an element of 'b'. This is like 'val_eq' but also
 * supports 'rank in card' and 'suit in card' comparisons.
 *)
let rec val_eq_in : Cast.pos -> Cast.value -> Cast.value -> bool =
  fun p v1 v2 ->
    match (v1, v2) with
      Cast.RankVal(r), Cast.ObjVal(o, Cast.Card) ->
        r = (as_rank p (card_rank v2))
    | Cast.NumVal(n),  Cast.ObjVal(o, Cast.Card) ->
        n = (as_num (card_rank v2))
    | Cast.SuitVal(s), Cast.ObjVal(o, Cast.Card) ->
        s = (as_suit (card_suit v2))
    | _ -> val_eq v1 v2

(*
 * Variables and Properties
 *)

(* Get the value from a variable. *)
let run_var : env -> Cast.variable -> Cast.value =
  fun env -> function
    Cast.Global(r, _) -> !r
  | Cast.Local(h, _)  -> env.e_locals.(h)
  | Cast.PlayersVar   -> env.e_players
  | Cast.TeamsVar     -> env.e_teams
  | Cast.StandardVar  -> Cast.ListVal(ref env.e_deck, Cast.Card)

(* Get the value from an object's property. *)
let run_prop : Cast.value -> Cast.prop -> Cast.pos -> Cast.value =
  fun v prop p ->
    match (v, prop) with
      Cast.ObjVal(v, _), Cast.ObjProp(n, _) -> v.(n)
    | Cast.ListVal(v, _), Cast.ListProp(lp, _) ->
        (match lp with
           Cast.Size   -> list_length !v
         | Cast.First  -> list_first !v p
         | Cast.Last   -> list_last !v p
         | Cast.Top    -> list_first !v p
         | Cast.Bottom -> list_last !v p)
    | _ -> Utils.ie "Invalid object property"

(* Store the value in the variable. *)
let set_var : env -> Cast.variable -> Cast.value -> Cast.pos -> unit =
  fun env var value p ->
    match var with
      Cast.Global(r, _) -> r := value
    | Cast.Local(h, _)  -> env.e_locals.(h) <- value
    | _ -> Utils.pos_error p "Can't assign to a read-only variable"

(* Set the value of an object's property. *)
let set_prop : Cast.value -> Cast.prop -> Cast.value -> Cast.pos -> unit =
  fun obj prop value p ->
    match (obj, prop) with
      Cast.ObjVal(v, _), Cast.ObjProp(n, _) -> v.(n) <- value
    | Cast.ListVal(v, _), Cast.ListProp(lp, _) ->
        Utils.pos_error p "Can't assign read-only list properties"
    | _ -> Utils.ie "Invalid object property"

(* Set the value in the mutable list. *)
let set_list : Cast.value -> Cast.value list -> unit =
  fun list value ->
    match list with
      Cast.ListVal(v, _) -> v := value
    | _ -> Utils.ie "Setting non-list"


(*
 * The Deck
 *)

(* Return the unique index in a 52-card deck for this rank/suit combination *)
let card_idx : Cast.rank -> Cast.suit -> int =
  fun rank suit ->
    let n1 = rank_num rank in
    let n2 = suit_num suit in
    (n2 - 1) * 13 + (n1 - 1)

(* Create a new 52-card deck as a list of Card Cast.values. *)
let new_deck : unit -> Cast.value list =
  fun _ ->
    let cards = Array.make 52 Cast.UndefVal in
    let f rank suit =
      let idx = card_idx rank suit in
      cards.(idx) <- new_card rank suit
    in
    let ranks = List.map (num_rank (-1, -1)) (Utils.int_range 1 13) in
    let suits = [Ast.Clubs; Ast.Hearts; Ast.Spades; Ast.Diamonds] in
    ignore (Utils.xmap f ranks suits);
    Array.to_list cards

(* Given a rank and a suit, return the named card from the environment. *)
let get_card : env -> Cast.pos -> Cast.value -> Cast.value -> Cast.value =
  fun env p rank suit ->
    let idx = card_idx (as_rank p rank) (as_suit suit) in
    List.nth env.e_deck idx


(*
 * Expressions and References
 *)

(* Combine two values using a binary operator. *)
let rec run_binary : Cast.value -> Cast.op -> Cast.value
                  -> Cast.pos -> Cast.value =
  fun v1 op v2 p ->
    match op with
      Ast.Plus   -> Cast.NumVal((as_num v1) + (as_num v2))
    | Ast.Minus  -> Cast.NumVal((as_num v1) - (as_num v2))
    | Ast.Times  -> Cast.NumVal((as_num v1) * (as_num v2))
    | Ast.Divide ->
        let n1 = as_num v1 in
        let n2 = as_num v2 in
        if n2 = 0 then Utils.pos_error p "Division by zero"
                  else Cast.NumVal(n1 / n2)
    | Ast.Eq     -> Cast.BoolVal(val_eq v1 v2)
    | Ast.NotEq  -> Cast.BoolVal(not(val_eq v1 v2))
    | Ast.Lt     -> Cast.BoolVal((as_num v1) <  (as_num v2))
    | Ast.LtEq   -> Cast.BoolVal((as_num v1) <= (as_num v2))
    | Ast.Gt     -> Cast.BoolVal((as_num v1) >  (as_num v2))
    | Ast.GtEq   -> Cast.BoolVal((as_num v1) >= (as_num v2))
    | Ast.In     ->
        let l2 = as_list v2 in
        let b = List.exists (val_eq_in p v1) l2 in
        Cast.BoolVal(b)
        
    (* And and Or are done in run_binary_short *)
    | _ -> Utils.ie "Unexpected binary op"

(* Like run_binary but does short-circuit operators first *)
and run_binary_short : env -> Cast.expr -> Cast.op -> Cast.expr
                    -> Cast.pos -> Cast.value =
  fun env e1 op e2 p ->
    match op with
      Ast.And ->
        let v1 = run_expr env e1 in
        let b1 = as_bool v1 in
        if not b1 then
          Cast.BoolVal(false)
        else
          let v2 = run_expr env e2 in
          let b2 = as_bool v2 in
          Cast.BoolVal(b2)
    | Ast.Or ->
        let v1 = run_expr env e1 in
        let b1 = as_bool v1 in
        if b1 then
          Cast.BoolVal(true)
        else
          let v2 = run_expr env e2 in
          let b2 = as_bool v2 in
          Cast.BoolVal(b2)
    | _ ->
      let v1 = run_expr env e1 in
      let v2 = run_expr env e2 in
      run_binary v1 op v2 p

(* Run a string literal, producing a string value. *)
and run_str_lit : env -> Cast.str_lit -> Cast.value =
  fun env sl ->
    let part_to_str = function
        Cast.StrRef(r) -> to_str (run_ref env r)
      | Cast.StrLit(s) -> s
    in
    let str = Utils.catmap "" part_to_str sl in
    Cast.StrVal(str)

(* Run a reference, producing a value. *)
and run_ref : env -> Cast.reference -> Cast.value =
  fun env -> function
    Cast.Var(v, _) ->
      run_var env v
  | Cast.Prop(e, prop, p) ->
      let obj = run_expr env e in
      run_prop obj prop p
  | Cast.EmptyRef ->
      Utils.ie "Executing EmptyRef"

(* Store the value in the reference. *)
and set_ref : env -> Cast.reference -> Cast.value -> unit =
  fun env ref value ->
    match ref with
      Cast.Var(v, p) ->
        set_var env v value p
    | Cast.Prop(e, prop, p) ->
        let obj = run_expr env e in
        set_prop obj prop value p
    | Cast.EmptyRef ->
        Utils.ie "Setting EmptyRef"

(* Run a variable declaration, initializing the variable. *)
and run_var_decl : env -> Cast.var_decl -> unit =
  fun env -> function
    Cast.VarDecl(v, e, p) ->
      let value = if e = Cast.EmptyExpr
                    then Cast.UndefVal
                    else run_expr env e
      in set_var env v value p
  | Cast.AreaDecl(v, sl, facedown, squaredup, p) ->
      let name = run_str_lit env sl in
      let cards = Cast.ListVal(ref [], Cast.Card) in
      let facedown = Cast.BoolVal(facedown) in
      let squaredup = Cast.BoolVal(squaredup) in
      let value = new_obj Cast.Area [name; cards; facedown; squaredup] in
      set_var env v value p

(* Run an expression, producing a value. *)
and run_expr : env -> Cast.expr -> Cast.value =
  fun env -> function
    Cast.Not(e, _) ->
      let b = as_bool (run_expr env e) in
      Cast.BoolVal(not b)
  | Cast.Binary(e1, op, e2, p) ->
      run_binary_short env e1 op e2 p
  | Cast.Defined(r, _) ->
      let v = run_ref env r in
      let b = if v = Cast.UndefVal then false else true in
      Cast.BoolVal(b)
  | Cast.CanPlay(rule, r1, r2, _) ->
      let player = run_ref env r1 in
      let dest = run_ref env r2 in
      let cl = run_rule env rule player dest in
      let b = (List.length cl) <> 0 in
      Cast.BoolVal(b)
  | Cast.CardExpr(e1, p, e2) ->
      let v1 = run_expr env e1 in
      let v2 = run_expr env e2 in
      let ranks = elt_or_list v1 in
      let suits = elt_or_list v2 in
      let cards = Utils.xmap (get_card env p) ranks suits in
      (match cards with
         h :: [] -> h
       | _ -> Cast.ListVal(ref cards, Cast.Card))
  | Cast.RangeExpr(e1, e2, p) ->
      let v1 = run_expr env e1 in
      let v2 = run_expr env e2 in
      let n1 = as_num v1 in
      let n2 = as_num v2 in
      let range = Utils.int_range n1 n2 in
      let f n = Cast.RankVal(num_rank p n) in
      let vs = List.map f range in
      Cast.ListVal(ref vs, Cast.Rank)
  | Cast.RankSeqExpr(e1, e2, _) ->
      let v1 = run_expr env e1 in
      let v2 = run_expr env e2 in
      let l1 = elt_or_list v1 in
      let l2 = elt_or_list v2 in
      let all = l1 @ l2 in
      Cast.ListVal(ref all, Cast.Rank)
  | Cast.SuitSeqExpr(e1, e2, _) ->
      let v1 = run_expr env e1 in
      let v2 = run_expr env e2 in
      let l1 = elt_or_list v1 in
      let l2 = elt_or_list v2 in
      let all = l1 @ l2 in
      Cast.ListVal(ref all, Cast.Suit)
  | Cast.WildRank(_, _, e) ->
      let v = run_expr env e in
      let suits = elt_or_list v in
      let suits = List.map as_suit suits in
      let f card =
        let suit = as_suit (card_suit card) in
        List.mem suit suits
      in
      let matches = List.filter f env.e_deck in
      Cast.ListVal(ref matches, Cast.Card)
  | Cast.WildSuit(e, p, _) ->
      let v = run_expr env e in
      let ranks = elt_or_list v in
      let ranks = List.map (as_rank p) ranks in
      let f card =
        let rank = as_rank p (card_rank card) in
        List.mem rank ranks
      in
      let matches = List.filter f env.e_deck in
      Cast.ListVal(ref matches, Cast.Card)
  | Cast.WildCard(_, _, _) ->
      Cast.ListVal(ref env.e_deck, Cast.Card)
  | Cast.List(es, t, _) ->
      let vals = List.map (run_expr env) es in
      (* Some of vals are elements and some are lists *)
      let vals = List.map elt_or_list vals in
      let vals = List.concat vals in
      Cast.ListVal(ref vals, t)
  | Cast.Ref(r, p) ->
      let v = run_ref env r in
      if v = Cast.UndefVal then
        Utils.pos_error p "Undefined value";
      v
  | Cast.Literal(v, _) -> v
  | Cast.StrLiteral(sl, _) -> run_str_lit env sl
  | Cast.EmptyExpr -> Utils.ie "Executing EmptyExpr"

(* Return the cards that can be played using a rule. *)
and run_rule : env -> Cast.rule -> Cast.value -> Cast.value
            -> Cast.value list =
  fun env (e, _) player dest ->
    let hand   = player_hand player in
    let cards  = as_list hand in
    let dest   = to_cardlist dest in
    let f card =
      let locals = Array.of_list [player; card; dest] in
      let env = { env with e_locals = locals } in
      let v = run_expr env e in
      as_bool v
    in
    List.filter f cards

(* Return the Cast.value list of an ordering *)
and run_ordering : env -> Cast.ordering -> Cast.value -> Cast.value list =
  fun env (e, _) cards ->
    let locals = Array.of_list [cards] in
    let env = { env with e_locals = locals } in
    let ord = run_expr env e in
    as_list ord


(*
 * Statements
 *)

(*
 * Move a card from one CardList to another CardList.
 *)
let move_card : Cast.value -> Cast.value -> Cast.value -> unit =
  fun card src dst ->
    (* Remove the card from src *)
    let cl1 = as_list src in
    let cl1 = Utils.lremove val_eq card cl1 in
    set_list src cl1;
    (* Store the card in dst *)
    let cl2 = as_list dst in
    let cl2 = card :: cl2 in
    set_list dst cl2

(*
 * Convert a list of questions to a list of string/block pairs.
 * Only include questions where the predicate is true, if there
 * is a predicate.
 *)
let run_questions : env -> Cast.question list -> (string * Cast.block) list =
  fun env qs ->
    let f res = function
        Cast.Uncond(sl, block, _) ->
          let s = as_str (run_str_lit env sl) in
          (s, block) :: res
      | Cast.Cond(sl, e, block, _) ->
          let s = as_str (run_str_lit env sl) in
          let b = as_bool (run_expr env e) in
          if b then (s, block) :: res else res
    in
    let qs = List.fold_left f [] qs in
    List.rev qs

(*
 * Run a single statement, threading the skip string.
 *
 * If the skip string is the empty string, then the
 * statement is run as normal. A new skip string will
 * be produced if this is a "skip" statement - otherwise,
 * the empty skip string will be the result.
 *
 * If the skip string is not the empty string, a skip
 * has previously been executed, and this statement is
 * not run unless it is the matching label.
 *)
let rec run_stmt : env -> skipstr -> Cast.stmt -> skipstr =
  fun env skip stmt ->
    match (skip, stmt) with
      "", Cast.Ask(r, qs, _) ->
            let player = run_ref env r in
            let name = as_str (player_name player) in
            let choices = run_questions env qs in
            let msg = name ^ ": Choose an action:" in
            let block = Ui.lchoice msg choices in
            run_block env block
    | "", Cast.Assign(r, e, p) ->
            let v = run_expr env e in
            set_ref env r v; ""
    | "", Cast.Compound(r, op, e, p) ->
            let v1 = run_ref env r in
            let v2 = run_expr env e in
            let v = run_binary v1 op v2 p in
            set_ref env r v; ""
    | "", Cast.DealCard(e, r1, r2, _) ->
            let card = run_expr env e in
            let v1 = run_ref env r1 in
            let cl1 = to_cardlist v1 in
            let v2 = run_ref env r2 in
            let cl2 = to_cardlist v2 in
            move_card card cl1 cl2; ""
    | "", Cast.DealCards(e, r1, r2, _) ->
            let v1 = run_ref env r1 in
            let cl1 = to_cardlist v1 in
            let v2 = run_ref env r2 in
            let cl2 = to_cardlistlist v2 in
            let len = List.length cl2 in
            let n = if e = Cast.EmptyExpr
                      then List.length (as_list cl1)
                      else len * (as_num (run_expr env e))
            in
            (* deal cards one at a time to each cardlist in cl2 *)
            for i = 1 to n do
              let cards = as_list cl1 in
              let card = List.hd cards in
              let dest = List.nth cl2 (i mod len) in
              move_card card cl1 dest
            done; ""
    | "", Cast.Forever(block, _) ->
            let new_skip = run_block env block in
            run_stmt env new_skip stmt
    | "", Cast.For(v, e1, e2, block, p) ->
            let v1 = run_expr env e1 in
            let v1 = as_list v1 in
            (* Permute the list so e2 is first *)
            let permute l e =
              if e = Cast.EmptyExpr then l else
                let v = run_expr env e in
                try
                  let (front, back) = Utils.split val_eq v l in
                  back @ front
                with Not_found ->
                  Utils.pos_error p "Starting element is not in list"
            in
            let v1 = permute v1 e2 in
            let f skipstr value =
              if skipstr <> "" then skipstr else
                (set_var env v value p;
                 run_block env block)
            in
            List.fold_left f "" v1
    | "", Cast.If(ess, _) ->
            let f (expr, _) =
              expr = Cast.EmptyExpr || as_bool (run_expr env expr)
            in
            (try
              let (_, block) = List.find f ess in
              run_block env block
            with Not_found -> "")
    | "", Cast.Invoke(a, _) ->
            run_action env a; ""
    | "", Cast.Label(_, _) -> ""
    | s,  Cast.Label(t, _) ->
            if s = t then "" else s   
    | "", Cast.Let(v, _) ->
            run_var_decl env v; ""
    | "", Cast.Message(r1, e, _) ->
            let s = as_str (run_expr env e) in
            if r1 = Cast.EmptyRef then
              Ui.show_all s
            else
              (let p = player_name (run_ref env r1) in
               let p = as_str p in
               Ui.show p s);
            ""
    | "", Cast.Order(r, o, _) ->
            let cards = run_ref env r in
            let ord = run_ordering env o cards in
            let cl = as_list cards in
            let cl = Utils.order val_eq cl ord in
            set_list cards cl; ""
    | "", Cast.Play(rule, r1, r2, p) ->
            let player = run_ref env r1 in
            let dest = to_cardlist (run_ref env r2) in
            let cl = run_rule env rule player dest in
            if cl = [] then
              Utils.pos_error p "No cards match the given rule";
            (* Ask the user which card to play *)
            let name = as_str (player_name player) in
            let pairs = List.map (fun c -> (to_str c, c)) cl in
            let msg = name ^ ": Choose a card:" in
            let card = Ui.lchoice msg pairs in
            (* Move the card from the player's hand to the destination *)
            let hand = player_hand player in
            move_card card hand dest;
            card_set_played_by card player;
            let msg = name ^ " played " ^ (to_str card) ^ "." in
            Ui.show_all msg;
            ""
    | "", Cast.Rotate(r, _) ->
            let list = run_ref env r in
            let v = as_list list in
            let v = match v with
                [] -> v
              | [_] -> v
              | h::t -> t @ [h]
            in
            set_list list v; ""
    | "", Cast.Shuffle(r, _) ->
            let list = run_ref env r in
            let v = as_list list in
            let v = Utils.shuffle v in
            set_list list v; ""
    | "", Cast.Skip(s, _) -> s
    | "", Cast.Winner(r, _) ->
            let w = run_ref env r in
            let s = to_str w in
            raise(Winner(s))
    | _ -> skip

(* Run a block by running all of its statements. *)
and run_block : env -> Cast.block -> skipstr =
  fun env sl -> List.fold_left (run_stmt env) "" sl

(*
 * Run an action.
 *
 * This sets up a new "activation record" with storage for
 * the local variables in the action, and then runs
 * the block.
 *)
and run_action : env -> Cast.action -> unit =
  fun env (num, block, _) ->
    let env = { env with e_locals = Array.make num Cast.UndefVal; }
    in ignore (run_block env block)


(*
 * The Game
 *)

(*
 * Initialize the players by asking the user for
 * the number of players and the names of each.
 *)
let init_players : Cast.ptcount -> env =
  function
    Ast.PlayerCount(il, _) ->
      let np = Ui.ichoice "How many players are playing?" il in
      let msg = "This game has " ^ (string_of_int np) ^ " players." in
      Ui.show_all msg;
      let players = Array.make np Cast.UndefVal in
      for i = 1 to np do
        let msg = "Enter a name for Player " ^ (string_of_int i) ^ ":" in
        let name = Ui.istring msg in
        players.(i - 1) <- new_player name
      done;
      let players = Array.to_list players in
      { e_players = Cast.ListVal(ref players, Cast.Player);
        e_teams   = Cast.UndefVal;
        e_deck    = new_deck ();
        e_locals  = Array.make 0 Cast.UndefVal; }
  | Ast.TeamCount(il, np, _) ->
      let msg = "This game has " ^ (string_of_int np) ^ " players per team." in
      Ui.show_all msg;
      let nt = Ui.ichoice "How many teams are playing?" il in
      let msg = "This game has " ^ (string_of_int nt) ^ " teams." in
      Ui.show_all msg;
      let all_players = Array.make (np * nt) Cast.UndefVal in
      let teams = Array.make nt Cast.UndefVal in
      for ti = 1 to nt do
        let players = Array.make np Cast.UndefVal in
        for pi = 1 to np do
          let msg = "Enter a name for Player "
                      ^ (string_of_int pi) ^ " on team "
                      ^ (string_of_int ti) ^ ":" in
          let name = Ui.istring msg in
          let idx = (pi - 1) * nt + (ti - 1) in
          let player = new_player name in
          players.(pi - 1) <- player;
          all_players.(idx) <- player
        done;
        let team = new_team (Array.to_list players) in
        (* Update each player->team to point to this team *)
        Array.iter (fun p -> player_set_team p team) players;
        teams.(ti - 1) <- team
      done;
      let all_players = Array.to_list all_players in
      let teams = Array.to_list teams in
      { e_players = Cast.ListVal(ref all_players, Cast.Player);
        e_teams   = Cast.ListVal(ref teams, Cast.Team);
        e_deck    = new_deck ();
        e_locals  = Array.make 0 Cast.UndefVal; }

(*
 * Initialize each global variable by running
 * the initializer expression and storing the
 * value into the global.
 *)
let init_globals : env -> Cast.var_decl list -> unit =
  fun env vars -> List.iter (run_var_decl env) vars

(*
 * Run the game.
 * 
 * This initializes the game, runs the main action, and
 * declares a winner.
 *)
let run_game : Cast.game -> unit =
  fun (name, count, vars, main, p) ->
    let env = { e_players = Cast.UndefVal;
                e_teams   = Cast.UndefVal;
                e_deck    = [];
                e_locals  = Array.make 0 Cast.UndefVal; } in
    let name = run_str_lit env name in
    let name = as_str name in
    Ui.show_all "------------------------------------------";
    Ui.show_all ("Welcome to " ^ name);
    Ui.show_all "------------------------------------------";
    let env = init_players count in
    init_globals env vars;
    try run_action env main;
        Ui.show_all "No winner has been declared."
    with Winner(s) -> print_endline ("The game was won by " ^ s)
