(*
 * This module defines the checked AST structure.
 *
 * This is like the AST but with all identifiers
 * resolved and all types checked.
 *
 * This represents a semantically valid program and builds
 * on a lot of the AST nodes.
 *)

(**********)
(* Basics *)

(* Handle - Refers to a variable's location in the environment *)
type handle = int

(* Offset - Refers to a property in an Object value's prop array *)
type offset = int

(* Position *)
type pos = Ast.pos

(* Expression and Types *)
type t = Area | Boolean | Card | Number
       | Player | Rank | String | Suit | Team
       | L of t
       | EmptyType

let str_type = function
    Area      -> "Area"
  | Boolean   -> "Boolean"
  | Card      -> "Card"
  | Number    -> "Number"
  | Player    -> "Player"
  | Rank      -> "Rank"
  | String    -> "String"
  | Suit      -> "Suit"
  | Team      -> "Team"
  
  | L(Boolean) -> "BooleanList"
  | L(Card)    -> "CardList"
  | L(Number)  -> "NumberList"
  | L(Player)  -> "PlayerList"
  | L(Rank)    -> "RankList"
  | L(String)  -> "StringList"
  | L(Suit)    -> "SuitList"
  | L(Team)    -> "TeamList"

  | EmptyType -> "EmptyType"
  | _ -> raise(Failure("Internal Error: Unexpected type"))


(* Rank and Suit *)
type rank = Ast.rank
type suit = Ast.suit

(* Values *)
type value =
    BoolVal of bool
  | NumVal of int
  | StrVal of string
  | RankVal of rank
  | SuitVal of suit
  | ObjVal of value array * t      (* Properties *)
  | ListVal of value list ref * t  (* Elements *)
  | UndefVal


(*********************************************)
(* Expressions, References, and Declarations *)

type op = Ast.op

type expr =
  | Not of expr * pos
  | Binary of expr * op * expr * pos
  | Defined of reference * pos
  | CanPlay of rule * reference * reference * pos
  | CardExpr of expr * pos * expr
  | RangeExpr of expr * expr * pos
  | RankSeqExpr of expr * expr * pos
  | SuitSeqExpr of expr * expr * pos
  | WildRank of pos * pos * expr
  | WildSuit of expr * pos * pos
  | WildCard of pos * pos * pos
  | List of expr list * t * pos
  | Ref of reference * pos
  | Literal of value * pos
  | StrLiteral of str_lit * pos
  | EmptyExpr

and reference =
    Var of variable * pos
  | Prop of expr * prop * pos
  | EmptyRef

and variable =
    Global of value ref * t
  | Local of handle * t
  | PlayersVar
  | TeamsVar
  | StandardVar

and lprop = Size | First | Last | Top | Bottom

and prop =
    ObjProp of offset * t
  | ListProp of lprop * t

and str_part =
    StrRef of reference
  | StrLit of string

and str_lit = str_part list

and rule = expr * pos

and ordering = expr * pos


(****************)
(* Declarations *)

type var_decl =
    VarDecl of variable * expr * pos
  | AreaDecl of variable * str_lit * bool * bool * pos


(**************)
(* Statements *)

type stmt =
    Ask of reference * question list * pos
  | Assign of reference * expr * pos

    (* Compound: Op can only be plus, minus, times, or divide *)
  | Compound of reference * op * expr * pos

    (* DealCard: First expr is Card *)
  | DealCard of expr * reference * reference * pos

    (* DealCards: First expr empty for "all" or Numeric *)
  | DealCards of expr * reference * reference * pos

  | Forever of block * pos
  
    (* For: Second expr could be empty - start at beginning of list *)
  | For of variable * expr * expr * block * pos
  
    (* If: Each expr/block pair is if..elseif..elseif.
           Last pair may have EmptyExpr for unconditional else. *)
  | If of (expr * block) list * pos

  | Invoke of action * pos
  | Label of string * pos
  | Let of var_decl * pos
  
    (* Message: Empty ref means message to all *)
  | Message of reference * expr * pos

  | Order of reference * ordering * pos
  | Play of rule * reference * reference * pos
  | Rotate of reference * pos
  | Shuffle of reference * pos
  | Skip of string * pos
  | Winner of reference * pos

and question =
    Uncond of str_lit * block * pos
  | Cond of str_lit * expr * block * pos

and block = stmt list

  (* Action: Number of locals and a body *)
and action = int * block * pos


(************)
(* The Game *)

type ptcount = Ast.ptcount

(* Name, number of players, global variables, and the main action *)
type game = str_lit * ptcount * var_decl list * action * pos
