(*
 * This module defines the type of the basic AST
 * that is created by the parser.
 *
 * There are also utilities to turn AST nodes
 * into strings that can be re-parsed back into
 * AST nodes.
 *
 * This represents a syntactically valid program.
 *)

(***********)
(* Helpers *)

let str_indent ind = String.make ind ' '


(**********)
(* Basics *)

(* Position: Line and Column *)
type pos = int * int

(* Identifiers are strings in the AST *)
type id = string


(***************)
(* Expressions *)

(* Ranks *)
(* Minor must be 2..10 *)
type rank = Ace | King | Queen | Jack | Minor of int

(* Unparse a rank *)
let str_rank = function
    Ace      -> "A"
  | King     -> "K"
  | Queen    -> "Q"
  | Jack     -> "J"
  | Minor(i) -> (string_of_int i)


(* Suits *)
type suit = Clubs | Hearts | Spades | Diamonds

(* Un-parse a suit *)
let str_suit = function
    Clubs    -> "C"
  | Hearts   -> "H"
  | Spades   -> "S"
  | Diamonds -> "D"


(* Operators *)
type op = Plus | Minus | Times | Divide | Eq | NotEq 
        | Lt | LtEq | Gt | GtEq | And | Or | In 

(* Un-parse an operator *)
let str_op = function
    Plus   -> "+"
  | Minus  -> "-"
  | Times  -> "*"
  | Divide -> "/"
  | Eq     -> "=="
  | NotEq  -> "!="
  | Lt     -> "<"
  | LtEq   -> "<="
  | Gt     -> ">"
  | GtEq   -> ">="
  | And    -> "and"
  | Or     -> "or"
  | In     -> "in"


(* Expressions and References *)
(* Some have positions for keywords or operators *)
type expr =
  | Not of expr * pos
  | Binary of expr * op * expr * pos
  | Defined of reference * pos
  | CanPlay of id * reference * reference * pos
  | CardExpr of expr * pos * expr
  | RangeExpr of expr * expr * pos
  | SeqExpr of expr * expr * pos
  | WildRank of pos * pos * expr
  | WildSuit of expr * pos * pos
  | WildCard of pos * pos * pos
  | List of expr list * pos
  | Ref of reference * pos
  | BoolLit of bool * pos
  | NumLit of int * pos
  | StringLit of str_lit * pos
  | RankLit of rank * pos
  | SuitLit of suit * pos
  | EmptyExpr

and reference =
    Id of id * pos
  | Prop of expr * id * pos
  | Players of pos
  | Teams of pos
  | Standard of pos
  | EmptyRef  (* Used in a few places for opt ref *)

and str_part =
    StrRef of reference
  | StrLit of string

and str_lit = str_part list


(* Unparse an expression *)
let rec str_expr = function
    Not(e, _) ->
      "not " ^ (str_pexpr e)

  | Binary(e1, o, e2, _) ->
      (str_pexpr e1) ^ " " ^ (str_op o) ^ " " ^ (str_pexpr e2)

  | Defined(r, _) ->
      "defined " ^ (str_ref r)

  | CanPlay(s, r1, r2, _) ->
      "canplay " ^ s ^ " from " ^ (str_ref r1) ^ " to " ^ (str_ref r2)

  | CardExpr(e1, _, e2) -> (str_expr e1) ^ "~" ^ (str_expr e2)
  
  | RangeExpr(e1, e2, _) -> (str_expr e1) ^ ".." ^ (str_expr e2)
  
  | SeqExpr(e1, e2, _) -> (str_expr e1) ^ "," ^ (str_expr e2)

  | WildRank(_, _, e) -> "%~" ^ (str_expr e)
  
  | WildSuit(e, _, _) -> (str_expr e) ^ "~%"
  
  | WildCard(_, _, _) -> "%~%"

  | List(es, _) ->
      "[" ^ (Utils.catmap "; " str_expr es) ^ "]"

  | Ref(r, _) -> str_ref r

  | BoolLit(b, _) ->
      if b then "True" else "False"

  | NumLit(n, _) -> string_of_int n

  | StringLit(s, _) -> str_escape s
  | RankLit(r, _) -> str_rank r
  | SuitLit(s, _) -> str_suit s
  | EmptyExpr -> ""

and str_ref = function
    Id(s, _)      -> s
  | Prop(e, s, _) -> (str_pexpr e) ^ "->" ^ s
  | Players(_)    -> "players"
  | Teams(_)      -> "teams"
  | Standard(_)   -> "standard"
  | EmptyRef      -> ""

and str_escape lit =
  let str_cescape = function
      '\\' -> "\\\\"
    | '\n' -> "\\n"
    | '\t' -> "\\t"
    | '"'  -> "\\\""
    | '{'  -> "\\{"
    | '}'  -> "\\}"
    | x    -> String.make 1 x
  in
  let lit_str = function
      StrRef(r) -> "{" ^ (str_ref r) ^ "}"
    | StrLit(s) -> Utils.str_map str_cescape s
  in
  "\"" ^ (Utils.catmap "" lit_str lit) ^ "\""

and str_pexpr e = "(" ^ (str_expr e) ^ ")"


(**************)
(* Statements *)

(* The position marks the line/col of the beginning of the statement *)
type stmt =
    Ask of reference * question list * pos
  | Assign of reference * expr * pos
  
    (* Compound: Op can only be plus, minus, times, or divide *)
  | Compound of reference * op * expr * pos
  
    (* Deal: First expr empty for "all" *)
  | Deal of expr * reference * reference * pos

  | Forever of block * pos
  
    (* For: Second expr could be empty - start at beginning of list *)
  | For of id * expr * expr * block * pos
  
    (* If: Each expr/block pair is if..elseif..elseif.
           Last pair may have EmptyExpr for unconditional else. *)
  | If of (expr * block) list * pos

  | Invoke of id * pos
  | Label of id * pos
  | Let of id * expr * pos
  
    (* Message: Empty ref means message to all *)
  | Message of reference * expr * pos

  | Order of reference * id * pos
  | Play of id * reference * reference * pos
  | Rotate of reference * pos
  | Shuffle of reference * pos
  | Skip of id * pos
  | Winner of reference * pos

and question =
    Uncond of str_lit * block * pos
  | Cond of str_lit * expr * block * pos

and block = stmt list

(* Un-parse a statement *)
let rec str_stmt ind stmt =
  (str_indent ind) ^
  (match stmt with
    Ask(r, qs, _) ->
      "ask " ^ (str_ref r) ^ " " ^ (str_quests ind qs)

  | Assign(r, e, _) ->
      (str_ref r) ^ " = " ^ (str_expr e) ^ ".\n"

  | Compound(r, o, e, _) ->
      (str_ref r) ^ " " ^ (str_op o) ^ "= " ^ (str_expr e) ^ ".\n"

  | Deal(e, r1, r2, _) ->
      let what = function
          EmptyExpr -> "all"
        | e         -> str_expr e
      in
      "deal " ^ (what e) ^ " from " ^ (str_ref r1) ^
                           " to " ^ (str_ref r2) ^ ".\n"

  | Forever(ss, _) ->
      "forever " ^ (str_block ind ss)

  | For(s, e1, e2, ss, _) ->
      let starting = function
          EmptyExpr -> ""
        | e         -> " starting at " ^ (str_expr e2)
      in
      "for " ^ s ^ " in " ^ (str_expr e1) ^ (starting e2) ^
                   " " ^ (str_block ind ss)

  | If(ess, _) -> str_ifs ind ess
  
  | Invoke(s, _) -> s ^ "().\n"
  
  | Label(s, _) -> "label " ^ s ^ ".\n"
  
  | Let(s, e, _) -> "let " ^ s ^ " be " ^ (str_expr e) ^ ".\n"
  
  | Message(r, e, _) ->
      let who = function
          EmptyRef -> ""
        | r        -> (str_ref r) ^ " "
      in
      "message " ^ (who r) ^ (str_expr e) ^ ".\n"

  | Order(r, s, _) ->
      "order " ^ (str_ref r) ^ " by " ^ s ^ ".\n"

  | Play(s, r1, r2, _) ->
      "play " ^ s ^ " from " ^ (str_ref r1) ^ " to " ^ (str_ref r2) ^ ".\n"

  | Rotate(r, _) -> "rotate " ^ (str_ref r) ^ ".\n"

  | Shuffle(r, _) -> "shuffle " ^ (str_ref r) ^ ".\n"

  | Skip(s, _) -> "skip to " ^ s ^ ".\n"

  | Winner(r, _) -> "winner " ^ (str_ref r) ^ ".\n"

  )

and str_quests ind qs =
  "{\n" ^ (Utils.catmap "" (str_quest (ind + 2)) qs) ^ (str_indent ind) ^ "}\n"

and str_quest ind q =
  (str_indent ind) ^
  (match q with
    Uncond(s, ss, _) -> (str_escape s) ^ " " ^ (str_block ind ss)
  | Cond(s, e, ss, _) -> (str_escape s) ^ " if " ^ (str_expr e)
                                        ^ " " ^ (str_block ind ss)
  )

and str_block ind ss =
  "{\n" ^ (Utils.catmap "" (str_stmt (ind + 2)) ss) ^ (str_indent ind) ^ "}\n"

and str_ifs ind ess =
  match ess with
    [] -> raise(Failure("Unexpected empty if/elseif/else list"))

  | ((e, ss) :: rest) ->
      "if " ^ (str_expr e) ^ " " ^ (str_block ind ss) ^
              (Utils.catmap "" (str_elses ind) rest)

and str_elses ind (e, ss) =
  match e with
      EmptyExpr -> (str_indent ind) ^ "else " ^ (str_block ind ss)
    | e         -> (str_indent ind) ^ "elseif " ^ (str_expr e) ^ " " ^ (str_block ind ss)


(****************)
(* Declarations *)

(* Declarable types *)
type dtype = Boolean
           | Card
           | CardList
           | Deck
           | Number
           | Player
           | PlayerList
           | Rank
           | RankList
           | String
           | Suit
           | SuitList
           | Team
           | TeamList

(* Un-parse a type *)
let str_dtype = function
    Boolean    -> "Boolean"
  | Card       -> "Card"
  | CardList   -> "CardList"
  | Deck       -> "Deck"
  | Number     -> "Number"
  | Player     -> "Player"
  | PlayerList -> "PlayerList"
  | Rank       -> "Rank"
  | RankList   -> "RankList"
  | String     -> "String"
  | Suit       -> "Suit"
  | SuitList   -> "SuitList"
  | Team       -> "Team"
  | TeamList   -> "TeamList"


(* For Area decl *)
type areaopt = Faceup | Facedown | Squaredup | Spreadout

type areaopts = (areaopt * pos) list

(* Un-parse area options *)
let str_areaopt = function
    (Faceup, _)    -> "faceup"
  | (Facedown, _)  -> "facedown"
  | (Squaredup, _) -> "squaredup"
  | (Spreadout, _) -> "spreadout"

let str_areaopts opts = Utils.catmap ", " str_areaopt opts


(* Declarations *)
type decl =
    Area of id * str_lit * areaopts * pos  (* Id, Display, Opts *)
  | Action of id * block * pos             (* Id, Body *)
  | Rule of id * id list * expr * pos      (* Id, Args, Body *)
  | Ordering of id * id * expr * pos       (* Id, Arg, Body *)
  | Var of dtype * id * expr * pos         (* Type, Id, Init *)

(* Un-parse a declaration *)
let rec str_decl = function
    Area(s1, s2, [], _) ->
      "Area " ^ s1 ^ " labeled " ^ (str_escape s2) ^ ".\n"

  | Area(s1, s2, opts, _) ->
      "Area " ^ s1 ^ " labeled " ^ (str_escape s2) ^ " is "
                   ^ (str_areaopts opts) ^ ".\n"

  | Action(s, ss, _) ->
      "Action " ^ s ^ " " ^ (str_block 0 ss)

  | Rule(s, ss, e, _) ->
      "Rule " ^ s ^ (str_args ss) ^ " = " ^ (str_expr e) ^ ".\n"

  | Ordering(s, a, e, _) ->
      "Ordering " ^ s ^ (str_args [a]) ^ " = " ^ (str_expr e) ^ ".\n"

  | Var(t, s, e, _) ->
      let init = function
          EmptyExpr -> ""
        | e         -> " = " ^ (str_expr e)
      in
      (str_dtype t) ^ " " ^ s ^ (init e) ^ ".\n"

and str_args ss = "(" ^ (String.concat ", " ss) ^ ")"

(* Un-parse a list of declarations *)
let str_decls ds = Utils.catmap "" str_decl ds


(************)
(* The Game *)

(* Player/team designations *)
type ptcount =
    PlayerCount of int list * pos      (* 2, 3, ... players *)
  | TeamCount of int list * int * pos  (* 2, 3, ... teams of 2 *)

(* Game name, player/team count, list of decls, position of Game keyword *)
type game = str_lit * ptcount * decl list * pos

(* Un-parse a game *)
let rec str_game (s, c, ds, _) =
  "Game " ^ (str_escape s) ^ " requires " ^
      (str_ptcount c) ^ ".\n" ^ (str_decls ds)

and str_ptcount = function
    PlayerCount(ns, _)  ->
      (Utils.catmap " or " string_of_int ns) ^ " players"
      
  | TeamCount(ns, n, _) ->
      (Utils.catmap " or " string_of_int ns) ^
                    " teams of " ^ (string_of_int n)
