package symtb;

import excep.*;

public class TGSymbol {
	/* all valid types must be >= 0 for testing purposes */
	public static final int NO_TYPE = -1;
	public static final int TG_NUM_TYPE = 0;
	public static final int TG_STR_TYPE = 1;
	
    private String hashKey;
    private Object value;
    private int type;
    private int runtimeType;
    
    public TGSymbol(String hashKey, TGNum value) {
    	this.hashKey = hashKey;
        this.value = value;
        this.type = NO_TYPE;
        this.runtimeType = TG_NUM_TYPE;
    }
    
    public TGSymbol(String hashKey, String value) {
    	this.hashKey = hashKey;
        this.value = value;
        this.type = NO_TYPE;
        this.runtimeType = TG_STR_TYPE;
    }
    
    public TGSymbol(TGNum value) {
    	this.hashKey = "";
    	this.value = value;
    	this.type = NO_TYPE;
    	this.runtimeType = TG_NUM_TYPE;
    }
    
    public TGSymbol(String value) {
    	this.hashKey = "";
    	this.value = value;
    	this.type = NO_TYPE;
    	this.runtimeType = TG_STR_TYPE;
    }
    
    public TGSymbol(int i) {
    	this.hashKey = "";
    	this.type = NO_TYPE;
    	this.runtimeType = TG_NUM_TYPE;
    	this.value = new TGNum(i);
    }
    
    public TGSymbol(double d) {
    	this.hashKey = "";
    	this.type = NO_TYPE;
    	this.runtimeType = TG_NUM_TYPE;
    	this.value = new TGNum(d);
    }

    /**
     * Used to find the type of the symbol; knowing what type of symbol
     * is stored in the symbol is necessary before getting its value so the
     * appropriate method can be called
     * @return The type of the symbol: TG_NUM_TYPE, TG_STR_TYPE, or TG_HASH_TYPE
     */
    public int getType(){
        return type;
    }
    
    public boolean hasType() {
    	return this.type == NO_TYPE;
    }
    
    public boolean isNum() {
    	return this.type == TG_NUM_TYPE;
    }
    
    public boolean isInt() {
    	return this.type == TG_NUM_TYPE && ((TGNum)this.value).isInt();
    }
    
    public boolean isDouble() {
    	return this.type == TG_NUM_TYPE && ((TGNum)this.value).isDouble();
    }
    
    public boolean isString() {
    	return this.type == TG_STR_TYPE;
    }
    
    public boolean isRuntimeNum() {
    	return this.runtimeType == TG_NUM_TYPE;
    }
    
    public boolean isRuntimeInt() {
    	return this.runtimeType == TG_NUM_TYPE && ((TGNum)this.value).isInt();
    }
    
    public boolean isRuntimeDouble() {
    	return this.runtimeType == TG_NUM_TYPE && ((TGNum)this.value).isDouble();
    }
    
    public boolean isRuntimeString() {
    	return this.runtimeType == TG_STR_TYPE;
    }
    
    public TGNum getNumValue() 
    		throws TGTypeException
    {
    	if (this.runtimeType != TG_NUM_TYPE)
    		throw new TGTypeException("request for a num value for the non-num hash entry '" +
    				this.hashKey + "'");
    	return (TGNum)value;
    }

    public Integer getIntegerValue() 
    		throws TGTypeException
    {
    	if (this.runtimeType != TG_NUM_TYPE || !((TGNum)value).isInt())
    		throw new TGTypeException("request for an integer value for the non-integer hash entry '" +
    				this.hashKey + "'");
        return ((TGNum)value).getIntValue();
    }
    
    public Double getDoubleValue() 
    		throws TGTypeException
    {
    	if (this.runtimeType != TG_NUM_TYPE || !((TGNum)value).isDouble())
    		throw new TGTypeException("request for a double value for the non-double hash entry '" +
    				this.hashKey + "'");
    	return ((TGNum)value).getDoubleValue();
    }
    
    public String getStringValue() 
    		throws TGTypeException
    {
    	if (this.runtimeType != TG_STR_TYPE)
    		throw new TGTypeException("request for a string value for the non-string hash entry '" +
    				this.hashKey + "'");
    	return (String)value;
    }

    public String getKey(){
        return hashKey;
    }
    
    public void setKey(String newKey) {
    	this.hashKey = newKey;
    }
    
    public void setValue(int i) 
    		throws TGTypeException
    {
    	if (this.type != NO_TYPE && this.type != TG_NUM_TYPE)
    		throw new TGTypeException("attempted assigment of an integer to the non-integer hash entry '" +
    				this.hashKey + "'");
    	this.value = new TGNum(i);
    }
    
    public void setValue(double d) 
    		throws TGTypeException
    {
    	if (this.type != NO_TYPE && this.type != TG_NUM_TYPE)
    		throw new TGTypeException("attempted assignment of a double value to the non-double hash entry '" +
    				this.hashKey + "'");
    	this.value = new TGNum(d);
    }
    
    public void setValue(String s) 
    		throws TGTypeException
    {
    	if (this.type != NO_TYPE && this.type != TG_STR_TYPE)
    		throw new TGTypeException("attempted assignment of a string value to the non-string hash entry '" +
    				this.hashKey + "'");
    	this.value = s;
    }
    
    /**
     * Gets the runtime type of the symbol
     * @return the integer value corresponding to type flags in this class:
     * 			TG_NUM_TYPE, or TG_STR_TYPE
     */
    public int getRuntimeType() {
    	return this.runtimeType;
    }
    
    /**
     * Sets the runtime type of the TGSymbol
     * @param type should be one of the public constants in TGSymbol (ie. TG_NUM_TYPE)
     */
    public void setRuntimeType(int type) {
    	this.runtimeType = type;
    }
    
    public String toString() {
    	if (this.value == null)
    		return "";
    	if (this.runtimeType == TG_NUM_TYPE)
    		return ((TGNum)(this.value)).toString();
    	return (String)this.value;
    }

}
