package symtb;

import excep.*;

import java.util.*;


/**
 *  Things to note about the TGHash class:
 *  
 *  The keys of the elements are the names of the TGSymbols in the linked list
 *  
 *  If two keys of the hash are the same, the first one is returned
 */
public class TGHash implements Cloneable{
   private LinkedList<TGSymbol> elements;
   private int type;
   private String name;
   
   public TGHash() {
	   this("", TGSymbol.NO_TYPE);
   }
   
   public TGHash(String name) {
	   this(name, TGSymbol.NO_TYPE);
   }
   
   /**
    * Creates a new TGHash that is restricted to the given type
    * @param type One of the types defined in TGSymbol
    */
   public TGHash(String name, int type) {
	   this.elements = new LinkedList<TGSymbol>();
	   this.type = type;
	   this.name = name;
   }
   
   public int getType() {
	   return this.type;
   }
   
   public String getName() {
	   return this.name;
   }
   
   public boolean hasType() {
	   return this.type != TGSymbol.NO_TYPE;
   }
   
   /**
    * Restricts the types that can be in the current hash, deletes all values
    * currently in the hash
    * @param type One of the types defined in TGSymbol
    */
   public void setType(int type) {
	   this.type = type;
	   this.elements = new LinkedList<TGSymbol>();
   }
   
   /**
    * Adds a new element to the end of the hash
    * @param symbol The element to add
    */
   public void addAsLast(TGSymbol symbol) 
   		throws TGTypeException
   {
	   if (this.type != TGSymbol.NO_TYPE && symbol.getRuntimeType() != this.type)
		   throw new TGTypeException("attempted addition of type " + symbol.getType() + 
				   " to a hash of type " + this.type);
	   this.elements.addLast(symbol);
   }
   
   /**
    * Adds a new element to the front of the hash
    * @param symbol The element to add
    */
   public void addAsFirst(TGSymbol symbol) 
   		throws TGTypeException
   {
	   if (this.type != TGSymbol.NO_TYPE && symbol.getRuntimeType() != this.type)
		   throw new TGTypeException("attempted addition of type " + symbol.getType() +
				   " to a hash of type " + this.type);
	   this.elements.addFirst(symbol);
   }
   
   /**
    * Concatenates the values from the passed TGHash onto the end of the list
    * of elements for the TGHash on which the method is called
    * @param tail the other TGHash whose elements to add as the new tail of
    * 			the current TGHash's elements
    */
   public void concatenateToEnd(TGHash tail) {
	   TGSymbol[] symbolsToAdd = new TGSymbol[tail.getSymbolCount()];
	   symbolsToAdd = tail.getSymbols();
	   for (int i = 0; i < symbolsToAdd.length; i++)
		   this.elements.addLast(symbolsToAdd[i]);
   }
   
   /**
    * Used to see if the hash is empty
    * @return Returns true if the hash does not contain any elements
    */
   public boolean isEmpty() {
	   return this.elements.isEmpty();
   }

   /**
    * Used to get the last element of hash
    * @return Returns the last element of the hash
    */
   public TGHash getTail () {
   	TGSymbol el = this.elements.getLast ();
	TGHash ret = new TGHash ();
   	
	try {
		ret.addAsFirst (el);
	} catch (TGTypeException e) {
		System.out.println ("Failed to get last element of list.");
	}

	return ret;
   }

   /**
    * Used to get the length of the hash
    * @return Return the length of the hash
    */
   public TGHash getLength () {
   	int el = this.elements.size ();
	TGHash num = new TGHash ();
   	
	try {
		num.setValue (el);
	} catch (TGTypeException e) {
		System.out.println ("Failure when trying to get length of variable.");
	}

	return num;
   }
   /**
    * Allows array-like accesses of a TGHash by getting the symbol given a valid index
    * @param index The index of the element sought
    * @return The TGSymbol at the given index of the hash
    */
   public TGSymbol getSymbol(int index) {
	   return this.elements.get(index);
   }
   
   public TGSymbol getFirstSymbol() {
	   return this.elements.getFirst();
   }
   
   /**
    * Gets the TGSymbol of the specified name
    * @param name The name of the symbol to find
    * @return Returns the first TGSymbol object with the specified name and
    * 			null for not found
    */
   public TGSymbol getElement(String hashKey)
   		throws TGSymbolNotFoundException
   {
	   for (int i = 0; i < this.elements.size(); i++) {
		   TGSymbol current = this.elements.get(i);
		   if (current.getKey().equals(hashKey))
			   return current;
	   }
	   
	   throw new TGSymbolNotFoundException(hashKey);
   }
   
   /**
    * Gets the elements of the hash as an array of TGSymbols
    * @return The elements of the hash as an array
    */
   public TGSymbol[] getSymbols() {
	   TGSymbol[] result = new TGSymbol[this.elements.size()];
	   return this.elements.toArray(result);
   }
   
   public int getSymbolCount() {
	   return this.elements.size();
   }
   
   public void setValue(Integer i) 
   		throws TGTypeException
   {
	   if (this.type != TGSymbol.TG_NUM_TYPE && this.type != TGSymbol.NO_TYPE)
			throw new TGTypeException(this.name);
	   this.elements = new LinkedList<TGSymbol>();
	   this.elements.addFirst(new TGSymbol(new TGNum(i)));
   }
   
   public void setValue(Double d) 
   		throws TGTypeException
   {
	   if (this.type != TGSymbol.TG_NUM_TYPE && this.type != TGSymbol.NO_TYPE)
		   throw new TGTypeException(this.name);
	   this.elements = new LinkedList<TGSymbol>();
	   this.elements.addFirst(new TGSymbol(new TGNum(d)));
   }
   
   public void setValue(String s) 
   		throws TGTypeException
   {
	   if (this.type != TGSymbol.TG_STR_TYPE && this.type != TGSymbol.NO_TYPE)
		   throw new TGTypeException(this.name);
	   this.elements = new LinkedList<TGSymbol>();
	   this.elements.addFirst(new TGSymbol(s));
   }
   
   /**
    * Copies the values from the parameter into the hash and checks all types
    * as necessary
    * @param h the hash values to put into this hash
    * @throws TGTypeException if there is a type constraint on this hash and at least
    * one element of the parameter hash does not match that type constraint
    */
   public void setValue(TGHash h) 
   		throws TGTypeException
   {
	   if (this.type == TGSymbol.NO_TYPE) {
		   this.elements = new LinkedList<TGSymbol>();
		   TGSymbol[] symbols = h.getSymbols();
		   
		   for (int i = 0; i < symbols.length; i++)
			   this.elements.addLast(symbols[i]);
	   }
	   else {
		   LinkedList<TGSymbol> potentialElements = new LinkedList<TGSymbol>();
		   TGSymbol[] symbols = h.getSymbols();
		   
		   for (int i = 0; i < symbols.length; i++) {
			   if (symbols[i].getType() != TGSymbol.NO_TYPE &&
					   symbols[i].getRuntimeType() != this.type)
				   throw new TGTypeException(this.name);
			   else
				   potentialElements.addLast(symbols[i]);
		   }
		   
		   this.elements = potentialElements;
	   }
   }
   
   public void setName(String newName) {
	   this.name = newName;
   }
   
   /**
    * Erases the values in the TGHash
    */
   public void zero() {
	   this.elements = new LinkedList<TGSymbol>();
   }
   
   /**
    * Takes in a string representation of a non-list hash value and converts it into
    * its TGHash representation
    * @param name the name for the TGHash to be returned
    * @param value the string representation of the value
    * @return the corresponding TGHash representation of the value
    * @throws TGTypeException if the provided string representation of the value cannot
    * 			be successfully parsed
    */
   public static TGHash tgHashFromString(String name, String value) 
   		throws TGTypeException
   {
	   TGHash result = null;
	   
	   try {
		   Integer i = Integer.parseInt(value);
		   result = new TGHash(name, TGSymbol.TG_NUM_TYPE);
		   result.addAsFirst(new TGSymbol(new TGNum(i)));
	   } catch (NumberFormatException e) {
		   try {
			   Double d = Double.parseDouble(value);
			   result = new TGHash(name, TGSymbol.TG_NUM_TYPE);
			   result.addAsFirst(new TGSymbol(new TGNum(d)));
		   }
		   catch (NumberFormatException ex) {
			   result = new TGHash(name, TGSymbol.TG_STR_TYPE);
			   result.addAsFirst(new TGSymbol(name, value));
		   }
		   
	   }
	   
	   if (result != null)
		   result.setName(name);
	   
	   return result;
   }
   
   public String toString() {
	   String result = "";
	   TGSymbol[] symbols = new TGSymbol[this.elements.size()];
	   symbols = this.elements.toArray(symbols);
	   for (int i = 0; i < symbols.length; i++) {
		   result += symbols[i].toString();
		   // Separate lists by commas
		   if (!(i == (symbols.length - 1)))
		   	result += ",";
	   }
	   return result;
   }
   
   public TGHash clone() {
	   TGHash clone = new TGHash(this.name, this.type);
	   TGSymbol[] thisSymbols = this.getSymbols();
	   
	   try {
		   for (int i = 0; i < thisSymbols.length; i++) {
			   if (thisSymbols[i].isRuntimeInt())
				   clone.addAsLast(new TGSymbol(new TGNum(thisSymbols[i].getIntegerValue().intValue())));
			   else if (thisSymbols[i].isRuntimeDouble())
				   clone.addAsLast(new TGSymbol(new TGNum(thisSymbols[i].getDoubleValue().doubleValue())));
			   else // COPIES the value, not just passes it
				   clone.addAsLast(new TGSymbol(thisSymbols[i].getStringValue().substring(0)));
		   }
	   }
	   catch (TGTypeException e) {
		   System.out.println("Compiler error in TGHash.clone");
		   System.exit(1);
	   }
	   
	   return clone;
   }
   
}
