package symtb;

import java.util.*;
import org.antlr.runtime.tree.*;
import gram.*;
import excep.*;
import main.*;

public class TGAbstractSymbolTable {
    public boolean debug = TableGen.debug;

    private CommonTree ast;
    private TGScope globalScope;
	
    private LinkedList<FunctionScopeEntry> functionList;
	
    public TGAbstractSymbolTable(CommonTree ast) {
    	this.ast = ast;
    	this.globalScope = new TGScope(null);
    	this.functionList = new LinkedList<FunctionScopeEntry>();
    }
    
    public void printScopes() {
        if (debug)
    		System.out.println("\nScopes:");
    	printScopes(this.globalScope, "");
    }
    
    private void printScopes(TGScope scope, String prefix) {
    	if (debug)
    		System.out.println(prefix + scope.toString());
    	for (int i = 0; i < scope.getNumChildScopes(); i++)
    		printScopes(scope.getChildScope(i), prefix + "\t");
    }
    
    private void printFunctionList() {
    	for (int i = 0; i < functionList.size(); i++)
		if (debug)
    			System.out.println(functionList.get(i).toString());
    }
    
    /**
     * The main method called right after creation to actually build the data structure
     * 
     * ~!~Del
     */
    public void build() {
    	try {
    		this.buildFunctionList(ast);
        	this.printFunctionList(); // debugging, implemented above
        	this.buildScope(ast, globalScope);
    	}
    	catch (TGIllegalDeclarationException e) {
    		System.out.println("Function declaration encountered outside global scope: " +
    				e.getMessage());
    		System.exit(-1);
    	}
    	
    	/* note that it is not necessary to manually pair up functions with the
    	 * just-created scopes because it is internal to buildScopes()
    	 */
    }
    
    /* a method that builds up data on functions since they can be declared anywhere
     * in global scope and are not necessarily forward-declared as with variables
     * 
     * ~!~Del
     */
    private void buildFunctionList(Tree ast) throws TGIllegalDeclarationException {
    	/* function declarations must be at the global scope, so check all
    	 * first-level children of the tree.  If the child is not itself a function
    	 * definition, it's tree should not contain any function definitions.  If it
    	 * is, then its block's subtree should not have any
    	 */
    	for (int i = 0; i < ast.getChildCount(); i++) {
    		Tree currentChild = ast.getChild(i);
    		
    		if (currentChild.getType() == TableGenParser.FUNC_DEF) {
    			TGScope functionScope = new TGScope(null);
    			
    			/* step1: register all function data with this.functionList */
    			String funcName = currentChild.getChild(0).getText();
    			LinkedList<FunctionParameter> params = new LinkedList<FunctionParameter>();
    			
    			int j; // must be declared outside for final call to verifyNoFuncDefs
    			for (j = 1; j < currentChild.getChildCount() && 
    						currentChild.getChild(j).getType() != TableGenParser.BLOCK; j++)
    			{
    				Tree funcParam = currentChild.getChild(j);
    				
    				/* case for a type-checked parameter */
    				if (funcParam.getChildCount() >= 2) {
    					int paramType = tgTypeStringToInt(funcParam.getChild(0).getText());
    					String paramName = funcParam.getChild(1).getText();
    					
    					if (previousDeclarationOf(paramName, this.globalScope) == null)
    						params.addLast(new FunctionParameter(paramName, paramType));
    					else
    						throw new TGIllegalDeclarationException(paramName, currentChild.getLine());
    					
    					functionScope.addSymbol(paramName, paramType);
    				}
    				
    				/* parameter looking at does not have a specified type */
    				else {
    					params.addLast(new FunctionParameter(funcParam.getText()));
    					functionScope.addSymbol(funcParam.getText(), TGSymbol.NO_TYPE);
    				}
    			}
    			
    			FunctionScopeEntry funcScopeEntry = new FunctionScopeEntry(funcName, params);
    			funcScopeEntry.scope = functionScope;
    			this.functionList.addLast(funcScopeEntry);
    			
    			/* step2: make sure the block doesn't contain any function definitions */
    			verifyNoFunctionDefs(ast.getChild(ast.getChildCount() - 1));
    		}
    		
    		/* child of global scope, not a function definition */
    		else {
    			if (ast.getChild(i) != null)
    				verifyNoFunctionDefs(ast.getChild(i));
    		}
    	}
    }
    
    /* 
     * This method is called from buildFunctionList() and verifies that there are no
     * function definitions in the passed tree's subtrees
     * 
     * the passed tree should not be at the global scope
     * 
     * ~!~Del
     */
    private void verifyNoFunctionDefs(Tree tree)
    			throws TGIllegalDeclarationException
    {	
    	/* verify that the type of each node is NOT the "func" keyword */
    	for (int i = 0; i < tree.getChildCount(); i++) {
    		Tree child = tree.getChild(i);
    		
    		if (child.getType() == TableGenParser.FUNC_DEF) {
    			String functionName = child.getChild(0).getText();
    			int lineNumber = child.getLine();
    			throw new TGIllegalDeclarationException(functionName, lineNumber);
    		}
    		
    		if (child.getChildCount() > 0)
    			verifyNoFunctionDefs(child);
    	}
    }
    
    /* 
     * A strictly internal method for recursively building up the abstract
     * symbol table.  It must make sure to set the scopes of functions when they are
     * created.
     * 
     * ~!~Del
     */
    private void buildScope(Tree tree, TGScope currentScope) {
    	boolean elseIsOk = false;
    	
    	if (debug)
    		System.out.println("*dbg: SymbTb.buildScope-- tree="+tree.toStringTree());
    	
    	for (int i = 0; i < tree.getChildCount(); i++) {
    		Tree child = tree.getChild(i);
		if (debug)
    			System.out.println("**dbg: SymbTb.buildScope-- child="+child.toStringTree());
    		try {
    			/*
        		 * do error checking on tree structure here -- don't do it from a subtree
        		 */
    			if (child.getType() == TableGenLexer.IF_KEYWORD)
    				elseIsOk = true;
    			else if (child.getType() == TableGenLexer.ELSEIF_KEYWORD ||
        				child.getType() == TableGenLexer.ELSE_KEYWORD)
        		{
        			if (!elseIsOk)
        				throw new TGIllegalElseException(child.getLine());
        		}
        		else
        			elseIsOk = false;
    			
    			if (child.getType() == TableGenLexer.FUNC_DEF)
        			doFuncDef(child);
        		else if (child.getType() == TableGenLexer.ASSIGNMENT_OP)
        			doAssignmentOp(child, currentScope);
        		else if (child.getType() == TableGenLexer.PUT_OP)
        			doPutOp(child, currentScope);
        		else if (child.getType() == TableGenParser.WHILE_KEYWORD)
        			doWhileLoop(child, currentScope);
        		else if (child.getType() == TableGenParser.FOR_KEYWORD)
        			doForLoop(child, currentScope);
        		else if (child.getType() == TableGenParser.FOREACH_KEYWORD)
        			doForeachLoop(child, currentScope);
        		else if (child.getType() == TableGenParser.IF_KEYWORD ||
        				child.getType() == TableGenParser.ELSEIF_KEYWORD)
        			doIf(child, currentScope);
        		else if (child.getType() == TableGenParser.ELSE_KEYWORD)
        			doElse(child, currentScope);
        		else if (child.getType() == TableGenParser.FUNC_CALL)
        			doFuncCall(child, currentScope);
    			
    		}
    		catch (TGUndefinedSymbolException e) {
    			System.out.println(e.getMessage());
    			System.exit(-1);
    		}
    		catch (TGIllegalDeclarationException e) {
    			System.out.println(e.getMessage());
    			System.exit(-1);
    		}
    		catch (TGMalformedFunctionCallException e) {
    			System.out.println(e.getMessage());
    			System.exit(-1);
    		}
    		catch (TGIllegalElseException e) {
    			System.out.println(e.getMessage());
    			System.exit(-1);
    		}
    		
    	}
    }
    
    /*
     * Adds the result of a recursive call to buildScopes on the function's block
     * to the list of children scopes of currentScope and then update the entry in the
     * instance variable holding the list of function information
     * 
     * ~!~Del
     */
    private void doFuncDef(Tree funcDefTree) {
    	String funcName = funcDefTree.getChild(0).getText();
		Tree funcBlock = funcDefTree.getChild(funcDefTree.getChildCount() - 1);
		
		FunctionScopeEntry[] funcEntries = new FunctionScopeEntry[this.functionList.size()];
		funcEntries = this.functionList.toArray(funcEntries);
		
		for (int j = 0; j < funcEntries.length; j++) {
			if (funcEntries[j].name.equals(funcName)) {
				buildScope(funcBlock, funcEntries[j].scope);
				break;
			}
		}
    }
    
    /* 					ASSIGNMENT
	 * 
	 * child1: TYPE_DECL -> if not empty, has a type declaration
	 * child2: the identifier being given a value
	 * child3: LIST_CONCAT -> for each child, is a value unless it is a ">", in which
	 * 			case the child represents a hash key assignment -- forces a hash type
	 * 
	 * ~!~Del
	 */
    private void doAssignmentOp(Tree assignmentTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {		
    	if (assignmentTree.getChild(0) == null || assignmentTree.getChild(0).getType() != TableGenLexer.TYPE_DECL)
			System.err.println("bad value for first child of assignment op");
		
		/* gets the existing symbol table entry for the l-value in the assignment */
		TGHash declaredSymbol = previousDeclarationOf(assignmentTree.getChild(1).getText(), parentScope);
		
		/* error if attempting to redeclare a variable */
		if (declaredSymbol != null && assignmentTree.getChild(0).getChildCount() > 0) {
			System.out.print("Attempted redeclaration of variable: ");
			throw new TGUndefinedSymbolException(declaredSymbol.getName(), assignmentTree.getChild(1).getLine());
		}
		
		/* assignmentTree.getChild(2) is the LIST_CONCAT that holds r-values */
		Tree rvalues = assignmentTree.getChild(2);
		validateTreeSymbols(rvalues, parentScope);
		
		/* add the symbol to the current scope if it has not already been declared */
		if (declaredSymbol == null) {
			int type;
			String variableName = assignmentTree.getChild(1).getText();
			
			/* the variable has a type declaration */
			if (assignmentTree.getChild(0).getChildCount() > 0) {
				type = tgTypeStringToInt(assignmentTree.getChild(0).getChild(0).getText());
				
				parentScope.addSymbol(variableName, type);
			}
			
			/* no type declaration with the variable */
			else {
				type = TGSymbol.NO_TYPE;
				parentScope.addSymbol(new TGHash(variableName));
				
			}
			//System.out.println("dbg: adding symbol '" + variableName + "' to current scope as type " + type);
		}
    }
    
    private void doPutOp(Tree putOpTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {
    	/* handle the expression on the left of the putop -- it IS the expression subtree */
    	validateTreeSymbols(putOpTree.getChild(0), parentScope);
    	
    	/* handle the coordinate -- validate both expressions */
    	validateTreeSymbols(putOpTree.getChild(1).getChild(0), parentScope);
    	validateTreeSymbols(putOpTree.getChild(1).getChild(1), parentScope);
    	
    }
    
    private void doWhileLoop(Tree whileTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {
    	/* validate all symbols in the expression */
    	validateTreeSymbols(whileTree.getChild(0), parentScope);
    	
    	/* make a child scope and build it up for the while loop using a call to 
    	 * buildScope()
    	 */
    	TGScope whileScope = new TGScope(parentScope);
    	buildScope(whileTree.getChild(1), whileScope);
    }
    
    /* ~!~Del */
    private void doForLoop(Tree forLoopTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {
    	/* since both expressions and assignment are allowed in the first/last parts of
    	 * the for loop declaration, we must make sure to add any new variables in the
    	 * assignment to the scope for the for loop before processing the body -- this is
    	 * taken care of by doAssignmentOp()
    	 */
    	TGScope forScope = new TGScope(parentScope);
    	
    	if (forLoopTree.getChild(0).getType() == TableGenParser.ASSIGNMENT_OP)
    		doAssignmentOp(forLoopTree.getChild(0), forScope);
    	
    	/* pass the currentScope below to keep from the extra function call if passing the
    	 * for scope -- I mean, we know it's empty . . .
    	 */
    	else
    		validateTreeSymbols(forLoopTree.getChild(0), parentScope);
    	
    	/* make sure the boolean expression has valid symbols */
    	validateTreeSymbols(forLoopTree.getChild(1), forScope);
    	
    	/* does this 'if' case have a problem with defining a new variable in the final statement?
    	 * is that a runtime exception if so?
    	 */
    	if (forLoopTree.getChild(2).getType() == TableGenParser.ASSIGNMENT_OP)
    		doAssignmentOp(forLoopTree.getChild(2), forScope);
    	else
    		validateTreeSymbols(forLoopTree.getChild(2), forScope);
    	
    	buildScope(forLoopTree.getChild(3), forScope);
    }
    
    /* ~!~Del */
    private void doForeachLoop(Tree foreachLoopTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException, TGIllegalDeclarationException
    {
    	TGScope foreachScope = new TGScope(parentScope);
    	int type = TGSymbol.NO_TYPE;
    	
    	Tree iteratorChild = foreachLoopTree.getChild(1);
    	String iteratorName = iteratorChild.getText();
    	TGHash previouslyDeclared = previousDeclarationOf(iteratorName, parentScope);
    	
    	/* iterator must be a new variable -- throw error otherwise */
    	if (previouslyDeclared != null)
    		throw new TGIllegalDeclarationException(previouslyDeclared.getName(), iteratorChild.getLine());
    	
    	/* handle typed iterator */
    	if (foreachLoopTree.getChild(0).getChildCount() > 0)
    		type = tgTypeStringToInt(foreachLoopTree.getChild(0).getChild(0).getText());
    	
    	foreachScope.addSymbol(iteratorName, type);
    	
    	buildScope(foreachLoopTree.getChild(3), foreachScope);
    	
    }
    
    private void doIf(Tree ifTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {
    	TGScope ifScope = new TGScope(parentScope);
    	validateTreeSymbols(ifTree.getChild(0), parentScope); // expression to evaluate
    	buildScope(ifTree.getChild(1), ifScope); // build the scope for the block
    }
    
    private void doElse(Tree elseTree, TGScope parentScope) 
    			throws TGUndefinedSymbolException
    {
    	TGScope elseScope = new TGScope(parentScope);
    	buildScope(elseTree.getChild(0), elseScope);
    }
    
    /**
     * A method that will validate a function call given a FUNC_CALL AST node
     * @param funcCallTree the FUNC_CALL AST node corresponding to the function call
     * @param parentScope the scope from which this function was called
     * @throws TGUndefinedSymbolException when a named parameter is not found in the
     * 			list of valid parameter names for the function, or if the function
     * 			was not defined
     * @throws TGMalformedFunctionCallException when the number of parameters is incorrect
     * 
     * ~!~Del
     */
    private void doFuncCall(Tree funcCallTree, TGScope parentScope)
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {	
    	String funcName = funcCallTree.getChild(0).getText();
    	if (debug)
    		System.out.println (funcName +  ": " + funcCallTree.toStringTree ());

    	// If reversed function, do something special.
    	if (funcName.matches ("^_.+")) {
    		return;
    	}
    		
    	int callParamCount = funcCallTree.getChildCount() - 1; // don't count the name as a parameter
    	FunctionScopeEntry function = getFunctionScopeEntry(funcName);
    	
    	if (function == null) {
    		throw new TGUndefinedSymbolException(funcName, funcCallTree.getLine());
    	}
    	
    	if (callParamCount != function.getParamCount())
    		throw new TGMalformedFunctionCallException(funcName, funcCallTree.getLine());
    	
    	boolean sawNamedParam = false;
    	boolean sawUnnamedParam = false;
    	
    	/* check each parameter for valid values */
    	for (int j = 1; j < funcCallTree.getChildCount(); j++) {
    		Tree param = funcCallTree.getChild(j);
    		
    		/* check named parameter */
    		if (param.getText().equals(":")) {
    			String paramName = param.getChild(0).getText();
    			boolean isValidParam = isValidFuncParam(funcName, paramName);
    			
    			if (!isValidParam) {
    				System.out.println("Unable to match function '" + funcName + 
    								" with parameter " + paramName + ": ");
    				throw new TGUndefinedSymbolException(paramName, param.getChild(0).getLine());
    			}
    			sawNamedParam = true;
    		}
    		
    		/* check literal or identifier as value-only parameter */
    		else {
    			if (!isValidSymbol(param.getText(), parentScope)) 
    				throw new TGUndefinedSymbolException(param.getText(), param.getLine());
    			sawUnnamedParam = true;
    		}
    		
    		/* cannot allow mixing for interpreting issues that follow */
    		if (sawNamedParam && sawUnnamedParam)
    			throw new TGMalformedFunctionCallException("Illegal mixing of named/unnamed parameters "+
    					"in call to function '" + funcName + "'", funcCallTree.getLine());
    	}
    }
    
    /**
     * Returns the TGSymbol object corresponding to the previous declaration of the given
     * variable or function name name or null if the name is not already defined
     * @param name The name of the identifier to search for
     * @return The corresponding TGSymbol object if a match was found, otherwise null.  If 
     * 			the symbol found was a function, the name field is the matched name but the
     * 			type is NO_TYPE.  It is also a temporary returned value, not one that is 
     * 			actually held in a given scope.
     * 
     * ~!~Del
     */
    private TGHash previousDeclarationOf(String name, TGScope scope) {
    	TGHash[] symbols; // symbols for the current scope
    	
    	for (; scope != null; scope = scope.getParentScope()) {
    		symbols = scope.getSymbols();
        	for (int i = 0; i < symbols.length; i++) {
        		if (symbols[i].getName().equals(name))
        			return symbols[i];
        	}
    	}
    	
    	/* check the list of functions */
    	FunctionScopeEntry[] funcs = new FunctionScopeEntry[this.functionList.size()];
    	funcs = this.functionList.toArray(funcs);
    	for (int i = 0; i < funcs.length; i++) {
    		if (funcs[i].name.equals(name))
    			return new TGHash(name); /* returns a temporary symbol with no type for a function */
    	}
    	
    	return null;
    }
    
    /**
     * Makes sure that all values in the given tree have been previously declared, ignoring operators
     * and including function calls and their parameters.  This method is primarily used in static-
     * semantic analysis checks of statements in the buildScopes() method
     * @param tree The subtree of the AST to evaluate
     * @throws a TGUndefinedSymbolException if a tree symbol has not been previously defined
     * 
     * ~!~Del
     */
    private void validateTreeSymbols(Tree tree, TGScope currentScope) 
    			throws TGUndefinedSymbolException, TGMalformedFunctionCallException
    {	
    	/* the current node is a tree with children */
		if (tree.getChildCount() >= 1) {
			
			if (tree.getType() == TableGenParser.FUNC_CALL)
				doFuncCall(tree, currentScope);
			
			/* user must keep track of making sure list keys are valid */
			else if (tree.getType() == TableGenParser.HASH_ASSIGN);
			else if (tree.getType() == TableGenParser.DECIMAL_POINT)
				validateTreeSymbols(tree.getChild(0), currentScope);
			
			/* trees with children but that are not function call trees -- validate children */
			else  {
				for (int i = 0; i < tree.getChildCount(); i++)
					validateTreeSymbols(tree.getChild(i), currentScope);
			}
		}
		
		/* the current node is not a tree with children -- must be an identifier, literal, operator,
		 * function call name, or unnamed function call parameter (named function call parameter 
		 * handled in the tree case)
		 */
		else if (!isValidSymbol(tree.getText(), currentScope))
				throw new TGUndefinedSymbolException(tree.getText(), tree.getLine());
    	
    }
    
    /* ~!~Del */
    private boolean isValidFuncParam(String funcName, String paramName) {
    	boolean paramFound = false;
		FunctionScopeEntry[] funcs = new FunctionScopeEntry[this.functionList.size()];
		funcs = this.functionList.toArray(funcs);
		
		for (int k = 0; k < funcs.length; k++) {
			
			/* we found the function to match the named parameter with */
			if (funcs[k].name.equals(funcName)) {
				FunctionParameter[] params = new FunctionParameter[funcs[k].params.size()];
				params = funcs[k].params.toArray(params);
				
				/* look through the parameters for a match */
				for (int l = 0; l < params.length; l++) {
					if (params[l].name.equals(paramName))
						paramFound = true;
				}
			}
			
		}
		
		return paramFound;
    }
    
    /* Gets the type of the r-values in assignment ~!~ Del */
    public static int tgTypeFromListConcatNode(Tree listConcatNode) {
    	int type;
    	
    	try {
			Integer.parseInt(listConcatNode.getChild(0).getText());
			type = TGSymbol.TG_NUM_TYPE;
		}
		catch (NumberFormatException e) {
			try {
				Double.parseDouble(listConcatNode.getChild(0).getText());
				type = TGSymbol.TG_NUM_TYPE;
			}
			catch (NumberFormatException ex) {
				type = TGSymbol.TG_STR_TYPE;
			}
		}
    	
    	return type;
    }
    
    /* ~!~Del */
    public boolean isValidSymbol(String str, TGScope scope) {
    	if (
    			isStringLiteral(str) ||
    			isNumberLiteral(str) ||
    			isOperator(str)
    	   )
    		return true;
    	
    	TGHash declared = previousDeclarationOf(str, scope);
		if (declared != null)
			return true;
    	
    	return false;
    }
    
    /* ~!~Del */
    public static boolean isNumberLiteral(String str) {
    	try {
    		Integer.parseInt(str);
    		return true;
    	}
    	catch (NumberFormatException e) {
    		try {
    			Double.parseDouble(str);
    			return true;
    		}
    		catch (NumberFormatException ex) {
    			return false;
    		}
    	}
    }
    
    public static boolean isStringLiteral(String str) {
    	return str.charAt(0) == '"';
    }
    
    public static boolean isOperator(String str) {
    	if (
    		str.equals("+") ||
    		str.equals("-") ||
    		str.equals("*") ||
    		str.equals("/") ||
    		str.equals("&&") ||
    		str.equals("||") ||
    		str.equals("==") ||
    		str.equals("->") ||
    		str.equals("=") ||
    		str.equals("<") ||
    		str.equals(">") ||
    		str.equals("<=") ||
    		str.equals(">=") ||
    		str.equals("!=")
    	   )  
    	   return true;
    	
    	
    	return false;
    }
    
    /**
     * This method takes in a string representation of a TG primitive type such as "num"
     * and returns the corresponding value from the TGSymbol class
     * @param typeDeclaration The string representation of the type declaration
     * @return The integer value corresponding to the type as in the TGSymbol class 
     * 				(< 0 for invalid type)
     * 
     * ~!~Del
     */
    public static int tgTypeStringToInt(String typeDeclaration) {
    	if (typeDeclaration.equals("num"))
    		return TGSymbol.TG_NUM_TYPE;
    	else if (typeDeclaration.equals("str"))
    		return TGSymbol.TG_STR_TYPE;
    	else
    		return TGSymbol.NO_TYPE;
    }
    
    public TGScope getGlobalScope() {
    	return this.globalScope;
    }
    
    /**
     * Gets the TGScope representing the scope of the specified function.  Note that
     * all internal scopes such as in for loops and if blocks are children of the
     * returned scope.
     * @param funcName The name of the function to get the scope of
     * @return The TGScope of the specified function with the specified name or 
     * 			null for not found
     * 
     * ~!~Del
     */
    public TGScope getFunctionScope(String funcName) {
    	FunctionScopeEntry[] funcEntries = new FunctionScopeEntry[this.functionList.size()];
    	funcEntries = this.functionList.toArray(funcEntries);
    	
    	for (int i = 0; i < funcEntries.length; i++) {
    		if (funcEntries[i].name.equals(funcName))
    			return funcEntries[i].scope;
    	}
    	
    	return null;
    }
    
    /**
     * Returns the function scope entry corresponding to the given name
     * @param functionName 
     * @return
     */
    private FunctionScopeEntry getFunctionScopeEntry(String functionName) {
    	FunctionScopeEntry[] funcScopeEntries = new FunctionScopeEntry[this.functionList.size()];
    	funcScopeEntries = this.functionList.toArray(funcScopeEntries);
    	
    	for (int i = 0; i < funcScopeEntries.length; i++) {
    		if (funcScopeEntries[i].name.equals(functionName))
    			return funcScopeEntries[i];
    	}
    	
    	return null;
    }
}


/* ~!~Del */
class FunctionScopeEntry {
	String name;
	LinkedList<FunctionParameter> params;
	TGScope scope;
	
	/* this constructor is so that an initial pass of the tree can be made to validate names of
	 * functions during tree walking since nothing is said about the declaration order
	 */
	public FunctionScopeEntry(String name) {
		this.name = name;
	}
	
	public FunctionScopeEntry(String name, LinkedList<FunctionParameter> parameters) {
		this.name = name;
		this.params = parameters;
	}
	
	public String toString() {
		String result = "Function Name: " + this.name;
		result += "\n\t";
		
		for (int i = 0; i < params.size(); i++) {
			FunctionParameter p = params.get(i);
			result += "param " + i + ": type=" + p.type + ", name=" + p.name + "\n\t";
		}
		
		result += "\n";
		return result;
	}
	
	public int getParamCount() {
		return this.params.size();
	}
}

/* ~!~Del */
class FunctionParameter {
	String name;
	int type; /* types as defined in TGSymbol.java */
	
	public FunctionParameter(String name) {
		this.name = name;
		this.type = TGSymbol.NO_TYPE;
	}
	
	public FunctionParameter(String name, int type) {
		this.name = name;
		this.type = type;
	}
}
