package main;

import excep.*;
import gram.*;
import symtb.*;
import java.util.*;
import org.antlr.runtime.tree.*;

public class Interpreter {
	public static boolean debug = TableGen.debug;
	/* values for referring to the type of a symbol -- need to have
	 * a distinct hash type too because not everything is a hash when
	 * talking about values
	 */
	public static final int INTEGER_SYMBOL_TYPE = 1;
	public static final int DOUBLE_SYMBOL_TYPE = 2;
	public static final int STRING_SYMBOL_TYPE = 3;
	public static final int HASH_SYMBOL_TYPE = 4;
	
	private TGAbstractSymbolTable symbols;
	private CommonTree ast;
	private LinkedList<TGScope> symbolTable;
	private Vector<Vector<String>> outputTable;
	
	private Stack<TGHash> returnValues;

	// These are needed so that we don't print a bunch of 
	// blank at the end.
	private int maxRow;
	private int maxCol;
	
	/* needed to check to see if the previous statement executed was an if/elseif statement
	 * so an 'else' block can be ignored as necessary
	 */
	private boolean ifJustExecuted = false;
	
	/*
	 * A global boolean variable that allows an executing loop or scope to simply
	 * stop executing when a function returns
	 */
	private boolean functionIsReturning = false;
	
        private int initialTableSize = 10;
        private int tableExpansionSize = 100;
	
	public Interpreter(CommonTree ast, TGAbstractSymbolTable symbols) 
	{
		this.ast = ast;
		this.symbols = symbols;
		this.symbolTable = new LinkedList<TGScope>();
		this.returnValues = new Stack<TGHash>();
		
		this.outputTable = new Vector<Vector<String>> (initialTableSize, tableExpansionSize);

		for (int i = 0; i < outputTable.capacity(); i++){
                    this.outputTable.add(new Vector<String> (initialTableSize, tableExpansionSize));
                    for (int j = 0; j < outputTable.get(i).capacity(); j++)
                        outputTable.get(i).add("");
                }

		this.maxRow = -1;
		this.maxCol = -1;
	}
	
	public void interpret() {
		this.pushScope(this.symbols.getGlobalScope());
		this.interpretBlock(ast);
	}
	
	/**
	 * This method interprets the AST subtree it is given
	 * @param tree the AST subtree to interpret
	 * @param treeIsBlock pass 'true' if only the children of the passed tree are to be
	 * 			executed as in a block statement or 'false' if the tree is just a
	 * 			statement itself
	 */
	private void interpretBlock(Tree tree) {
		if (debug)
			System.out.println("dbg: Int.interpretBlock-- #children="+tree.getChildCount());
		for (int childNumber = 0; childNumber < tree.getChildCount(); childNumber++) {
			Tree currentChild = tree.getChild(childNumber);
			if (debug)
				System.out.println("dbg: Int.interpretBlock-- child="+currentChild.toStringTree());
			interpretStatement(currentChild);
			if (functionIsReturning)
			    return; 
		}
		
		/* if the current scope has been entirely executed and it is the global scope */
		if (this.symbolTable.size() == 1 && this.symbolTable.get(0).equals(symbols.getGlobalScope()))
			outputTable();
	}
        
	private void interpretStatement(Tree statement) {
		if (statement.getType() == TableGenParser.ASSIGNMENT_OP)
			doAssignmentOp(statement);
		else if (statement.getType() == TableGenParser.PUT_OP)
			doPutOp(statement);
		/*
		 * this case covered by the final statement handling expressions . . . I think.
		 * else if (statement.getType() == TableGenParser.FUNC_CALL)
			doFuncCall(statement, false);
		*/
		else if (statement.getType() == TableGenParser.WHILE_KEYWORD)
			doWhileLoop(statement);
		else if (statement.getType() == TableGenParser.FOR_KEYWORD)
			doForLoop(statement);
		else if (statement.getType() == TableGenParser.FOREACH_KEYWORD)
			doForeachLoop(statement);
		else if (statement.getType() == TableGenParser.IF_KEYWORD)
			doIf(statement);
		else if (statement.getType() == TableGenParser.ELSEIF_KEYWORD) 
			//internal checking of ifJustExecuted flag
			doElseif(statement);
		else if (statement.getType() == TableGenParser.ELSE_KEYWORD) 
			//internal checking of ifJustExecuted flag
			doElse(statement);
        else if (statement.getType() == TableGenParser.RETURN_KEYWORD) {
        	if (debug)
		    	System.out.println ("Return statement found...about to evaluate this: " + 
					     statement.getChild(0).toStringTree ());
            TGHash returnValue = evaluateExpression(statement.getChild(0));
            functionIsReturning = true;
		    if (debug)
		    	System.out.println ("RValue " + returnValue.toString ());
            this.returnValues.push(returnValue.clone());
        }
		
		/* handle expressions */
		else if (
					statement.getType() == TableGenParser.ADDITIVE_OP ||
					statement.getType() == TableGenParser.MULTIPLICATIVE_OP ||
					statement.getType() == TableGenParser.COMMA ||
					statement.getType() == TableGenParser.FUNC_CALL ||
					statement.getType() == TableGenParser.BOOL_OP
				)
		{
			evaluateExpression(statement);
		}
		
		if (statement.getType() != TableGenParser.IF_KEYWORD && 
		    statement.getType() != TableGenParser.ELSEIF_KEYWORD)
			this.ifJustExecuted = false;
	}
	
	/* ~!~Del */
	private void doAssignmentOp(Tree assignmentTree)
	{
		String lValueName = assignmentTree.getChild(1).getText();
		
		try {
			TGHash symbol = getSymbol(lValueName, assignmentTree.getChild(1).getLine());
			TGHash value = evaluateExpression(assignmentTree.getChild(2));
			symbol.setValue(value);
			
			if (debug)
				System.out.println("dbg: doAssignmentOp-- value type as TGNum=" 
							+ symbol.getFirstSymbol().getRuntimeType());
			
			if (debug)
				System.out.println("dbg: doAssignmentOp-- assinging value '" + value.toString() +
						   "' to variable '" + lValueName +"'");
		}
		catch (TGSymbolNotFoundException e) {
			System.out.println(e.getMessage());
			System.exit(-1);
		}
		catch (TGTypeException e) {
			System.out.println(new TGTypeException(e.getMessage(), assignmentTree.getLine()));
			System.exit(-1);
		}
	}
	
	/*
	 * Currently only handles coordinates, not identifiers as lists
	 * 
	 * ~!~Del
	 */
	private void doPutOp(Tree putOpTree) {
		TGHash value = evaluateExpression(putOpTree.getChild(0));
		TGHash row = evaluateExpression(putOpTree.getChild(1).getChild(0));
		TGHash col = evaluateExpression(putOpTree.getChild(1).getChild(1));
		
		try {
			if (debug)
				System.out.println("dbg doPutOp-- putting "+value.toString()+" into " +
						   "["+row.getFirstSymbol().getIntegerValue()+"|"+
						   col.getFirstSymbol().getIntegerValue()+"]");
			
			putInOutputTable(value, 
					 row.getFirstSymbol().getIntegerValue(), 
					 col.getFirstSymbol().getIntegerValue());
		}
		catch (TGTypeException e) {
			System.out.println(new TGTypeException(e.getMessage(), putOpTree.getLine()));
		}
		catch (NoSuchElementException e) {
			System.out.println("Invalid coordinates '["+row.toString()+"|"+col.toString()+"]' " +
					"for coordinates on line " + putOpTree.getLine());
			System.exit(-1);
		}
	}
	
	private void putInOutputTable(TGHash value, int row, int col) {
            if (row > maxRow){
                if (row > outputTable.size()){
                    for (int i = outputTable.size(); i <= row; i++){
                        outputTable.add(new Vector<String> (initialTableSize, tableExpansionSize));
                        for (int j = 0; j < outputTable.get(i).capacity(); j++)
                            outputTable.get(i).add("");
                    }
                }
                maxRow = row;
            }
            
            if (col > maxCol){
                if (col > outputTable.get(row).size()){
                    for (int i = 0; i < outputTable.size(); i++){
                        for (int j = outputTable.get(row).size(); j <= col; j++){
                            outputTable.get(i).add("");
                        }
                    }
                }
                maxCol = col;
            }
            this.outputTable.get(row).setElementAt(value.toString(), col);
	}
	
	/**
	 * Executes the function call specified and pushes a return value onto the local variable
	 * returnStack if required, throwing an error on failure
	 * @param funcCallTree the tree with the FUNC_CALL node as the root
	 * @param returnRequired a boolean value indicating if a return values should be
	 * 			pushed onto the stack
	 * @return a boolean value indicating if a return value was pushed onto the return stack
	 * 
	 * ~!~Del
	 */
	private boolean doFuncCall(Tree funcCallTree) 
			throws TGMissingReturnValueException
	{
		String funcName = funcCallTree.getChild(0).getText();

		/* intercept reserved functions */
		if (funcName.charAt(0) == '_') {
	                // _tail function always returns something
			if (funcName.equals ("_tail")) {
	            doTailFunc (funcCallTree);
				return true;
			}

			// _length functions always returns something.
			else if (funcName.equals ("_length")) {
				doLengthFunc (funcCallTree);
				return true;
			}
	                
			else if (funcName.equals ("_avg")) {
				doAvgFunc (funcCallTree);
				return true;
			}
	                
			else if (funcName.equals ("_sum")) {
				doSumFunc (funcCallTree);
				return true;
			}

			else if (funcName.equals ("_round")) {
				doRoundFunc (funcCallTree);
				return true;

			}
			throw new RuntimeException("Function name beginning with '_' not a valid " +
					"reserved function\n");
		}
		
		
														 
		try {
			//String funcName = funcCallTree.getChild(0).getText();

			Tree funcDefTree = this.getFunctionTree(funcName);
			Tree blockTree = funcDefTree.getChild(funcDefTree.getChildCount() - 1);
			
			TGScope functionScope = this.symbols.getFunctionScope(funcName);
			
			if (functionScope == null)
				throw new RuntimeException("Compiler Error: could not get scope of function '" +
						funcName + "'");
			
			/* initialize parameters */
			
			/* case for first being all named parameters  */
			if (funcCallTree.getChildCount() > 1 && 
					funcCallTree.getChild(1).getType() == TableGenParser.COLON) 
			{
				for (int i = 1; i < funcCallTree.getChildCount(); i++) {
					/* easiest to hold value in TGHash because it can associate 
					 * names and values already */
					String paramName = funcCallTree.getChild(i).getChild(0).getText();
					
					try {
						TGHash value = evaluateExpression(funcCallTree.getChild(i).getChild(1));
						setValueInScope(paramName, value, HASH_SYMBOL_TYPE, functionScope);
					}
					catch (TGTypeException e) {
						System.out.println(new TGTypeException(e.getMessage(), 
								   funcCallTree.getLine()).toString());
						System.exit(-1);
					}
					if (debug)
						System.out.println("dbg: Int.doFuncCall, all named params, completed i="+i);
				}
			}
			
			/* case for first all being unnamed parameters, symbols matched up as OK in 
			 * static-semantic analysis
		     */
			else {
				int funcTreeParams = funcCallTree.getChildCount() - 1;
				TGHash[] symbols = functionScope.getSymbols();
				
				for (int i = 0; i < funcTreeParams; i++) {
					// i+1 because we must exclude the function name
					try {
						TGHash paramValue = evaluateExpression(funcCallTree.getChild(i+1));
						symbols[i].setValue(paramValue);
					}
					catch (TGTypeException e) {
						System.out.println(new TGTypeException(e.getMessage(), 
								   funcCallTree.getLine()).toString());
						System.exit(-1);
					}
				}
			}
			
			/* execute the function, return value if necessary -- 
			 * interpret ignores return statements 
			 */
			this.pushScope(functionScope);
			interpretBlock(blockTree);
			if (functionIsReturning) {
				// Functions with nothing in them in need to 
				// return something anyway.
				if (returnValues.isEmpty ())
					returnValues.push (new TGHash ());
				functionIsReturning = false;
			}

			this.popScope();
			
			//if (blockTree.toStringTree ().contains ("FUNC_CALL"))
			//	return true;

 		        return true;
		}
		catch (TGSymbolNotFoundException e) {
			System.out.print(e.getMessage());
			System.out.println(" from doFuncCall.");
			System.exit(-1);
		}
		//if (debug)
			//System.out.println("dbg: doFuncCall-- top return stack value="+this.returnValues.get(0));
		return false;
	}

	private void doTailFunc (Tree funcCallTree) {
		// Tail only takes a list as input.
		if (funcCallTree.getChildCount() != 2) {
			System.out.println("Error: Call to _tail can only take one argument.");
			System.exit (-1);
		}

		TGHash tail = evaluateExpression(funcCallTree.getChild(1));
		this.returnValues.push(tail.getTail());
	}

	private void doLengthFunc (Tree funcCallTree) {
		if (funcCallTree.getChildCount() != 2) {
			System.out.println("Error: Call to _length can only take one argument; I think you gave it " 
					 	+ funcCallTree.getChildCount());
			System.exit (-1);
		}

		TGHash len = evaluateExpression(funcCallTree.getChild(1));
		this.returnValues.push(len.getLength ());
	}
        
	private void doAvgFunc(Tree funcCallTree){
	if (funcCallTree.getChildCount() != 2) {
		System.out.println("Error: Call to _avg can only take one argument; I think you gave it " 
					+ funcCallTree.getChildCount());
		System.exit (-1);
	}
			TGHash arg = evaluateExpression(funcCallTree.getChild(1));
			TGHash value = new TGHash();
			double sum = 0;
			boolean is_double = false;
			try {
				for (int i = 0; i < arg.getSymbolCount(); i++){
					if (arg.getSymbol(i).isDouble() || arg.getSymbol(i).isRuntimeDouble()){
						is_double = true;
						sum += arg.getSymbol(i).getDoubleValue().doubleValue();
					}
					else
						sum += arg.getSymbol(i).getIntegerValue().intValue();
				}
				if (!is_double && (((int)sum) % arg.getSymbolCount() != 0))
					is_double = true;
				
				if (is_double)
					value.addAsFirst(new TGSymbol(new TGNum(sum / arg.getSymbolCount())));
				else
					value.addAsFirst(new TGSymbol(new TGNum((int)(sum / arg.getSymbolCount()))));
				
				this.returnValues.push(value);
			}
			catch (TGTypeException e){
				
			}
	}
	
	private void doSumFunc(Tree funcCallTree){
		if (funcCallTree.getChildCount() != 2) {
			System.out.println("Error: Call _sum can only take one argument; I think you gave it " 
						+ funcCallTree.getChildCount());
			System.exit (-1);
		}
		TGHash arg = evaluateExpression(funcCallTree.getChild(1));
		TGHash value = new TGHash();
		double sum = 0;
		boolean is_double = false;
		try {
			for (int i = 0; i < arg.getSymbolCount(); i++){
				if (arg.getSymbol(i).isDouble() || arg.getSymbol(i).isRuntimeDouble()){
					is_double = true;
					sum += arg.getSymbol(i).getDoubleValue().doubleValue();
				}
				else
					sum += arg.getSymbol(i).getIntegerValue().intValue();
			}
			
			if (is_double)
				value.addAsFirst(new TGSymbol(new TGNum(sum)));
			else
				value.addAsFirst(new TGSymbol(new TGNum((int)sum)));
			
			this.returnValues.push(value);
		}
		catch (TGTypeException e){
			
		}
	}

	private void doRoundFunc (Tree funcCallTree) {
		if (funcCallTree.getChildCount() < 2 || funcCallTree.getChildCount() > 3) {
			System.out.println("Error: Round function can only take one or two agruments; I think you gave it "
						+ funcCallTree.getChildCount());
			System.exit (-1);
		}

		try {
			TGHash roundTo = new TGHash ();
			if (funcCallTree.getChildCount() == 3) {
				roundTo = evaluateExpression (funcCallTree.getChild(2));
			} else {
                	        roundTo.addAsFirst(new TGSymbol("temp", new TGNum(100.0))); // Default to 100.
			}
			TGHash value = new TGHash ();
			value = evaluateExpression (funcCallTree.getChild(1));
	
				/* value is an integer; don't round */
	                if (value.getFirstSymbol().isRuntimeInt())
				this.returnValues.push(value);
	
	                /* at least one operand is a double -- the result is also a double */
	                else {
	                        double val = value.getFirstSymbol().getDoubleValue();
				double rnd = roundTo.getFirstSymbol().getDoubleValue();
	
				double ret = Math.round (val * rnd) / rnd;
	
				TGHash rounded = new TGHash ();
				rounded.addAsFirst(new TGSymbol("temp", new TGNum(ret)));
				this.returnValues.push(rounded);
			}
		} catch (TGTypeException e) {
			System.out.println("Error: types to not match in doRoundFunc.");
		} 
			 

	}
	
	private void doWhileLoop(Tree whileTree){
        TGHash result;
        Tree test = whileTree.getChild(0);
        Tree block = whileTree.getChild(1);
        TGScope whileScope = this.symbolTable.peek().getNextChildScope();
        try {
            while (true){
                result = evaluateExpression(test);
                int i = result.getFirstSymbol().getNumValue().getIntValue();
                if (i == 1) {
                    whileScope.resetScopeIterator();
                    this.pushScope(whileScope);
                    interpretBlock(block);
                    this.popScope();
                    if (functionIsReturning)
                        return;
                }
                else
                    break;
            }
        }
        catch (TGTypeException ex){
            System.out.println(new TGTypeException(ex.getMessage(),test.getLine()).toString());
            System.exit(-1);
        }
	}
	
	private void doForLoop(Tree forTree){
		if (debug)
			System.out.println("dbg: doForLoop-- top scope="+this.symbolTable.peek());
		TGScope forScope = this.symbolTable.peek().getNextChildScope();
		if (debug)
			System.out.println("dbg: doForLoop-- forScope="+forScope.toString());
		
		pushScope(forScope);
                TGHash result;
                Tree init = forTree.getChild(0);
                Tree test = forTree.getChild(1);
                Tree count = forTree.getChild(2);
                Tree block = forTree.getChild(3);
        
        try {
        	interpretStatement(init);
        	while (true){
        		result = evaluateExpression(test);
        		int i = result.getFirstSymbol().getNumValue().getIntValue();
                if (i == 1) {
                	/* reset the child scope iterator for each loop execution */
                	this.symbolTable.peek().resetScopeIterator();
                    interpretBlock(block);
                    interpretStatement(count);
                    if (functionIsReturning)
                        break;
                }
                else
                    break;
            }
        }
        catch (TGTypeException ex){
             System.out.println(new TGTypeException(ex.getMessage(), test.getLine()).toString());
             System.exit(-1);
        }
        popScope();
	}
	
	private void doForeachLoop(Tree foreachTree) {
		try {
			TGScope foreachScope = this.symbolTable.peek().getNextChildScope();
			TGHash itemList = getSymbol(foreachTree.getChild(2).getText(), foreachTree.getLine());
			this.pushScope(foreachScope);
			TGHash iterator = getSymbol(foreachTree.getChild(1).getText(), foreachTree.getLine());
			
			if (debug)
				System.out.println("dbg: doForeachLoop-- itemList="+itemList.toString()+
						", iterator="+iterator.toString());
			
			TGSymbol[] listSymbols = itemList.getSymbols();
			for (int i = 0; i < listSymbols.length; i++) {
				TGHash itrVal = new TGHash();
				itrVal.addAsLast(listSymbols[i]);
				iterator.setValue(itrVal);
				foreachScope.resetScopeIterator();
				
				interpretBlock(foreachTree.getChild(3));
				if (functionIsReturning)
                    			break;
			}
			
			this.popScope();
		}
		catch (TGSymbolNotFoundException e) {
			System.out.println(e.getMessage());
			System.exit(-1);
		}
		catch (TGTypeException e) {
			System.out.println(new TGTypeException(e.getMessage(), foreachTree.getLine()).toString());
			System.exit(-1);
		}
		
	}

	private void doIf (Tree ifTree) {
	    TGHash result;
	    Tree test = ifTree.getChild(0);
	    Tree block = ifTree.getChild(1);
	    try {
	        result = evaluateExpression(test);
	        int i = result.getFirstSymbol().getNumValue().getIntValue();
	        if (i == 1) {
	        	pushScope(symbolTable.peek().getNextChildScope());
	        	interpretBlock(block);
	        	popScope();
	        	this.ifJustExecuted = true;
	        	
	        	if (functionIsReturning)
	        		return;
	        }
	        else
	        	symbolTable.peek().getNextChildScope(); // discard the if's scope
	    } catch (TGTypeException ex){
	    	System.out.println(ex.getMessage());
	    	System.exit(-1);
	    }
	}
	
	private void doElseif(Tree elseifTree) {
		if (!this.ifJustExecuted)
			doIf(elseifTree);
	}
	
	private void doElse(Tree elseTree) {
		this.pushScope(this.symbolTable.peek().getNextChildScope());
		if (!this.ifJustExecuted)
			interpretBlock(elseTree.getChild(0));
		this.popScope();
	}
	
	/**
	 * Gets the FUNC_DEF node corresponding to the function with the specified name
	 * @param functionName the name of the function to find in the tree
	 * @return a tree with the root being the FUNC_DEF node of the function declaration
	 * 			for the function of the specified name
	 * ~!~Del
	 */
	private Tree getFunctionTree(String functionName)
			throws TGSymbolNotFoundException
	{
		int i;
		for (i = 0; i < this.ast.getChildCount(); i++) {
			Tree funcDefCandidate = this.ast.getChild(i);
			if (funcDefCandidate.getType() == TableGenParser.FUNC_DEF &&
					funcDefCandidate.getChild(0).getText().equals(functionName)) 
			{
				return funcDefCandidate;
			}
		}
		
		throw new TGSymbolNotFoundException(functionName);
	}
	
	/**
	 * Only looks inside the current scope for setting the value of the symbol with the
	 * specified name -- created primarily for initializing parameters before a function call.
	 * This function assumes that the scope to search has already been pushed onto the
	 * symbol table and is on the top of the symbol table stack.
	 * @param symbolName the name of the symbol to set the value of
	 * @param value the value to which to set the symbol
	 * @param type the type of the passed Object value -- as in TGSymbol
	 * @throws TGSymbolNotFoundException when the symbol is not found in the current scope
	 * @throws TGTypeException when setting the value fails because of invalid types
	 * 
	 * ~!~Del
	 */
	private void setValueInScope(String symbolName, Object value, int type, TGScope scope)
			throws TGSymbolNotFoundException, TGTypeException
	{
		TGHash[] symbols = scope.getSymbols();
		
		for (int i = 0; i < symbols.length; i++) {
			if (symbols[i].getName().equals(symbolName)) {
				if (type == INTEGER_SYMBOL_TYPE)
					symbols[i].setValue((Integer)value);
				else if (type == DOUBLE_SYMBOL_TYPE)
					symbols[i].setValue((Double)value);
				else if (type == STRING_SYMBOL_TYPE)
					symbols[i].setValue((String)value);
				else if (type == HASH_SYMBOL_TYPE)
					symbols[i].setValue((TGHash)value);
				else
					throw new RuntimeException("Invalid type passed " +
								   "to Interpreter.setCurrentScopeSymbolValue");
			}
		}
	}
	
	/**
	 * Use this method to set the value of a symbol.  It will throw the appropriate
	 * exceptions based on the symbol table values and the type of the parameter
	 * @param symbolName the name of the symbol of which to change the value
	 * @param value the value to which to set the symbol-- this may be of type Integer,
	 * 			Double, String or TGHash
	 * @param type the static final flag as declared at the top of the file corresponding to
	 * 			the type of the value passed in
	 * @throws TGSymbolNotFoundException when the symbol could not be found in the symbol table
	 * @throws TGTypeException when the underlying assignment implementation detects a type issue 
	 * 			due to programmer-imposed type constraints in the program
	 * 
	 * ~!~Del
	 */
	private void setSymbolValue(String symbolName, Object value, int type) 
			throws TGSymbolNotFoundException, TGTypeException
	{	
		boolean found = false;
		TGScope[] scopes = new TGScope[this.symbolTable.size()];
		scopes = this.symbolTable.toArray(scopes);
		
		for (int i = 0; i < scopes.length; i++) {
			TGHash[] symbols = scopes[i].getSymbols();
			
			for (int j = 0; j < symbols.length; j++) {
				if (symbols[j].getName().equals(symbolName)) {
					found = true;
					if (type == INTEGER_SYMBOL_TYPE)
						this.symbolTable.get(i).getSymbol(j).setValue((Integer)value);
					else if (type == DOUBLE_SYMBOL_TYPE)
						this.symbolTable.get(i).getSymbol(j).setValue((Double)value);
					else if (type == STRING_SYMBOL_TYPE)
						this.symbolTable.get(i).getSymbol(j).setValue((String)value);
					else if (type == HASH_SYMBOL_TYPE)
						this.symbolTable.get(i).getSymbol(j).setValue((TGHash)value);
					else
						throw new RuntimeException("Invalid type passed to " + 
									   "Interpreter.setSymbolValue");
				}
			}
			
			if (found)
				break;
		}
		
		if (!found)
			throw new TGSymbolNotFoundException(symbolName);
	}
	
	/**
	 * Gets the TGSymbol of the corresponding name from the symbol table
	 * @param symbolName the name of the symbol to return
	 * @param lineNumber the line number of the program on which the symbol is referenced
	 * @return the TGSymbol object with the specified name
	 * @throws TGSymbolNotFoundException if the symbol is not currently
	 * 			visible in the symbol table
	 * 
	 * ~!~Del
	 */
	private TGHash getSymbol(String symbolName, int lineNumber)
			throws TGSymbolNotFoundException
	{
		TGHash result = null;
		TGScope[] scopes = new TGScope[this.symbolTable.size()];
		scopes = this.symbolTable.toArray(scopes);
		
		for (int i = 0; i < scopes.length; i++) {
			TGHash[] symbols = scopes[i].getSymbols();
			
			for (int j = 0; j < symbols.length; j++) {
				if (symbols[j].getName().equals(symbolName))
					result = this.symbolTable.get(i).getSymbol(j);
			}
			
			if (result != null)
				break;
		}
		
		if (result == null)
			throw new TGSymbolNotFoundException(symbolName, lineNumber);
		return result;
	}
	
	private void pushScope(TGScope scope) {
		this.symbolTable.addFirst(scope);
	}
	
	/**
	 * Implementation of the "." operator to get a value out of a hash
	 * @param hashName the name of the actual symbol to look inside
	 * @param hashKey the hash key of the value inside the symbol's element list
	 * @param lineNumber the line number the request is coming from for proper
	 * 			formation of error messages
	 * @return a new TGHash object holding the value extracted from the specified hash
	 * 
	 * ~!~Del
	 */
	private TGHash lookupSymbolInHash(String hashName, String hashKey, int lineNumber) 
			throws TGSymbolNotFoundException
	{
		TGHash result = null;
		TGSymbol element;
		
		try {
			result = getSymbol(hashName, lineNumber);
			element = result.getElement(hashKey);
			if (element.isRuntimeDouble())
				result.setValue(element.getDoubleValue());
			else if (element.isRuntimeInt())
				result.setValue(element.getIntegerValue());
			else if (element.isRuntimeString())
				result.setValue(element.getStringValue());
			else {
				System.out.println("Unrecognized runtime type in lookupSymboInHash on line " 
						+ lineNumber);
				System.exit(-1);
			}
		}
		catch (TGTypeException e) {
			System.out.println(new TGTypeException(e.getMessage(), lineNumber).toString());
			System.exit(-1);
		}
		catch (TGSymbolNotFoundException e) {
			System.out.print(e.getMessage());
			System.out.println (" in lookupSymbolInHash.");
			System.exit (-1);
		}
		
		return result;
	}
	
	/**
	 * Discards the scope on the top of the symbolTable scope stack
	 */
	private void popScope() {
		/* the scope is gone, so zero all the values here so they are ready
		 * for the next time the scope is used
		 */
		TGScope popped = this.symbolTable.removeFirst();
		TGHash[] symbols = popped.getSymbols();
		
		for (int i = 0; i < symbols.length; i++)
			symbols[i].zero();
	}
	
	/**
	 * Evaluates the given expression tree
	 * @param expressionTree the tree representing the expression to evaluate
	 * @return a temporary TGHash variable
	 */
	private TGHash evaluateExpression(Tree expressionTree) 
			throws TGMissingReturnValueException
	{
		TGHash result = null, x = null, y = null;
		
		try {
		    if (expressionTree.getType() == TableGenParser.FUNC_CALL) {
		    	boolean pushedReturnValue = doFuncCall(expressionTree);
		    	if (debug)
		    		System.out.println ("Finished FuncCall.");
		    	if (pushedReturnValue)
				// This only happens when function has a return 
				// statement. Functions without return statements 
				// don't push anything onto the stack, hense why 
				// we have to do the following.
				if (!this.returnValues.isEmpty ())
		    			result = this.returnValues.pop();
				if (debug)
					System.out.println ("Return value popped: " + 
					result.toString () + " " + returnValues.empty());
		    }

		    else if (expressionTree.getType() == TableGenParser.Identifier) {
		    	result = getSymbol(expressionTree.getText(), expressionTree.getLine());
		    }
			
		    else if (expressionTree.getType() == TableGenParser.Number_literal){
		        result = new TGHash();
		        try {
            	    	int value = Integer.parseInt(expressionTree.getText());
            	    	result.addAsFirst(new TGSymbol("temp", new TGNum(value)));
            	}
            	catch (NumberFormatException es) {
            		try {
            	        double value = Double.parseDouble(expressionTree.getText());
            	        result.addAsFirst(new TGSymbol("temp", new TGNum(value)));
            		}
            		catch (NumberFormatException ex) {
            			throw new RuntimeException("Unable to parse '" + 
            	    		  expressionTree.getText() +
            	    		  "' as a number literal on line " + expressionTree.getLine());
            	   }
            	}
	        }
	        
	        else if (expressionTree.getType() == TableGenParser.String_literal) {
	        	result = new TGHash("temp", TGSymbol.TG_STR_TYPE);
	            String value = expressionTree.getText();
	            result.addAsFirst(new TGSymbol("temp", value));
	        }
	        
	        else if (expressionTree.getType() == TableGenParser.ADDITIVE_OP) {
	        	result = new TGHash("temp", TGSymbol.TG_NUM_TYPE);
	            x = evaluateExpression(expressionTree.getChild(0));
	            y = evaluateExpression(expressionTree.getChild(1));
	            if (expressionTree.getText().equals("+"))
	                result = add(x, y);
	            else
	                result = subtract(x, y);
	            
	        }
	        
	        else if (expressionTree.getType() == TableGenParser.MULTIPLICATIVE_OP) {
	        	result = new TGHash("temp", TGSymbol.TG_NUM_TYPE);
	            x = evaluateExpression(expressionTree.getChild(0));
	            y = evaluateExpression(expressionTree.getChild(1));
	            if (expressionTree.getText().equals("*"))
	                result = mult(x, y);
	            else
	                result = divide(x, y);
	        }

	        else if (expressionTree.getType() == TableGenParser.BOOL_OP) {
	        	result = new TGHash("temp", TGSymbol.TG_NUM_TYPE);
	            x = evaluateExpression(expressionTree.getChild(0));
	            y = evaluateExpression(expressionTree.getChild(1));
	            result = compare(x, y, expressionTree.getText());
	        }
	        
	        else if (expressionTree.getType() == TableGenParser.LOGICAL_OP) {
	        	result = new TGHash("temp", TGSymbol.TG_NUM_TYPE);
	            x = evaluateExpression(expressionTree.getChild(0));
	            y = evaluateExpression(expressionTree.getChild(1));
	            
	            if (!x.getFirstSymbol().isRuntimeNum() || !y.getFirstSymbol().isRuntimeNum())
	                throw new TGTypeException("attempted operation includes a non-number value");
	            int left = x.getFirstSymbol().getIntegerValue().intValue();
	            int right = y.getFirstSymbol().getIntegerValue().intValue();
	            if (expressionTree.getText().equals("&&")){
	                if ((left == 1) && (right == 1))
	                    result.addAsFirst(new TGSymbol("temp", new TGNum(1)));
	                else
	                    result.addAsFirst(new TGSymbol("temp", new TGNum(0)));
	            }
	            else if (expressionTree.getText().equals("||")){
	                if ((left == 1) || (right == 1))
	                    result.addAsFirst(new TGSymbol("temp", new TGNum(1)));
	                else
	                    result.addAsFirst(new TGSymbol("temp", new TGNum(0)));
	            }
	        }
			
	        else if (expressionTree.getType() == TableGenParser.DECIMAL_POINT) {
	        	String hashName = expressionTree.getChild(0).getText();
	        	String hashKey = expressionTree.getChild(1).getText();
			if (debug)
	        		System.out.println("dbg: evaluateExpression-- inside DECIMAL_POINT case");
	        	result = lookupSymbolInHash(hashName, hashKey, expressionTree.getLine());
			if (debug)
	        		System.out.println("\t\t result="+result.toString());
	        }
		    
	        else if (expressionTree.getType() == TableGenParser.NUMERICAL_INDEX) {
	        	result = new TGHash();
	        	String listName = expressionTree.getChild(0).getText();
	        	TGHash index = evaluateExpression(expressionTree.getChild(1));
	        	
	        	if (index.getSymbolCount() != 1)
	        		throw new TGTypeException("array index access with a value that does not have "+
	        				"exactly 1 element");
	        	if (index.getFirstSymbol().getRuntimeType() != TGSymbol.TG_NUM_TYPE)
	        		throw new TGTypeException("array index access with an expression that does not " +
	        				"evaluate to a numeric value");
	        	if (!index.getFirstSymbol().isRuntimeInt())
	        		throw new TGTypeException("array index access with an expression that evaluates " +
	        				"do a fractional number rather than an integer");
	        	
	        	TGHash symbol = getSymbol(listName, expressionTree.getLine());
	        	int indexValue = index.getFirstSymbol().getIntegerValue();
	        	if (debug)
	        		System.out.println("dbg: evaluateExpression--NUMERICAL_INDEX, symbol="+ symbol.toString());
	        	if (indexValue >= symbol.getSymbolCount())
	        		throw new TGTypeException("array index ("+indexValue+") exceeds # of elements "+
	        				"in the array ("+symbol.getSymbolCount()+")");
	        	
	        	TGSymbol value = symbol.getSymbol(indexValue);
	        	result.addAsLast(value);
	        }
			
	        else if (expressionTree.getType() == TableGenParser.HASH_ASSIGN) {
			if (debug)
	        		System.out.println("dbg: evaluateExpression-- inside HASH_ASSIGN case");
	        	
	        	result = new TGHash();
	        	String hashKey = expressionTree.getChild(0).getText();
	        	TGHash value = evaluateExpression(expressionTree.getChild(1));
	        	
	        	/* change this to allow for lists of lists */
	        	if (value.getSymbolCount() != 1) {
	        		System.out.println("Invalid expression after list value for key '" +
	        				hashKey + "'");
	        		System.exit(-1);
	        	}
	        	
	        	value.getFirstSymbol().setKey(hashKey);
	        	result.addAsLast(value.getFirstSymbol());
	        	
			if (debug)
	        		System.out.println("dbg: evaluateExpression-- ending HASH_ASSIGN case");
	        }
			
	        else if (expressionTree.getType() == TableGenParser.COMMA) {
	        	TGHash leftValue = evaluateExpression(expressionTree.getChild(0));
	        	TGHash rightValue = evaluateExpression(expressionTree.getChild(1));
	        	leftValue.concatenateToEnd(rightValue);
	        	result = leftValue;
	        	
	        	if (debug)
	        		System.out.println("dbg: evaluateExpression-- concatenation result="+result.toString());
	        }
			
		    if (result == null) {
		    	if (debug)
		    		System.out.println("dbg: null result in evaluateExpression at line " + 
					           expressionTree.getLine() + ". " + expressionTree.toStringTree ());
				//no exit because of return statements
		    }
		}
		catch (TGTypeException e) {
			System.out.println(new TGTypeException(e.getMessage(), 
					expressionTree.getLine()).toString());
			System.exit(-1);
		}
		catch (TGSymbolNotFoundException e) {
			System.out.print(e.getMessage());
			System.out.println (" in evaluateExpression.");
			System.exit(-1);
		}
		
		return result;
	}
	
	/**
	 * Adds the numerical values in the hashes and throws the appropriate errors,
	 * giving back a new TGHash object that is restricted to either NUM or STR type
	 * @param x a TGHash to add to the other parameter
	 * @param y a TGHash to add to the other parameter
	 * @return a TGHash object containing the result value as its first and only 
	 * 			symbol that is restricted to hold only that symbol's type
	 * @throws TGTypeException if either of the parameters has more than one element
	 * 			or one of the values to add is a non-number type
	 * 
	 * ~!~Del
	 */
	public static TGHash add(TGHash x, TGHash y) 
		throws TGTypeException
	{
		TGHash result = new TGHash();
		result.setType(TGSymbol.TG_NUM_TYPE);
		
		if (x.getSymbolCount() != 1 || y.getSymbolCount() != 1)
			throw new TGTypeException("invalid symbol count in add operation");
		
		if (!x.getFirstSymbol().isRuntimeNum() || !y.getFirstSymbol().isRuntimeNum())
			throw new TGTypeException("attempted addition includes a non-number value");
		
		/* both operands are integers -- the result is also an integer */
		if (x.getFirstSymbol().isRuntimeInt() && y.getFirstSymbol().isRuntimeInt()) {
			int value = x.getFirstSymbol().getIntegerValue().intValue() +
						y.getFirstSymbol().getIntegerValue().intValue();
			result.addAsFirst(new TGSymbol("temp", new TGNum(value)));
		}
		
		/* at least one operand is a double -- the result is also a double */
		else {
			if (x.getFirstSymbol().getNumValue().isInt()) {
				int xVal = x.getFirstSymbol().getIntegerValue().intValue();
				double yVal = y.getFirstSymbol().getDoubleValue().doubleValue();
				result.addAsFirst(new TGSymbol("temp", new TGNum(xVal+yVal)));
			}
			else {
				double xVal = x.getFirstSymbol().getDoubleValue().doubleValue();
				if (y.getFirstSymbol().getNumValue().isInt()) {
					int yVal = y.getFirstSymbol().getIntegerValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal+yVal)));
				}
				else {
					double yVal = y.getFirstSymbol().getDoubleValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal+yVal)));
				}
			}
		}
		
		return result;
	}
	
	/**
	 * Subtracts the numerical values in the hashes and throws the appropriate errors,
	 * giving back a new TGHash object that is restricted to either NUM or STR type
	 * @param x a TGHash to have the other parameter subtracted from it
	 * @param y a TGHash to subtract from the other parameter
	 * @return a TGHash object containing the result value as its first and only 
	 * 			symbol that is restricted to hold only that symbol's type
	 * @throws TGTypeException if either of the parameters has more than one element
	 * 			or one of the values to add is a non-number type
	 */
	public static TGHash subtract(TGHash x, TGHash y) 
		throws TGTypeException
	{
		TGHash result = new TGHash();
		result.setType(TGSymbol.TG_NUM_TYPE);
		
		if (x.getSymbolCount() != 1 || y.getSymbolCount() != 1)
			throw new TGTypeException("invalid symbol count in add operation");
		
		if (!x.getFirstSymbol().isRuntimeNum() || !y.getFirstSymbol().isRuntimeNum())
			throw new TGTypeException("attempted addition includes a non-number value");
		
		/* both operands are integers -- the result is also an integer */
		if (x.getFirstSymbol().isRuntimeInt() && y.getFirstSymbol().isRuntimeInt()) {
			int value = x.getFirstSymbol().getIntegerValue().intValue() -
						y.getFirstSymbol().getIntegerValue().intValue();
			result.addAsFirst(new TGSymbol("temp", new TGNum(value)));
		}
		
		/* at least one operand is a double -- the result is also a double */
		else {
			if (x.getFirstSymbol().getNumValue().isInt()) {
				int xVal = x.getFirstSymbol().getIntegerValue().intValue();
				double yVal = y.getFirstSymbol().getDoubleValue().doubleValue();
				result.addAsFirst(new TGSymbol("temp", new TGNum(xVal-yVal)));
			}
			else {
				double xVal = x.getFirstSymbol().getDoubleValue().doubleValue();
				if (y.getFirstSymbol().getNumValue().isInt()) {
					int yVal = y.getFirstSymbol().getIntegerValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal-yVal)));
				}
				else {
					double yVal = y.getFirstSymbol().getDoubleValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal-yVal)));
				}
			}
		}
		
		return result;
	}
        
	/**
	 * Multiplies the numerical values in the hashes and throws the appropriate errors,
	 * giving back a new TGHash object that is restricted to either NUM or STR type
	 * @param x a TGHash multiplicand 1
	 * @param y a TGHash multiplicand 2
	 * @return a TGHash object containing the result value as its first and only 
	 * 			symbol that is restricted to hold only that symbol's type
	 * @throws TGTypeException if either of the parameters has more than one element
	 * 			or one of the values to add is a non-number type
	 */
	public static TGHash mult(TGHash x, TGHash y) 
		throws TGTypeException
	{
		TGHash result = new TGHash();
		result.setType(TGSymbol.TG_NUM_TYPE);
		
		if (x.getSymbolCount() != 1 || y.getSymbolCount() != 1)
			throw new TGTypeException("invalid symbol count in multiply operation");
		
		if (!x.getFirstSymbol().isRuntimeNum() || !y.getFirstSymbol().isRuntimeNum())
			throw new TGTypeException("attempted multiplication includes a non-number value");
		
		/* both operands are integers -- the result is also an integer */
		if (x.getFirstSymbol().isRuntimeInt() && y.getFirstSymbol().isRuntimeInt()) {
			int value = x.getFirstSymbol().getIntegerValue().intValue() *
						y.getFirstSymbol().getIntegerValue().intValue();
			result.addAsFirst(new TGSymbol("temp", new TGNum(value)));
		}
		
		/* at least one operand is a double -- the result is also a double */
		else {
			if (x.getFirstSymbol().getNumValue().isInt()) {
				int xVal = x.getFirstSymbol().getIntegerValue().intValue();
				double yVal = y.getFirstSymbol().getDoubleValue().doubleValue();
				result.addAsFirst(new TGSymbol("temp", new TGNum(xVal*yVal)));
			}
			else {
				double xVal = x.getFirstSymbol().getDoubleValue().doubleValue();
				if (y.getFirstSymbol().getNumValue().isInt()) {
					int yVal = y.getFirstSymbol().getIntegerValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal*yVal)));
				}
				else {
					double yVal = y.getFirstSymbol().getDoubleValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal*yVal)));
				}
			}
		}
		
		return result;
	}
        
	/**
	 * Divides the numerical values in the hashes and throws the appropriate errors,
	 * giving back a new TGHash object that is restricted to either NUM or STR type
	 * @param x a TGHash dividend
	 * @param y a TGHash divisor
	 * @return a TGHash object containing the result value as its first and only 
	 * 			symbol that is restricted to hold only that symbol's type
	 * @throws TGTypeException if either of the parameters has more than one element
	 * 			or one of the values to add is a non-number type
	 */
	public static TGHash divide(TGHash x, TGHash y) 
		throws TGTypeException
	{
		TGHash result = new TGHash();
		result.setType(TGSymbol.TG_NUM_TYPE);
		
		if (x.getSymbolCount() != 1 || y.getSymbolCount() != 1)
			throw new TGTypeException("invalid symbol count in divide operation");
		
		if (!x.getFirstSymbol().isRuntimeNum() || !y.getFirstSymbol().isRuntimeNum())
			throw new TGTypeException("attempted division includes a non-number value");
		
		// Division must return double if dividing two ints such as: 4 / 3 
                if (x.getFirstSymbol().isRuntimeInt() && y.getFirstSymbol().isRuntimeInt()) {
                       boolean remainder = true;
                       if (x.getFirstSymbol().getIntegerValue().intValue() %
                           y.getFirstSymbol().getIntegerValue().intValue() > 0)
                               remainder = false;
                       double value = (double)x.getFirstSymbol().getIntegerValue().intValue() /
                                      (double)y.getFirstSymbol().getIntegerValue().intValue();
                       /* If the arithmetic is returns int, return int, otherwise return double.
                        */
                       if (remainder)
                               result.addAsFirst(new TGSymbol("temp", new TGNum((int)value)));
                       else
                               result.addAsFirst(new TGSymbol("temp", new TGNum(value)));
                }
		
		/* at least one operand is a double -- the result is also a double */
		else {
			if (x.getFirstSymbol().getNumValue().isInt()) {
				int xVal = x.getFirstSymbol().getIntegerValue().intValue();
				double yVal = y.getFirstSymbol().getDoubleValue().doubleValue();
				result.addAsFirst(new TGSymbol("temp", new TGNum(xVal/yVal)));
			}
			else {
				double xVal = x.getFirstSymbol().getDoubleValue().doubleValue();
				if (y.getFirstSymbol().getNumValue().isInt()) {
					int yVal = y.getFirstSymbol().getIntegerValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal/yVal)));
				}
				else {
					double yVal = y.getFirstSymbol().getDoubleValue();
					result.addAsFirst(new TGSymbol("temp", new TGNum(xVal/yVal)));
				}
			}
		}
		
		return result;
	}
        
	/**
	 * Compares the numerical values in the hashes and throws the appropriate errors,
	 * giving back 1 (true) or 0 (false)
	 * @param x a TGHash number 1
	 * @param y a TGHash number 2
	 * @return a TGHash object containing TGNum 1 or 0
	 * @throws TGTypeException if either of the parameters has more than one element
	 * 			or one of the values to add is a non-number type
	 * 
	 * Daniel
	 * ~!~Del -- modified to allow string comparison as per the LRM
	 */
        public static TGHash compare(TGHash x, TGHash y, String type) throws TGTypeException {
            TGHash result = new TGHash();
            
            if (x.getSymbolCount() != 1 || y.getSymbolCount() != 1)
                    throw new TGTypeException("invalid symbol count in comparison operation");
            
            result.setType(TGSymbol.TG_NUM_TYPE);
            
            /* string comparison */
            if (x.getFirstSymbol().getRuntimeType() == TGSymbol.TG_STR_TYPE &&
            		y.getFirstSymbol().getRuntimeType() == TGSymbol.TG_STR_TYPE) 
            {
            	String xVal = x.getFirstSymbol().getStringValue();
        		String yVal = y.getFirstSymbol().getStringValue();
        		
        		if ( (xVal.equals(yVal) && type.equals("==")) || (!xVal.equals(yVal) && type.equals("!=")))
        			result.addAsFirst(new TGSymbol("temp", new TGNum(1)));
        		else
        			result.addAsFirst(new TGSymbol("temp", new TGNum(0)));
            }
            else {
            	try {
            		if (!x.getFirstSymbol().isRuntimeNum() || !y.getFirstSymbol().isRuntimeNum())
                        throw new TGTypeException("invalid comparision: cannot compare strings and numbers");
            		
            		double left = x.getFirstSymbol().getDoubleValue().doubleValue();
                	double right = y.getFirstSymbol().getDoubleValue().doubleValue();

                	if (compare (left, right, type))
            			result.addAsFirst(new TGSymbol("temp", new TGNum(1)));
            		else
            			result.addAsFirst(new TGSymbol("temp", new TGNum(0)));
        	    } catch (TGTypeException e1) {
        	    	int left = x.getFirstSymbol().getIntegerValue().intValue();
        	    	int right = y.getFirstSymbol().getIntegerValue().intValue();

        	    	if (compare (left, right, type))
        	    		result.addAsFirst(new TGSymbol("temp", new TGNum(1)));
        	    	else
        	    		result.addAsFirst(new TGSymbol("temp", new TGNum(0)));
        	    }
        	}
    			
    	    return result;
        }

	public static boolean compare (double left, double right, String type) {
		return (type.equals("==") && (left == right)) ||
                	(type.equals("!=") && (left != right)) ||
                	(type.equals("<") && (left < right)) ||
                	(type.equals(">") && (left > right)) ||
                	(type.equals("<=") && (left <= right)) ||
                	(type.equals(">=") && (left >= right));
	}

	public static boolean compare (int left, int right, String type) {
		return (type.equals("==") && (left == right)) ||
                        (type.equals("!=") && (left != right)) ||
                        (type.equals("<") && (left < right)) ||
                        (type.equals(">") && (left > right)) ||
                        (type.equals("<=") && (left <= right)) ||
                        (type.equals(">=") && (left >= right));
	}
	
	/**
	 * Outputs the table data in tab-delimited format
	 */
	private void outputTable() {
		for(int i = 0; i <= maxRow; ++i)
		{
			int size = outputTable.get (i).size();
			for(int j = 0; j < size; ++j)
			{
				String str = outputTable.get(i).get(j);
				
				// Make java not print "nulls"
				try {
					str.length (); // this is a random op
						       // can be anything
				} catch (NullPointerException e) {
					System.out.print("\t");
					continue;
				}
				
				// Remove quotes from strings.
				str = str.replaceFirst ("\"", "");
				str = str.replaceAll ("\"$", "");
				
				// Remove quotes and commas from between strings.
				str = str.replaceAll ("\",[ ]*\"", " ");

				// Remove escapes ("\")
				str = str.replace ("\\", "");

				System.out.print(str);
                         	
				// only prints tabs between rows
				// and not at the end of the table
				if (j < size - 1)
					System.out.print("\t");
			}


			// only print new lines between rows
			// and not at the end of the table
            		if (i < outputTable.size() - 1)
            		System.out.print("\n");
		}
	}
}
