grammar TableGen;

options {
	ASTLabelType=CommonTree;
	output=AST;
	k=2;
}

tokens {
	BLOCK;
	CONDITIONAL;
	COORD;
	FUNC_CALL;
	FUNC_DEF;
	LIST_CONCAT;
	PROGRAM;
	TYPE_DECL;
	VAR_LOCATION;
	NUMERICAL_INDEX;
}

/* ===================== BEGIN LEXER =========================*/

/******************* Primitives ************************/
ASSIGNMENT_OP : '=' ;

ADDITIVE_OP : '+' | '-' ;

BOOL_OP : '<' | '>' | '<=' | '>=' | '==' | '!=';

COLON : ':' ;

COMMA : ',' ;

COORD_SEPARATOR : '|' ;

DECIMAL_POINT : '.' ;

//DIGIT : '\u0030'..'\u0039' ;
fragment
DIGIT : '0'..'9' ;

ELSE_KEYWORD : 'else' ;

ELSEIF_KEYWORD : 'elseif' ;

FOREACH_KEYWORD : 'foreach' ;

FOR_KEYWORD : 'for' ;

FUNC_KEYWORD : 'func' ;	

HASH_ASSIGN : '>>' ;

IF_KEYWORD : 'if' ;

IN_KEYWORD : 'in' ;

L_BRACE : '{' ;

L_BRACKET : '[' ;

L_PAREN : '(' ;

fragment
LETTER : 'a'..'z' | 'A'..'Z' ;

LINE_TERMINATOR
	: ('\n' '\r') => '\n' '\r'
	| '\n'
	| '\r'
	;
	
LOGICAL_OP
	: '&&' 
	| '||'
	;

ML_COMMENT_BEGIN : '/*';

ML_COMMENT_END : '*/';

MULTIPLICATIVE_OP : '*' | '/' ;
	
PUT_OP 	: '->' ;


QUOTES  : '"' ;

R_BRACE : '}' ;	

R_BRACKET : ']' ;

R_PAREN : ')' ;	

RETURN_KEYWORD : 'return' ;

SEMICOLON : ';' ;

SL_COMMENT : '//' ;

fragment
UNDERSCORE : '_' ;

VARTYPE : 'str' | 'num' | 'list';	

// as at: http://www.columbia.edu/kermit/utf8-t1.html, verified OK from
//   http://www.unicode.org/charts/PDF/U0000.pdf
fragment
VALID_CHARS : '\u0020'..'\u007e' ;

WHILE_KEYWORD : 'while' ;

WS  :  (' '|'\t') {$channel=HIDDEN; skip();} ;

Ml_Comment : ML_COMMENT_BEGIN (options {greedy=false;} : .*) ML_COMMENT_END {$channel=HIDDEN; skip();} ;
Sl_Comment : SL_COMMENT (options {greedy=false;} : .*) LINE_TERMINATOR {$channel=HIDDEN; skip();} ;

            // backslash
Escaped_char : '\u005c' ('"'| '-'| '+'| '/'| ',') ;

Identifier : UNDERSCORE? LETTER (LETTER | DIGIT | UNDERSCORE)* ; 

String_literal : QUOTES (Escaped_char | ~('"' | '\u005c'))* QUOTES ;

Number_literal
	: '-'? DIGIT+ (DECIMAL_POINT DIGIT+)?;


/* ===================== END LEXER ============================ */	
	
	
literal
	: Number_literal
	| String_literal
	;
	
coordinate : L_BRACKET expression COORD_SEPARATOR expression R_BRACKET
		-> ^(COORD expression expression) ;
	
/****************** EVERY POSSIBLE expression *********************/

/* chain of expression statements to assign precedence */
expression : bool_expr1 ;

bool_expr1
	: (bool_expr2 -> bool_expr2)
		(LOGICAL_OP bool_expr2 -> ^(LOGICAL_OP $bool_expr1 bool_expr2) )*
	;

bool_expr2
	: (add_expr -> add_expr)
		(BOOL_OP add_expr -> ^(BOOL_OP $bool_expr2 add_expr))*
	;

add_expr
	: (mult_expr -> mult_expr)
		( ADDITIVE_OP mult_expr -> ^(ADDITIVE_OP $add_expr mult_expr) )*
	;

mult_expr
	: (list_concat_expr -> list_concat_expr)
		( MULTIPLICATIVE_OP list_concat_expr 
			-> ^(MULTIPLICATIVE_OP $mult_expr list_concat_expr) )*
	;

list_concat_expr
	: (primary_expr -> primary_expr)
		( COMMA primary_expr -> ^(COMMA $list_concat_expr primary_expr) )*
	;

primary_expr
	: (par_expression -> par_expression)
	| (hash_key_specification) => (hash_key_specification -> hash_key_specification)
	| (identifier_reference -> identifier_reference)
	| (literal -> literal)
	;

par_expression : L_PAREN expression* R_PAREN -> expression ;

identifier_reference 
	: (Identifier L_PAREN) => 
		Identifier L_PAREN (func_call_param /*COMMA?*/)* R_PAREN -> ^(FUNC_CALL Identifier func_call_param*)
	| (Identifier DECIMAL_POINT) =>
		( i1=Identifier DECIMAL_POINT i2=Identifier -> ^(DECIMAL_POINT $i1 $i2) )
	| (Identifier L_BRACKET) => ( Identifier L_BRACKET expression R_BRACKET 
			-> ^(NUMERICAL_INDEX Identifier expression) )
	| Identifier
	;

hash_key_specification : Identifier HASH_ASSIGN hash_value
		-> ^(HASH_ASSIGN Identifier hash_value) ;

hash_value 
	: identifier_reference -> identifier_reference
	| literal -> literal
	| par_expression -> par_expression
	;

func_call_param 
	: named_func_param
	| expression
	;

named_func_param : Identifier COLON expression -> ^(COLON Identifier expression) ;
	
// either the location for the variable, or a table position-- 
//   essentially where one can put something or get something from
location
	: identifier_reference 
	| coordinate
	;
	
	
/******************** STATEMENT TYPES *************************/

assignment : (VARTYPE COLON)? Identifier ASSIGNMENT_OP expression
		-> ^(ASSIGNMENT_OP ^(TYPE_DECL VARTYPE?) Identifier  expression);

additional_elements: COMMA expression -> expression ;

t_if : IF_KEYWORD expression block -> ^(IF_KEYWORD expression block) ;
t_elseif : ELSEIF_KEYWORD expression block -> ^(ELSEIF_KEYWORD expression block) ;
t_else : ELSE_KEYWORD block -> ^(ELSE_KEYWORD block) ;

function_definition : FUNC_KEYWORD Identifier func_param* block
	-> ^(FUNC_DEF Identifier func_param* block ) ;
	
func_param
	: typed_func_param
	| untyped_func_param
	;

typed_func_param : VARTYPE COLON Identifier -> ^(COLON VARTYPE Identifier) ;
untyped_func_param : Identifier -> Identifier ;

for_loop : FOR_KEYWORD a1=assign_or_expr SEMICOLON expression SEMICOLON a2=assign_or_expr block
		-> ^(FOR_KEYWORD $a1 expression $a2 block) ;

assign_or_expr : ( ((assignment)=>assignment -> assignment) | (expression -> expression)) ;

// for each $i1 in $i2 , do BLOCK
// allowing the target of the IN keyword to be an arbitrary expression -- check at runtime
foreach_loop :
	FOREACH_KEYWORD (VARTYPE COLON)? Identifier IN_KEYWORD expression block
			-> ^(FOREACH_KEYWORD ^(TYPE_DECL VARTYPE?) Identifier expression block )
	;

while_loop : WHILE_KEYWORD expression block -> ^(WHILE_KEYWORD expression block) ;

return_stmt : RETURN_KEYWORD expression -> ^(RETURN_KEYWORD expression) ;

putop : expression PUT_OP location -> ^(PUT_OP expression location) ;

/******************* HIGH-LEVEL **********************/
program : LINE_TERMINATOR* statement* -> ^(PROGRAM statement*);

statement
	: 
		/* syntactic predicate on putop scans at most one line because the '->'
		 * denotes a putop rather than something else
		 */
		( ((putop) => putop -> putop)
		| (assignment -> assignment)
		| (t_if -> t_if)
		| (t_elseif -> t_elseif)
		| (t_else -> t_else)
		| (function_definition -> function_definition)
		| (for_loop -> for_loop)
		| (foreach_loop -> foreach_loop)
		| (while_loop -> while_loop)
		| (return_stmt -> return_stmt)
		| (expression -> expression)
		| (block -> block)
		) ( LINE_TERMINATOR+ -> $statement )
	;
	
block : L_BRACE LINE_TERMINATOR+ statement* R_BRACE -> ^(BLOCK statement*) ;

