/***********************************************************************

                             The Scanner

 ***********************************************************************/

class MyLexer extends Lexer;
options { k = 2; }

WHITESPACE : (' ' | '\t' | '\n' { newline(); } )+  { $setType(Token.SKIP); } ;
 
protected DIGITS : ('0'..'9')+ ;

NUM : DIGITS ('.' DIGITS { $setType(REAL); } )? ;

AND :    "&&" ;   LE :     "<=" ;   SEMI :   ';'  ;
OR  :    "||" ;   GT :     '>'  ;   LPAREN : '('  ;
ASSIGN : '='  ;   GE :     ">=" ;   RPAREN : ')'  ;
EQ :     "==" ;   LBRACE : '{'  ;   PLUS :   '+'  ;
NOT :    '!'  ;   RBRACE : '}'  ;   MINUS :  '-'  ;
NE :     "!=" ;   LBRACK : '['  ;   MUL :    '*'  ;
LT :     '<'  ;   RBRACK : ']'  ;   DIV :    '/'  ;

ID : ('_' | 'a'..'z' | 'A'..'Z') ('_' | 'a'..'z' | 'A'..'Z' | '0'..'9')* ;

/***********************************************************************

                              The Parser

This parses the following grammar:

program  ->  block
block    ->  { decls stmts }
decls    ->  decls decl | e
decl     ->  type ID ;
type     ->  type [ NUM ] | basic
stmts    ->  stmts stmt | e
stmt     ->  loc = bool ;
         |   IF ( bool ) stmt
         |   IF ( bool ) stmt ELSE stmt
         |   WHILE ( bool ) stmt
         |   DO stmt WHILE ( bool ) ;
         |   BREAK ;
         |   block
         |   ;
loc      ->  loc [ bool ] | ID
bool     ->  bool || join | join
join     ->  join && equality | equality
equality ->  equality == rel | equality != rel | rel
rel      ->  expr < expr | expr <= expr | expr >= expr
         |   expr > expr | expr
expr     -> expr + term | expr - term | term
term     ->  term * unary | term / unary | unary
unary    ->  ! unary | - unary | factor
factor   ->  ( bool ) | loc | NUM | REAL | TRUE | FALSE

The empty statement is not part of the grammar given in the book, but it is
part of the Java parser code in Appendix A.

 ***********************************************************************/

class MyParser extends Parser;
options { buildAST = true; }
tokens  { NEGATE; DECLS; }

program : LBRACE^ decls (stmt)* RBRACE! ;

decls : (decl)*  { #decls = #([DECLS, "DECLS"], #decls); } ;

decl : ("int" | "char" | "bool" | "float") (LBRACK! NUM RBRACK!)* ID SEMI! ;

stmt : loc ASSIGN^ bool SEMI!
     | "if"^ LPAREN! bool RPAREN! stmt (options {greedy=true;}: "else"! stmt)?
     | "while"^ LPAREN! bool RPAREN! stmt
     | "do"^ stmt "while"! LPAREN! bool RPAREN! SEMI!
     | "break" SEMI!
     | program
     | SEMI
     ;

loc      : ID^ (LBRACK! bool RBRACK!)* ;
bool     : join (OR^ join)* ;
join     : equality (AND^ equality)* ;
equality : rel ((EQ^ | NE^) rel)* ;
rel      : expr ((LT^ | LE^ | GT^ | GE^) expr)* ;
expr     : term ((PLUS^ | MINUS^) term)* ;
term     : unary ((MUL^ | DIV^) unary)* ;
unary    : MINUS^ unary { #unary.setType(NEGATE); } | NOT^ unary | factor ;
factor   : LPAREN! bool RPAREN! | loc | NUM | REAL | "true" | "false" ;

/***********************************************************************

   A tree walker that builds up an IR that can easily be converted
			to three-address-code

 ***********************************************************************/

class MyWalker extends TreeParser;
{
  SymbolTable top = null;
  int used = 0; // Number of bytes allocated for local declarations
}

program returns [Stmt s]
{ s = null; Stmt s1;}
  : #(LBRACE
      { SymbolTable saved_environment = top; top = new SymbolTable(top); }
      decls
      s=stmts
      { top = saved_environment; }
    )
  ;

decls
{ Type t = null; }
  : #(DECLS
       (t=type ID { top.put(#ID.getText(), t, used); used += t.width; } )* )
  ;

type returns [Type t]
{ t = null; }
  : ( "bool"  { t = Type.Bool;  }
    | "char"  { t = Type.Char;  }
    | "int"   { t = Type.Int;   }
    | "float" { t = Type.Float; } )
    (t=dims[t])?
  ;

dims[Type t1] returns [Type t]
{ t = t1; }
  : NUM (t=dims[t])? { t = new Array(Integer.parseInt(#NUM.getText()), t); }
  ;

stmts returns [Stmt s]
{ s = null; Stmt s1; }
  : s=stmt (s1=stmts { s = new Seq(s, s1); } )?
  ;

stmt returns [Stmt s]
{ Expr e1, e2;
  s = null;
  Stmt s1, s2;
}
  : #(ASSIGN e1=expr e2=expr
      { if (e1 instanceof Id) s = new Set((Id) e1, e2);
        else s = new SetElem((Access) e1, e2);
      }
    )
  | #("if" e1=expr s1=stmt
       ( s2=stmt { s = new Else(e1, s1, s2); }
       | /* nothing */ { s = new If(e1, s1); } ))
  | #("while"
       { While whilenode = new While();
         s2 = Stmt.Enclosing;
         Stmt.Enclosing = whilenode; }
       e1=expr
       s1=stmt
       { whilenode.init(e1, s1);
         Stmt.Enclosing = s2;
         s = whilenode; } )
  | #("do"
       { Do donode = new Do();
         s2 = Stmt.Enclosing;
         Stmt.Enclosing = donode; }
       s1=stmt
       e1=expr
       { donode.init(s1, e1);
         Stmt.Enclosing = s2;
         s = donode; } )
  | "break" { s = new Break(); }
  | s=program
  | SEMI { s = Stmt.Null; }
  ;

expr returns [Expr e]
{
  Expr a, b;
  e = null;
}
  : #(OR     a=expr b=expr { e = new Or(a, b); } )
  | #(AND    a=expr b=expr { e = new And(a, b); } )
  | #(EQ     a=expr b=expr { e = new Rel("==", a, b); } )
  | #(NE     a=expr b=expr { e = new Rel("!=", a, b); } )
  | #(LT     a=expr b=expr { e = new Rel("<", a, b); } )
  | #(LE     a=expr b=expr { e = new Rel("<=", a, b); } )
  | #(GT     a=expr b=expr { e = new Rel(">", a, b); } )
  | #(GE     a=expr b=expr { e = new Rel(">=", a, b); } )
  | #(PLUS   a=expr b=expr { e = new Arith("+", a, b); } )
  | #(MINUS  a=expr b=expr { e = new Arith("-", a, b); } )
  | #(MUL    a=expr b=expr { e = new Arith("*", a, b); } )
  | #(DIV    a=expr b=expr { e = new Arith("/", a, b); } )
  | #(NOT    a=expr        { e = new Not(a); } )
  | #(NEGATE a=expr        { e = new Unary("-", a); } )
  | NUM                    { e = new Constant(#NUM.getText(), Type.Int); }
  | REAL                   { e = new Constant(#REAL.getText(), Type.Float); }
  | "true"                 { e = Constant.True; }
  | "false"                { e = Constant.False; }
  | #(ID
       { Id i = top.get(#ID.getText());
         if (i == null) System.out.println(#ID.getText() + " undeclared");
         e = i;
       }
       ( a=expr
         { Type type = e.type;
           type = ((Array)type).of;
           Expr w = new Constant(type.width);
           Expr loc = new Arith("*", a, w);
         }
         ( a=expr
           { type = ((Array)type).of;
             w = new Constant(type.width);
             loc = new Arith("+", loc, new Arith("*", a, w));
           }
         )*
         { e = new Access(i, loc, type); }
       )?
     )
  ;
