/*
 * grammar.g : the lexer and the parser, in ANTLR grammar.
 *
 * @author Tiantian Zhou - tz2002@columbia.edu
 * 
 * @version $Id: grammar.g,v 1.16 2003/05/15 17:41:24 hanhua Exp $
 */

class MxAntlrLexer extends Lexer;

options{
    k = 2;  
    charVocabulary = '\3'..'\377';
    testLiterals = false;
    exportVocab = MxAntlr;
}

{
    int nr_error = 0;
    public void reportError( String s ) {
        super.reportError( s );
        nr_error++;
    }
    public void reportError( RecognitionException e ) {
        super.reportError( e );
        nr_error++;
    }
}

protected
ALPHA   : 'a'..'z' | 'A'..'Z' | '_' ;
    
protected
DIGIT   :   '0'..'9';

WS      : (' ' | '\t')+          { $setType(Token.SKIP); }
        ;

NL      : ('\n' | ('\r' '\n') => '\r' '\n' | '\r') 
            { $setType(Token.SKIP); newline(); }
        ;

COMMENT : ( "/*" ( 
                    options {greedy=false;} : 
                    (NL)
                    | ~( '\n' | '\r' )
                 )* "*/"
            | "//" (~( '\n' | '\r' ))* (NL)
          )                      { $setType(Token.SKIP); }
        ;

LDV_LDVEQ : "/'" (
                    ('=') => '=' { $setType(LDVEQ); }
                    | { $setType(LDV); } 
                 );
    
LPAREN  : '(';
RPAREN  : ')';
MULT    : '*';
PLUS    : '+';
MINUS   : '-';
RDV     : '/';
MOD     : '%';
SEMI    : ';';            
LBRACE  : '{';
RBRACE  : '}';
LBRK    : '[';
RBRK    : ']';
ASGN    : '=';
COMMA   : ',';
PLUSEQ  : "+=";
MINUSEQ : "-=";
MULTEQ  : "*=";
RDVEQ   : "/=";
MODEQ   : "%=";
GE      : ">=";
LE      : "<=";
GT      : '>';
LT      : '<';
EQ      : "==";
NEQ     : "!=";
TRSP    : '\'';
COLON   : ':';
DCOLON  : "::";

ID  options { testLiterals = true; }
        : ALPHA (ALPHA|DIGIT)*
        ;

/* NUMBER example:
 * 1, 1e, 1., 1.e10, 1.1, 1.1e10
 */

NUMBER  : (DIGIT)+ ('.' (DIGIT)*)? (('E'|'e') ('+'|'-')? (DIGIT)+)? ;

/* Hanhua: the definition of the string might be different */
STRING  : '"'!
              (  ~('"' | '\n')
               | ('"'!'"')
              )*
          '"'!
        ;


class MxAntlrParser extends Parser;

options{
    k = 2;
    buildAST = true;
    exportVocab = MxAntlr;
}


tokens {
   // PROG;             // Hanhua: PROG is actually statements
   STATEMENT;   
   ARRAY_ROW;
   ARRAY;   
   VAR_LIST;
   EXPR_LIST;
   FUNC_CALL;
   // FUNC_BODY;        // Hanhua: this level is not necessary
   // FUNC_BODY_ASGN;   // Hanhua: this level is not necessary
   FOR_CON;
   LOOP;
   UPLUS;
   UMINUS;
}

{
    int nr_error = 0;
    public void reportError( String s ) {
        super.reportError( s );
        nr_error++;
    }
    public void reportError( RecognitionException e ) {
        super.reportError( e );
        nr_error++;
    }
}

program
        : ( statement | func_def )* EOF!
            {#program = #([STATEMENT,"PROG"], program); }
        ;
    
statement
        : for_stmt    
        | if_stmt
        | loop_stmt
        | break_stmt
        | continue_stmt
        | return_stmt
        | load_stmt
        | assignment
        | func_call_stmt
        | LBRACE! (statement)* RBRACE!
            {#statement = #([STATEMENT,"STATEMENT"], statement); }
        ;

for_stmt
        : "for"^ LPAREN! for_con RPAREN! statement
        ;

for_con
        : ID ASGN! range (COMMA! ID ASGN! range)*
            {#for_con = #([FOR_CON,"FOR_CON"], for_con); }
        ;
    
if_stmt
        : "if"^ LPAREN! expression RPAREN! statement
            (options {greedy = true;}: "else"! statement )?
        ;
    
loop_stmt!
        : "loop" ( LPAREN! id:ID RPAREN! )? stmt:statement
            {
                if ( null == #id )
                    #loop_stmt = #([LOOP,"loop"], #stmt);
                else
                    #loop_stmt = #([LOOP,"loop"], #stmt, #id);
            }
        ;
    
break_stmt
        : "break"^ (ID)? SEMI!
        ;

continue_stmt
        : "continue"^ (ID)? SEMI!
        ;
    
return_stmt
        : "return"^ (expression)? SEMI!
        ;
    
load_stmt
        : "include"^ STRING SEMI!
        ;
    
assignment
        : l_value ( ASGN^ | PLUSEQ^ | MINUSEQ^ | MULTEQ^ | LDVEQ^ 
                    | MODEQ^ | RDVEQ^ 
                  ) expression SEMI!
        ;
    
func_call_stmt
        : func_call SEMI!     
        ;
    
func_call
        : ID LPAREN! expr_list RPAREN!
            {#func_call = #([FUNC_CALL,"FUNC_CALL"], func_call); }
        ;
    
expr_list
        : expression ( COMMA! expression )* 
            {#expr_list = #([EXPR_LIST,"EXPR_LIST"], expr_list); }
        |   /*nothing: Hanhua: although empty, we should have a node */
            {#expr_list = #([EXPR_LIST,"EXPR_LIST"], expr_list); }
        ;   
    
func_def
        : "func"^ ID LPAREN! var_list RPAREN! func_body
        ;
    
var_list
        : ID ( COMMA! ID )*
            {#var_list = #([VAR_LIST,"VAR_LIST"], var_list); }
        |   /* nothing. Hanhua: necessary to add an node for empty list */
            {#var_list = #([VAR_LIST,"VAR_LIST"], var_list); }
        ;

// Hanhua: an intermediate level is removed
func_body
        : ASGN! a:expression SEMI!  
            {#func_body = #a; }
        | LBRACE! (statement)* RBRACE!
            {#func_body = #([STATEMENT,"FUNC_BODY"], func_body); }
        ;

expression
        : logic_term ( "or"^ logic_term )*
        ;
    
logic_term
        : logic_factor ( "and"^ logic_factor )*
        ;
    
logic_factor
        : ("not"^)? relat_expr
        ;
    
relat_expr
        : arith_expr ( (GE^ | LE^ | GT^ | LT^ | EQ^ | NEQ^) arith_expr )?
        ;
    
arith_expr
        : arith_term ( (PLUS^ | MINUS^) arith_term )*
        ;
    
arith_term
        : arith_factor ( (MULT^ | LDV^ | MOD^ | RDV^) arith_factor )*
        ;

arith_factor
        : PLUS! r_value 
            {#arith_factor = #([UPLUS,"UPLUS"], arith_factor); } 
        | MINUS! r_value 
            {#arith_factor = #([UMINUS,"UMINUS"], arith_factor); }
        | r_value (TRSP^)*;

    
r_value
        : l_value 
        | func_call 
        | NUMBER 
        | STRING
        | "true"
        | "false"
        | array 
        | LPAREN! expression RPAREN!
        ;

l_value
        : ID^ ( LBRK! index RBRK! )*
        ;
    
index
        : range (COMMA! range)?
            {#index = #([EXPR_LIST,"INDEX"], index); }
        ;

range
        : expression (COLON^ (expression)? | DCOLON^ expression )? | COLON^
        ;

array_row
        : expression (COMMA! expression)*
            {#array_row = #([ARRAY_ROW,"ARRAY_ROW"], array_row); }
        ;

array
        : LBRK! array_row (SEMI! array_row)* RBRK!
            {#array = #([ARRAY,"ARRAY"], array); }
        ;

cl_statement
        : ( statement | func_def )
        | "exit"
            { System.exit(0); }
        | EOF!
            { System.exit(0); }
        ;
