#include "sysdep.h"
#include <stdio.h>
#include <stdarg.h> // var args for writenf()
#include <assert.h>
#ifdef __unix
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#endif
#include "sysutil.h"

char sysutil_errstr[256] = "";

int new_server_sock(int type, u_short port)
{
  int sock, i, ret;
  struct sockaddr_in adr;
  sock = socket(AF_INET, type, 0);
  if(sock < 0) {
    sprintf(sysutil_errstr, "Error creating socket\n");
    return -1;
  }
  i = 1;
  ret = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &i, sizeof(i));
  if (ret < 0) // not fatal error
    sprintf(sysutil_errstr, "setsockopt error\n");

  init_sockaddr(&adr, htonl(INADDR_ANY), port);
  if (bind(sock, (struct sockaddr *) &adr, sizeof(adr)) < 0) {
    sprintf(sysutil_errstr, "Error binding socket\n");
    closesocket(sock);
    return -1;
  }

  if (type == SOCK_STREAM)
	  if (listen(sock, 4) < 0) { // use a larger backlog for safety
      sprintf(sysutil_errstr, "socket listen error\n");
      closesocket(sock);
      return -1;
    }

  return sock; // success
}

int new_client_sock(int type, u_long dest_ip, u_short dest_port)
{
  int sock, i, ret;
  struct sockaddr_in adr;

  sock = socket(AF_INET, type, 0);
  if(sock < 0) {
    sprintf(sysutil_errstr, "Error creating socket\n");
    return -1;
  }
  i = 1;
  ret = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &i, sizeof(i));
  if (ret < 0) // not fatal error
    sprintf(sysutil_errstr, "setsockopt error\n");

  if (type == SOCK_DGRAM) {
    init_sockaddr(&adr, htonl(INADDR_ANY), 0);
    if (bind(sock, (struct sockaddr *) &adr, sizeof(adr)) < 0) {
      sprintf(sysutil_errstr, "Error binding socket\n");
      closesocket(sock);
      return -1;
    }
  }
  init_sockaddr(&adr, dest_ip, dest_port);
  if (connect(sock, (struct sockaddr *) &adr, sizeof(adr)) < 0) {
    sprintf(sysutil_errstr, "Error connecting socket\n");
    closesocket(sock);
    return -1;
  }

  return sock; // success
}

void init_sockaddr(struct sockaddr_in *sa, u_long ip_addr, u_short port)
{
#ifdef __FreeBSD__
  sa->sin_len = sizeof(sa);
#endif
  sa->sin_family = AF_INET;
  sa->sin_addr.s_addr = ip_addr;
  sa->sin_port = htons(port);
  memset(sa->sin_zero, 0, sizeof(sa->sin_zero));
}

void cleanup_socket(int &sock)
{
  if (sock >= 0) {
    //printf("closesocket(%d)\n", sock);
    closesocket(sock);
    sock = -1;
  }
}

int readn(int fd, char *ptr, int nbytes)
{
  int togo, m;
  togo = nbytes;
  while (togo > 0) {
    m = readsocket(fd, ptr, togo);
    if (m <= 0)
      return m;
    togo -= m;
    ptr += m;
  }
  return nbytes-togo;
}

int writenf(int sd, const char *fmt, ...)
{
#define MAXLINE 256
  char	buf[MAXLINE];
  int n;

  va_list ap;
  va_start(ap, fmt);
  n = vsprintf(buf, fmt, ap);
  va_end(ap);
  if (n >= 0) // no error in formatting
    writen(sd, buf);
  return n;
}


/* writen function from Stevens, adapted into two versions */
int writen(int fd, char *ptr)
{
  int nbytes = strlen(ptr);
  return writen(fd, ptr, nbytes);
}

int writen(int fd, char *ptr, int nbytes)
{
  int nleft, nwritten;
  nleft = nbytes;
  while(nleft > 0) {
	nwritten = writesocket(fd, ptr, nleft);
    if (nwritten <= 0)
      return nwritten;

    nleft -= nwritten;
    ptr += nwritten;
  }

  return (nbytes - nleft);
}

/* Readline function from Stevens book */
int readline(int fd, char *ptr, int maxlen)
{
  int n, rc;
  char c;

  for (n = 1; n < maxlen; n++) {
	if ( (rc = readsocket(fd, &c, 1)) == 1) {
      *ptr++ = c;
      if (c == '\n')
	break;
    }
    else if (rc == 0) {
      if (n == 1)
	return(0);
      else
	break;
    }
    else return(-1);
  }

  *ptr = 0;
  return(n);
}


inline void timev_sanity_test(const struct timeval &a)
{ assert(a.tv_usec >=0 && a.tv_usec < USPS); }

// assume no negative fields
struct timeval operator +(const struct timeval &a, const struct timeval &b)
{
  struct timeval c;

  timev_sanity_test(a);
  timev_sanity_test(b);
  c.tv_sec = a.tv_sec + b.tv_sec;
  c.tv_usec = a.tv_usec + b.tv_usec;
  c.tv_sec += c.tv_usec/USPS; // process carry
  c.tv_usec = c.tv_usec%USPS;
  return c;
}

struct timeval operator +(const struct timeval &a, int b)
{
  struct timeval c;
  c.tv_sec = b;
  c.tv_usec = 0;
  return a+c;
}

struct timeval operator -(const struct timeval &a, const struct timeval &b)
{
  struct timeval c;

  timev_sanity_test(a);
  timev_sanity_test(b);
  c.tv_sec = a.tv_sec - b.tv_sec;
  c.tv_usec = a.tv_usec - b.tv_usec;
  if (c.tv_usec < 0) {
    c.tv_usec += USPS; // process borrow
    c.tv_sec -= 1;
  }
  timev_sanity_test(c);
  return c;
}

bool operator >(const struct timeval &a, const struct timeval &b)
{
  double fa, fb;
  timev_sanity_test(a);
  timev_sanity_test(b);
  fa = a.tv_sec + a.tv_usec*1e-6;
  fb = b.tv_sec + b.tv_usec*1e-6;
  return (fa > fb);
}
