#include "sysdep.h"
#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>

#include "sysutil.h"
#include "udpexpt.h"
#include "hmac_md5.h"

#define CONTROL_PORT    4367             /* Default control port */
#define FILE_PORT       4369
#define PKTSIZE		32		/* default packet size */
#define NPKT		1000		/* default number of packets */
#define MAXHNAME	100		/* maximum hostname size */

typedef struct tag_s {
  int s;             /* socket id */
  int inuse;         /* In use? */
  char name[255];
  char machine[255];
} tag_t;

typedef struct run_cfg_t {
  int nways;        /* 1 or 2 way (OTT or RTT) */
  int duration;     /* in seconds */
  int gap;          /* in ms */
  char fname[64];   /* trace file name */
  int pktsize;      /* UDP data size in bytes */
  unsigned short udp_port;
  char sender[MAXHNAME];
  char receiver[MAXHNAME];
  unsigned long sender_ip;
  unsigned long receiver_ip;
  int retrieve_it;
  char keyfile_sender[64]; // file containing shared secret key
  char keyfile_receiver[64];
} RunCfg;

HMAC_auth *sender_auth, *receiver_auth;

int cntrl_port = CONTROL_PORT;
tag_t tagarray[10];
int numtags = 0;
int currentsock, prompt;

extern char *optarg;

int OpenConnection(char *, int, char *);
void init_sockaddr(struct sockaddr_in *sa, u_long ip_addr, u_short port);
int ReadFile(int, char *);

unsigned long host2ip(char *hostname);
int set_cur_sock(char *tag);

int main(int argc, char *argv[])
{
  int i, s;
  fd_set readfds[255];
  struct timeval tp;
  char output[255];
  char ch;
  FILE *tst_counter; /* a test number counter */

  /* here are the defaults for a runtime test */
  RunCfg T = { 1, 60, 100, "nothing", 36, 4368, "", "", 0, 0, 0, "", "" };

  /* Parse command line args, if any */
  while ((ch = getopt(argc, argv, "n:t:g:f:p:o:Rs:r:a:b:")) != EOF)
    switch (ch) {
    case 'n':
      T.nways = atoi(optarg);
      if (T.nways < 1 || T.nways > 2) {
	fprintf(stderr, "nways can only be 1 or 2\n");
	exit(1);
      }
      break;
    case 't':
      T.duration = atoi(optarg);
      break;
    case 'g':
      T.gap = atoi(optarg);
      break;
    case 'f':
      strcpy(T.fname, optarg);
      break;
    case 'p':
      T.pktsize = atoi(optarg);
      break;
    case 'o':
      T.udp_port = atoi(optarg);
      break;
    case 'R':
      T.retrieve_it = 1;
      break;
    case 's':
      /*T.sender_ip = inet_addr(optarg);*/
      strcpy(T.sender, optarg);
      T.sender_ip = host2ip(T.sender);
      break;
    case 'r':
      /*T.receiver_ip = inet_addr(optarg);*/
      strcpy(T.receiver, optarg);
      T.receiver_ip = host2ip(T.receiver);
      break;
    case 'a':
      strcpy(T.keyfile_sender, optarg);
      break;
    case 'b':
      strcpy(T.keyfile_receiver, optarg);
      break;
    case '?':
      fprintf(stderr, "invalid option\n");
      exit(1);
    }

  if (!strcmp(T.sender, "") || !strcmp(T.receiver, "")) {
    fprintf(stderr, "usage: monitor_auto [-n 1|2] [-t time] [-g gap]"
      " [-f filename] [-p pktsize]\n\t[-o port] [-R] -a keyfile_sender -b keyfile_receiver -s sender -r receiver\n"
            " the value in \"()\" is the default\n"
            " -n: 1-way or 2-way (1),\t\t-t: duration in seconds (60)\n"
            " -g: packet interval in ms (100),\t-f: tracefile name (from test.counter)\n"
            " -o: udp test port (4368),\t-R: retrieve the remote file\n"
            " -a & -b: specifies shared secret key file of sender & receiver, for auth\n"
            " -p: UDP data size in bytes (36),\t-s,-r: hostname or IP addr\n");
    exit(1);
  }

  if (!strcmp(T.fname, "nothing")) {
    int tnum; /* test counter value */
    tst_counter = fopen("test.counter", "r");
    if (tst_counter == NULL) {
      fprintf(stderr, "can't open test.counter\n");
      exit(1);
    }
    fscanf(tst_counter, "%d", &tnum);
    sprintf(T.fname, "test%d", tnum);
    printf("chosen filename test%d\n", tnum);
    fclose(tst_counter);
    tst_counter = fopen("test.counter", "w");
    fprintf(tst_counter, "%d\n", tnum+1);
    fclose(tst_counter);
  }

  /* Initialize */

  //HMAC_auth_central::init();

  for(i = 0; i < 10; i++) {
    tagarray[i].inuse = 0;
  }
  
  prompt = -1;
  currentsock = -1;

    /* doing batch mode */
  /* New Step 1, using HMAC-MD5 authentication, each message is a full
   request, thus only 1 packet is needed, so we must send to receiver
   first */
  OpenConnection(T.receiver, CONTROL_PORT, T.receiver);
  set_cur_sock(T.receiver);
  // constructor will read from socket the challenge nonce value
  receiver_auth = new HMAC_auth(currentsock, CONtroller, T.keyfile_receiver);
  struct test_request req_common;
  // first fill in the common part
  req_common.req_cmd = htonl(StartTEST); // macro defined in hmac_md5.h
  // mode will be filled in later
  req_common.un.start.sender_ip = T.sender_ip;
  req_common.un.start.receiver_ip = T.receiver_ip;
  req_common.un.start.data_port = htons(T.udp_port);
  req_common.un.start.numways = htons(T.nways);
  req_common.un.start.runtime = htonl(T.duration);
  req_common.un.start.pktgap = htonl(T.gap*1000); /* in us */
  req_common.un.start.pktsize = htonl(T.pktsize);

  req_common.un.start.mode = htonl(RECEIVER_mode); // defined in hmac_md5.h
  if (T.nways == 1)
    strcpy(req_common.un.start.filename, T.fname);
  // send out the request, along with authentication info
  receiver_auth->send_req((unsigned char *) &req_common, sizeof(req_common));
  // sleep for 2 seconds, to allow receiver to start

  /* Step 2, open sender connection, use hostname as tag */
  OpenConnection(T.sender, CONTROL_PORT, T.sender);
  set_cur_sock(T.sender);
  // constructor will read from socket the challenge nonce value
  sender_auth = new HMAC_auth(currentsock, CONtroller, T.keyfile_sender);
  req_common.un.start.mode = htonl(SENDER_mode); // defined in hmac_md5.h
  if (T.nways == 2) {
    strcpy(req_common.un.start.filename, T.fname);
  }

  sleep(1);
  sender_auth->send_req((unsigned char *) &req_common, sizeof(req_common));

  /* Now, wait for test to finish, then retrieve it */
  while(1) {

    int nf; // number of read fds

    FD_ZERO(readfds);
    for(i = 0, nf = 0; i < numtags; i++)
      if(tagarray[i].inuse == 1) {
        FD_SET(tagarray[i].s, readfds);
        nf++;
      }
    /* FD_SET(0, readfds); we don't need stdin in batch mode */
    if (nf == 0) {
      printf("\tall connections gone. exiting.\n");
      exit(1); // not normal condition
    }

    tp.tv_sec = 10;
    tp.tv_usec = 0;
    select(255, readfds, (fd_set *) 0, (fd_set *) 0, &tp);

    for(i = 0; i < numtags; i++)
      if ( (tagarray[i].inuse == 1) && FD_ISSET(tagarray[i].s, readfds) )
        if (readline(tagarray[i].s, output, 255) > 0) {
          printf("##%s says %s",tagarray[i].name, output);
          if (!strcmp(output, "Logging process terminated\n")
              || !strcmp(output, "Closed data receive connection\n")) {
            char *logger;
            fprintf(stderr, "runtime test finished.\n");
            if (T.retrieve_it) {
              HMAC_auth *retrieved;
              if (T.nways == 2) {
                logger = T.sender;
                retrieved = sender_auth;
              }else if (T.nways == 1) {
                logger = T.receiver;
                retrieved = receiver_auth;
              }else {
                printf("#ways %d != 1 or 2. Exiting\n", T.nways);
                exit(1);
              }
              req_common.req_cmd = htonl(GetFILE); // defined in hmac_md5.h
              req_common.un.file.file_port = htons(FILE_PORT);
              strcpy(req_common.un.file.filename, T.fname);
              retrieved->send_req((unsigned char *) &req_common, sizeof(req_common));
              sleep(2); /* wait for the other side to start up retrieval */
              s = OpenConnection(logger, FILE_PORT, "tmp_logger");
              ReadFile(s, T.fname);
              close(s);
            }
            /* need to close sockets */
            exit(0);
          }
        }
        else {
          printf("Connection to %s closed\n", tagarray[i].name);
          close(tagarray[i].s);
          tagarray[i].s = -1;
          tagarray[i].inuse = 0;
          if(prompt == i) {
            prompt = -1;
            currentsock = -1;
          }
        }

  } /* end of while */
  
  return 0;
}

int ReadFile(int s, char *fname) {

  FILE *fp;
  char buf[1024];
  int len, wr;

  if((fp = fopen(fname, "w")) == 0) {
    printf("Error opening file %s\n",fname);
    return (-1);
  }

  wr = 0;
  while((len = read(s, buf, 1024)) > 0) {
    
    if(fwrite(buf, sizeof(char), len, fp) < len) {
      fclose(fp);
      printf("Error writing to file %s\n",fname);
      return(-1);
    }
    wr += len;
  }

  if(len == 0) {
    fclose(fp);
    printf("Successfully wrote file, %d bytes\n",wr);
    return(wr);
  }
  else if (len < 0) {
    fclose(fp);
    printf("Error reading socket\n");
    return(-1);
  }

  return wr;
}

unsigned long host2ip(char *hostname)
{
  unsigned long ipaddr;
  struct hostent *hostinfo;
  struct in_addr *adrptr;

  if ((ipaddr = inet_addr(hostname)) == (unsigned long) -1) {
    if ((hostinfo = gethostbyname(hostname)) == NULL) {
      fprintf(stderr, "Error resolving name %s\n",hostname);
      exit(1);
    }

    adrptr = (struct in_addr *) hostinfo->h_addr_list[0];
    ipaddr = (int) adrptr->s_addr;
  }

  return ipaddr;
}


int OpenConnection(char *machine, int port, char *tag) {

  int i, s, foundit;
  int ipaddr;
  struct hostent *hostinfo;
  struct in_addr *adrptr, adrholder;

  /* Resolve the name */

  if((ipaddr = inet_addr(machine)) < 0) {
    printf("Resolving hostname %s -> ", machine);
    fflush(stdout);
    if((hostinfo = gethostbyname(machine)) == NULL) {
      printf("Error resolving name %s\n",machine);
      exit(1);
    }
    adrptr = (struct in_addr *) hostinfo->h_addr_list[0];
    ipaddr = (int) adrptr->s_addr;
  }

  adrholder.s_addr = ipaddr;
  printf("Connection to %s ", inet_ntoa(adrholder));
  fflush(stdout);

  /* Now open connection */
  s = new_client_sock(SOCK_STREAM, ipaddr, port);
  if (s < 0) {
    printf("Control: %s\n", sysutil_errstr);
    exit(1);
  }

  printf("established\n");

  if( tag != NULL) {
    foundit = -1;
    for(i = 0; i < numtags; i ++)
      if(tagarray[i].inuse == 0) {
	foundit = i;
	break;
      }
  
    if(foundit != -1) {
      tagarray[foundit].s = s;
      tagarray[foundit].inuse = 1;
      strcpy(tagarray[foundit].name, tag);
    } else {
      tagarray[numtags].s = s;
      tagarray[numtags].inuse = 1;
      strcpy(tagarray[numtags].name, tag);
      numtags++;
    }
  }

  return s;
}

/* returns 1 on success, 0 on error */
int set_cur_sock(char *tag)
{
  for(int i = 0; i < numtags; i++)
    if((tagarray[i].inuse == 1) && (strcmp(tagarray[i].name, tag) == 0)) {
      currentsock = tagarray[i].s;
      prompt = i;
      strcpy(tagarray[i].machine, tag);
      return 1;
    }
  return 0;
}

