#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <assert.h>
#include "TestSetup.h" // sysdep.h includes sys/types.h for sys/socket.h
#ifdef __unix
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#endif

#include "sysutil.h"

#define CLEANUP_thread(x) { \
  cleanup_thread(x); \
  return; \
}

extern char global_keyfile[]; // defined in module_mt.cpp

TestSetup::TestSetup(int new_sock)
  // nonce value will be generated and stored in auth_hdr.
  // keyfile is NULL, because it's already read at the beginning.
  : csock(new_sock), auth_hdr(new_sock, ENDpoint, NULL) // prog_type = endpoint
{
  data_sock = -1;
#ifdef WIN32
  dummy_sock = -1;
#endif

  data_port = DATA_PORT;
  file_port = FILE_PORT;
  keepalives = true;
  alive_gap = ALIVE_GAP;
  is_sender = false;
  pktsize = PKTSIZE;
  pktgap = to_timeval_usec(PKTGAP);
  runtime = RUNTIME;
  numways = 2;
  strcpy(log_filename, "udptrace_test");

  test_init = 0;
  is_sending = false;
  n_sentpackets = n_receivepackets = 1;
  next_alivetime = to_timeval(0);
  //gbl_transport = TRANSPORT_UDP;
  log_file = NULL;

  finished = retrieved = false;
}

int TestSetup::max_sd()
{
  int max = -1;
  if (max < csock) max = csock;
  if (max < data_sock) max = data_sock;
#ifdef WIN32
  if (max < dummy_sock) max = dummy_sock;
#endif
  if (max < 0) max = 0;
  return max;
}


void TestSetup::service()
{
  int rc;
  
  struct timeval t_cur = {0, 0}, delay;
  fd_set fdset;
  char data[1024];
  //, msg[255];

#ifdef WIN32
  dummy_sock = socket(AF_INET, SOCK_STREAM, 0);
  if (dummy_sock < 0) {
    printf("can't create dummy_sock\n");
    CLEANUP_thread(-1); // macro
  }
#endif
  donetime = to_timeval(0);

  // send a nonce value for the purpose of challenge-response authentication
  struct test_request challenge;
  challenge.req_cmd = htonl(CHALLENGE); // msg type is challenge
  challenge.un.nonce_val = htonl(get_nonce()); // stored in TestSetup object
  writen(csock, (char *)&challenge, sizeof(challenge)); // actual msg

  while (!retrieved) {
    // filling the fdset for select()
    FD_ZERO(&fdset);
#ifdef WIN32
    FD_SET(dummy_sock, &fdset);
#endif
    if(data_sock != -1)
      FD_SET(data_sock, &fdset);
    if(csock != -1)
      FD_SET(csock, &fdset);
    else // if control socket is invalid or closed, then terminate the test
      CLEANUP_thread(0);

    /* Wait for either packet arrival, or expiration of timer to send packet */
    gettimeofday(&t_cur, NULL);
    if ( is_sender && is_sending ) {
      delay = next_sendtime - t_cur;
      if (to_double(delay) < 0) // if we missed deadline
        delay = to_timeval(0);  // then sleep time should be 0
    }else {
      if (finished)
        delay.tv_sec = 5;
      else if ( !is_sender && (to_double(donetime-t_cur) < 10.0) )
        delay.tv_sec = 5;
      else
        delay.tv_sec = 60;
      delay.tv_usec = 0;
    }

    rc = select(max_sd()+1, &fdset, NULL, NULL, &delay);
    if (rc < 0)
      printf("select error: %d %s\n", get_errno(), get_errstring());

    /* process packet transmission if needed */
    gettimeofday(&t_cur, NULL);
    if ( is_sender && is_sending && (t_cur > next_sendtime) ) {
      next_sendtime += pktgap; /* compute next send time */
      EmitPacket(0, data); /* send packet */
    }

    /* process sockets if needed */
    if (is_ready(csock, &fdset))
      {
        if (ProcessControl() < 0)
          CLEANUP_thread(-1);
      }
    else if(is_ready(data_sock, &fdset)) // incoming packet on data socket
      {
        if (is_sender) { // if I am a sender, it must be an echoed packet
          rc = ProcessReflection(data);
          if (rc <= 0) {
            if (rc == 0)
              writen(csock,"Closed data receive connection\n");
            else
              writen(csock,"reflection error, closing data send connection\n");
            CLEANUP_thread(-1);
          }
        } else { // if I am a receiver, it must be a request packet
          rc = ProcessArrival(data);
          if(rc <= 0) {
            if (rc == 0)
              writen(csock,"Closed data receive connection\n");
            else
              writenf(csock, "recv error at %d, closing data receive connection\n", n_receivepackets);
            CLEANUP_thread(-1);
          }
        }
      }

    /* do keepalives if needed */
    if ( keepalives && (t_cur > next_alivetime)) {
      if (is_sender)
        writenf(csock, "Alive:sent %d packets\n", n_sentpackets);
      else
        writenf(csock, "Alive: got %d packets\n", n_receivepackets);
      next_alivetime = t_cur + alive_gap;
    }

    /* check whether test has finished */
    if ((donetime.tv_sec > 0) && (t_cur > donetime) ) {
      //donetime += 10; // wait for retriever
      is_sending = false;  // do not send any more packets
      if (log_file != NULL) {
        fclose(log_file);
        log_file = NULL;
      }
      // redundant also in cleanup_thread(), but need it here. but why?
      if (is_logger())
        writen(csock,"Logging process terminated\n");
      finished = true;
    }

  } // end of big while loop

  CLEANUP_thread(0); // after while loop, clean up everything

}

void TestSetup::cleanup_thread(int rc)
{
  //char data[1024];
  struct timeval t_finish;

  //if (is_sending == YES)
  //  EmitPacket(1, data);
  cleanup_socket(data_sock);
#ifdef WIN32
  cleanup_socket(dummy_sock); // so that memory is not wasted
#endif

  if (csock != -1) {
    writen(csock,"Closed data connection\n");
    // it seems that Solaris/Linux all have this "bug"?! If I write to a 
    // TCP socket after it is closed (by remote side who is also local),
    // the accept() on seed socket will terminate with SIGPIPE.
    cleanup_socket(csock); //  close socket after writing the last message
  }
  is_sending = false;
  if (log_file != NULL) {
    fclose(log_file);
    log_file = NULL;
  }
  gettimeofday(&t_finish, NULL);
  printf("test finished at %.19s.%06ld, freeing thread id %d\n",
         ctime(&t_finish.tv_sec), t_finish.tv_usec, pthread_self());
  //delete this; // is this legal?
  //pthread_exit(NULL);
}

/* stop: restart the test */
void TestSetup::EmitPacket(bool stop, char *data)
{
  if (data_sock < 0) {
    printf("data_sock is not initialized!\n");
    return;
  }

  struct udprecord *rec;
  struct timeval tp;

  rec = (struct udprecord *) data;
  memset((char *) rec, 0, sizeof(*rec));
  rec->num = htonl(n_sentpackets++);
  rec->length = htonl(pktsize);
  gettimeofday(&tp, NULL);
  rec->snd.tv_sec = htonl(tp.tv_sec);
  rec->snd.tv_usec = htonl(tp.tv_usec);

  // send a special packet to indicate stop of test, not used yet.
  if (stop == 1) rec->num = 0;

  if (SendPacket(data, pktsize, NULL) < 0) {
    char buf[128];
    sprintf(buf, "Error in UDP send: %d %s\n", get_errno(), get_errstring());
    writen(csock, buf);
  }
}

// returns # of bytes sent, or -1 if error
// if to is NULL, assumed a connected UDP socket
int TestSetup::SendPacket(char *data, int maxlen, struct sockaddr_in *to)
{
  int ret;

  socklen_t length = sizeof(struct sockaddr_in);
  if (to == NULL) // always use UDP
    ret = send(data_sock, (char *) data, maxlen, 0);
  else
    ret = sendto(data_sock, (char *) data, maxlen, 0, (SA *) to, length);
  if (ret < 0) {
    char buf[128];
    sprintf(buf, "Error in UDP send: %d %s\n", get_errno(), get_errstring());
    writen(csock, buf);
    return -1;
  }
  return ret;
}

/* if "from" is null, means we should use "recv" instead of "recvfrom" */
// returns # of received bytes, or -1 if error.
int TestSetup::GetPacket(char *data, int maxlen, struct sockaddr_in *from)
{
  int ret;

  //assert(gbl_transport == TRANSPORT_UDP);
  socklen_t length = sizeof(struct sockaddr_in);
  if (from == NULL)
    ret = recv(data_sock, data, maxlen, 0);
  else
    ret = recvfrom(data_sock, data, maxlen, 0, (SA *) from, &length);
  if(ret < 0) {
    char buf[128];
    sprintf(buf, "Error in UDP receive %d: %d %s, from:%p sent %d\n", ret,
            get_errno(), get_errstring(), from, n_sentpackets);
    writen(csock, buf);
    return -1;
  }
  return ret;
}

// returns 0 on success, -1 on failure.
int TestSetup::ProcessControl()
{
  char message[255];
  struct test_request req;
  int req_len, rc, auth_rc;

  auth_rc = auth_hdr.receive_req((unsigned char *)&req, &req_len);
  if (auth_rc != true)
    if (auth_rc == false) {
      sprintf(message, "authentication failed!\n");
      puts(message);
      writen(csock, message);
      return -1;
    }
    else if (auth_rc == 2) { // connection closed
      cleanup_socket(csock);
      return -1;
    }

  req.req_cmd = ntohl(req.req_cmd);
  switch (req.req_cmd)
    {
    case StartTEST: // macro defined in hmac_md5.h
      rc = start_test(req.un.start);
      break;
    case GetFILE:
      writen(csock, "Start retrieval\n");
      file_port = ntohs(req.un.file.file_port);
      rc = InitRetriever(req.un.file.filename);
      if (rc >= 0)
        printf("retrieval completed\n");
      else
        printf("retrieve failed!\n");
      retrieved = true; // whether success or fail, we need to get out of loop
      break;
    default:
      printf("illegal req_cmd %d", req.req_cmd);
      rc = -1;
      break;
    }

  return rc;
}

// returns # of bytes received, or 0 or -1 if error.
int TestSetup::ProcessArrival(char data[])
{
  struct sockaddr_in recvaddr;
  struct udprecord *rec;
  struct timeval tp;
  int length;

  length = GetPacket(data, 1024, &recvaddr);
  if (length > 0)
    n_receivepackets++;
  else
    return length;

  /* Fill in fields */
  rec = (struct udprecord *) data;
  gettimeofday(&tp, NULL);
  if(rec->num == 0) {
    donetime = tp + 3;
    return (-1);
  }
  rec->midhost.tv_sec = htonl(tp.tv_sec);
  rec->midhost.tv_usec = htonl(tp.tv_usec);

  if (numways == 2) {
    if(SendPacket(data, ntohl(rec->length), &recvaddr) < 0)
      writen(csock,"Error reflecting packet\n");
  }else if (numways == 1)
    WriteLog(rec);

  return length;
}

// returns # of bytes read, -1 on error.
int TestSetup::ProcessReflection(char data[])
{
  struct udprecord *rec;
  struct timeval tp;
  int length;

  length = GetPacket(data, 1024, NULL);
  if(length <= 0)
    return -1;

  /* Fill in fields */
  rec = (struct udprecord *) data;
  gettimeofday(&tp, NULL);
  rec->rcv.tv_sec = htonl(tp.tv_sec);
  rec->rcv.tv_usec = htonl(tp.tv_usec);

  WriteLog(rec);
  return length;
}

// returns 0 on success, -1 on failure.
int TestSetup::start_test(struct start_request &params)
{
  // char message[255];

  params.mode = ntohl(params.mode);
  if (params.mode == SENDER_mode)
    is_sender = true;
  else if (params.mode == RECEIVER_mode)
    is_sender = false;
  else {
    printf("unknown mode %d\n", params.mode);
    return -1;
  }
  sender_ip = params.sender_ip; // in big-endian, no need to change
  receiver_ip = params.receiver_ip;
  data_port = ntohs(params.data_port);
  numways = ntohs(params.numways);
  runtime = ntohl(params.runtime);
  pktgap = to_timeval_usec(ntohl(params.pktgap));
  pktsize = ntohl(params.pktsize);
  alive_gap = ntohl(params.alive_gap);
  if (alive_gap == 0)
    keepalives = false;
  else
    keepalives = true;
  strcpy(log_filename, params.filename);
  int rc;
  if (is_sender) {
    rc = InitializeSender();
    //printf("\tI am a sender\n");
    //sprintf(message, "Send process begun\n");
  } else {
    rc = InitializeReceiver();
    //printf("\tI am a receiver\n");
    //sprintf(message, "Receive process begun\n");
  }
  if (rc != 0)
    return rc;

  struct in_addr adrholder;
  adrholder.s_addr = sender_ip;
  printf("runtime\tpktgap\t#ways\tsender\t\treceiver\tlog_file\n");
  printf("%5ds\t%dms\t%d\t%s", runtime, to_usec(pktgap)/1000,
         numways, inet_ntoa(adrholder));
  adrholder.s_addr = receiver_ip;
  printf("\t%s\trf_%s\n", inet_ntoa(adrholder), log_filename);

  return 0;
}

// returns 0 on success, -1 on failure
int TestSetup::OpenLog()
{
  char fname[255];
  time_t tm;
  struct in_addr adr;

  sprintf(fname, "rf_%s", log_filename);
  if((log_file = fopen(fname, "w"))  == NULL) {
    writen(csock,"Logger unable to open logfile\n");
    return -1;
  }

  tm = time((time_t *) 0);
  fprintf(log_file, "#Internet weather station data file\n");
  fprintf(log_file, "#Generated on %s\n", ctime(&tm));
  adr.s_addr = receiver_ip;
  fprintf(log_file, "#recv host is %s\n", inet_ntoa(adr));
  adr.s_addr = sender_ip;
  fprintf(log_file, "#send host is %s\n", inet_ntoa(adr));
  fprintf(log_file, "#numways is %d\n", numways);
  fprintf(log_file, "#pktsize %d  pktgap %d  runtime %d\n",
          pktsize, to_usec(pktgap), runtime);
  fprintf(log_file, "#transport is UDP\n");
  //        (gbl_transport == TRANSPORT_TCP) ? ("TCP"):("UDP") );
  return 0;
}

void TestSetup::WriteLog(struct udprecord *rec)
{
  double snd1, mid1, rcv1;
  /* must convert back to host order before printing */
  rec->num = ntohl(rec->num);
  rec->snd.tv_sec = ntohl(rec->snd.tv_sec);
  rec->snd.tv_usec = ntohl(rec->snd.tv_usec);
  rec->midhost.tv_sec = ntohl(rec->midhost.tv_sec);
  rec->midhost.tv_usec = ntohl(rec->midhost.tv_usec);
  rec->rcv.tv_sec = ntohl(rec->rcv.tv_sec);
  rec->rcv.tv_usec = ntohl(rec->rcv.tv_usec);

  if (!test_init) {
    initsecs = (rec->snd.tv_sec < rec->midhost.tv_sec)
               ? rec->snd.tv_sec : rec->midhost.tv_sec;
    test_init = true;
  }

  snd1 = (rec->snd.tv_sec - initsecs) + (rec->snd.tv_usec / 1000000.0);
  mid1 = (rec->midhost.tv_sec - initsecs) + (rec->midhost.tv_usec / 1000000.0);

  /* only if it was 2-way connection */
  if (rec->rcv.tv_sec)
    rcv1 = (rec->rcv.tv_sec - initsecs) + (rec->rcv.tv_usec / 1000000.0);
  else rcv1 = 0.0;

  if (log_file != NULL)
	fprintf(log_file, "%d %.6f %.6f %.6f\n",
		    (int) rec->num, snd1, mid1, rcv1);
}

// returns 0 on success, -1 on error
int TestSetup::InitializeSender()
{
  //char buf[256];
  struct timeval t_cur;

  //assert(gbl_transport == TRANSPORT_UDP);
  //trans = (gbl_transport == TRANSPORT_TCP) ? SOCK_STREAM : SOCK_DGRAM;

  /* Open socket */
  data_sock = new_client_sock(SOCK_DGRAM, receiver_ip, data_port);
  if (data_sock < 0) {
    writen(csock, sysutil_errstr);
    return -1;
  }else
    writen(csock, "Sender UDP connection established\n");

  gettimeofday(&t_cur, NULL);
  donetime = t_cur + runtime; /* Schedule termination */

  next_sendtime = t_cur + pktgap; /* Schedule first packet */
  is_sending = true;

  if(keepalives) /* Shedule keep alives */
    next_alivetime = t_cur + alive_gap;

  if(numways == 2) /* Call function to create log file */
    return OpenLog();
  else
    return 0;
}

// returns 0 on success, -1 on error
int TestSetup::InitializeReceiver()
{
  char buf[256];
  struct timeval t_cur;

  //assert(gbl_transport == TRANSPORT_UDP);
  //trans = (gbl_transport == TRANSPORT_TCP) ? SOCK_STREAM : SOCK_DGRAM;

  /* Open socket */
  data_sock = new_server_sock(SOCK_DGRAM, data_port);
  if (data_sock < 0) {
    writen(csock, sysutil_errstr);
    return -1;
  }
  sprintf(buf, "Receiver UDP connected\n");
  writen(csock, buf);

  gettimeofday(&t_cur, NULL);
  // give an extra 5 seconds for sender, to avoid race conditions
  donetime = t_cur + (runtime + 5);

  if (keepalives) /* Shedule keep alives */
    next_alivetime = t_cur + alive_gap;

  if(numways == 1)
    return OpenLog();
  else
    return 0;
}

int TestSetup::InitRetriever(char *ext)
{
  int skt, s, len, wr;
  socklen_t length;
  struct sockaddr_in adr;
  char data[1024], filename[255];
  FILE *fp;

  skt = new_server_sock(SOCK_STREAM, file_port);
  if(skt < 0) {
    sprintf(data, "retriever: %s", sysutil_errstr);
    writen(csock, data);
    return -1;
  }

  length = sizeof(adr);
  s = accept(skt, (struct sockaddr *) &adr, &length);
  closesocket(skt); // close the seed socket
  if (s < 0) {
    writen(csock, "retriever accept failed\n");
	printf("retriever accept failed\n");
	return -1;
  }

  if (log_file != NULL) {
	  printf("log file should have been closed!?\n");
	  fclose(log_file);
	  log_file = NULL;
  }
  /* open file for reading */
  strcpy(filename, "rf_");
  strcat(filename, ext);
  if((fp = fopen(filename,"r")) == 0) {
    sprintf(data, "can't retrieve file %s\n", filename);
    writen(csock, data);
    closesocket(s);
    return -1;
  }

  wr = 0;
  while ((len = fread(data, sizeof(char), 1024, fp)) > 0) {

    if(writesocket(s, data, len) < len) {
	  printf("retrieve failed during transport\n");
      writen(csock, "retrieve failed during transport\n");
      fclose(fp);
      closesocket(s);
      return -1;
    }
    printf("written %d bytes\r", len);
    wr += len;
  }

  printf("finished reading retrieval file\n");
  fclose(fp);
  thread_sleep(2000); // wait 2 seconds for retriever to finish download
  closesocket(s);

  if (len < 0) {
	printf("read error while retrieving file\n");
    writen(csock, "read error while retrieving file\n");
    return -1;
  }

  return 0;
}
